<?php

/**
 *
 */
class Modula_Advanced_Shortcodes {

	/**
	 * Holds the class object.
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public static $instance;

	/**
	 * The name of the plugin.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $plugin_name = 'Modula Advanced Shortcodes';

	/**
	 * Unique plugin slug identifier.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $plugin_slug = 'modula-advanced-shortcodes';

	/**
	 * Primary class constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		// Load the plugin textdomain.
		add_action( 'init', array( $this, 'load_plugin_textdomain' ) );

		// Set admin requirements on admin_init so that we won't trigger get_packages from WPChill_Upsells on frontend
		add_action( 'admin_init', array( $this, 'set_admin_requirements' ), 25 );

		add_shortcode( 'modula_all_galleries', array( $this, 'modula_all_galleries' ) );
		add_shortcode( 'modula_filters', array( $this, 'modula_output_filters' ) );
		add_action('modula_extra_scripts',array($this,'global_filters'));

		add_filter('modula_uninstall_db_options',array($this,'advanced_shortcodes_uninstall'),20,1);

		add_filter( 'modula_debug_information', array( $this, 'debug_info' ), 50 );
		// Load the plugin.
		$this->init();
		register_activation_hook(MODULA_ADVANCED_SHORTCODES_FILE,array($this,'plugin_activation'));

		$this->init();
	
	}

    /**
     * Prevent plugin activation if Modula Pro is not installed and activated
     *
     * @since 1.0.1
     */
    public function plugin_activation(){
        if(!class_exists('Modula_PRO')){
            deactivate_plugins( plugin_basename( MODULA_ADVANCED_SHORTCODES_FILE ) );
            wp_die(__('Please install and activate Modula Pro before using Modula Advanced Shortcodes add-on.','modula-albums'));

        }

    }

	/**
	 * Loads the plugin textdomain for translation.
	 *
	 * @since 1.0.0
	 */
	public function load_plugin_textdomain() {
		load_plugin_textdomain( $this->plugin_slug, false, MODULA_ADVANCED_SHORTCODES_TRANSLATE );
	}

	/**
	 * Loads the plugin into WordPress.
	 *
	 * @since 1.0.0
	 */
	public function init() {

		// Load admin only components.
		if ( is_admin() ) {
			add_action( 'modula_pro_updater', array( $this, 'addon_updater' ), 15, 2 );
		}

	}

	/**
	 * Set admin requirements and check for license.
	 *
	 * @return void
	 * @since 1.0.4
	 */
	public function set_admin_requirements() {

		if ( class_exists( 'WPChill_Upsells' ) ) {
			$args           = apply_filters(
				'modula_upsells_args',
				array(
					'shop_url' => 'https://wp-modula.com',
					'slug'     => 'modula',
				)
			);
			$wpchill_upsell = WPChill_Upsells::get_instance( $args );

			if ( $wpchill_upsell && ! $wpchill_upsell->is_upgradable_addon( 'modula-advanced-shortcodes' ) ) {

				$this->require_admin();

			}
		}
	}

	public function addon_updater( $license_key, $store_url ) {

		if ( class_exists( 'Modula_Pro_Base_Updater' ) ) {
			$modula_addon_updater = new Modula_Pro_Base_Updater( $store_url, MODULA_ADVANCED_SHORTCODES_FILE,
				array(
					'version' => MODULA_ADVANCED_SHORTCODES_VERSION,        // current version number
					'license' => $license_key,               // license key (used get_option above to retrieve from DB)
					'item_id' => 272901,                      // ID of the product
					'author'  => 'MachoThemes',            // author of this plugin
					'beta'    => false,
				)
			);
		}
	}

	/**
	 * Loads all admin related files into scope.
	 *
	 * @since 1.0.0
	 */
	public function require_admin() {

		// Load the plugin.
		require_once MODULA_ADVANCED_SHORTCODES_PATH . 'includes/admin/class-modula-advanced-shortcodes-settings.php';
	}


	/**
	 * Returns the singleton instance of the class.
	 *
	 * @since 1.0.0
	 *
	 * @return object The Advanced_Shortcodes object.
	 */
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Modula_Advanced_Shortcodes ) ) {
			self::$instance = new Modula_Advanced_Shortcodes();
		}

		return self::$instance;

	}

    /**
     * Enqueue Modula global filters script
     */
	public function global_filters(){
		wp_register_script('modula-global-filters',MODULA_ADVANCED_SHORTCODES_URL.'assets/js/global-filters.min.js',array('modula'),MODULA_ADVANCED_SHORTCODES_VERSION,true);
		// comment the enqueue for the moment
	    // wp_enqueue_script('modula-global-filters');
    }

	/**
	 * Shortcode for displaying all galleries or galleries based on gallery_id request
	 */
	public function modula_all_galleries() {

		$output = '';
		$gallery_link = get_option( 'mas_gallery_link' );

		if(!$gallery_link){
			$gallery_link = 'gallery_id';
		}

		if ( isset( $_REQUEST[$gallery_link] ) ) {
			$gallery_id   = (int) $_REQUEST[ $gallery_link ];

			$gallery_args = array(
				'post_type' => 'modula-gallery',
				'post__in'  => array( $gallery_id ),
				'order'     => 'DESC'
			);

		} else {
			$gallery_args = array(
				'post_type'      => 'modula-gallery',
				'posts_per_page' => - 1,
				'orderby'        => 'menu_order',
				'order'          => 'DESC'
			);

		}

		$galleries    = new WP_Query( $gallery_args );

		if ( $galleries->have_posts() ) {

			while ( $galleries->have_posts() ){

				$galleries->the_post();
				$id = get_the_ID();

				if ( ! is_admin() ) {

					if ( post_password_required( $id ) ) {
						$output .= get_the_password_form( $id );
					} else {
						$output .= do_shortcode( '[modula id="' . esc_attr( $id ) . '"]' );
					}

				}
			}
			wp_reset_postdata();
		} else {

			if ( isset($gallery_id) ) {
				$output .= ( $galleries->found_posts == 0 ) ? __( 'No galleries found matching id', 'modula-advanced-shortcodes' ) . ' <strong>' . esc_html( $gallery_id ) . '</strong>' : "";
			} else {
				$output .= __( 'No galleries found.', 'modula-advanced-shortcodes' );
			}

		}

		return $output;
	}

	/**
	 * Shortcode to output only the filters
	 */

    public function modula_output_filters($atts) {

        $gallery_link = get_option('mas_gallery_link');

        $shortcode_atts = shortcode_atts(array(
            'gallery_id' => false,
        ), $atts);

        $filters_array = array();

        $gallery_args = array(
            'post_type'      => 'modula-gallery',
            'posts_per_page' => -1,
            'orderby'        => 'menu_order',
            'order'          => 'DESC',
        );

        $gallery_id = (isset($_REQUEST[$gallery_link])) ? $_REQUEST[$gallery_link] : $shortcode_atts['gallery_id'];

        // Check to see if $gallery_id is not false
        if ($gallery_id) {
            $gallery_args['post__in'] = array($gallery_id);
        }

        $galleries = new WP_Query($gallery_args);

        if ($galleries->have_posts()) {
            while ($galleries->have_posts()) {
                $galleries->the_post();
                $modula_settings = get_post_meta(get_the_ID(), 'modula-settings', true);
                $filters         = $modula_settings['filters'];
                if ($filters) {
                    foreach ($filters as $filter) {
                        if (!in_array($filter, $filters_array)) {
                            $filters_array[] = $filter;
                        }
                    }

                }
            }
            wp_reset_postdata();
        }

        $html = '<div class="modula modula-gallery modula-gallery-filters-shortcode"><div class="filters styled-menu menu--default "><ul class="menu__list">';
        $html .= '<li class="menu__item"><a data-filter="all" href="#" class="selected">All</a></li>';

        foreach ($filters_array as $filter) {
            $html .= '<li class="menu__item"><a class="menu__link" data-filter="' . esc_attr(urldecode(sanitize_title( $filter ))) . '" href="#jtg-filter-' . esc_attr(urldecode(sanitize_title( $filter ))) . '">' . esc_html($filter) . '</a></li>';
        }

        $html .= '</ul></div></div>';

        return $html;

    }


	/**
	 * Uninstall process
	 *
	 * @param $db_options
	 *
	 * @return mixed
	 *
	 * @since 1.0.2
	 */
    public function advanced_shortcodes_uninstall($db_options){

    	$db_options[] = 'mas_gallery_link';

    	return $db_options;
    }

	/**
	 * Add debug info
	 *
	 * @param $info
	 *
	 * @return mixed
	 * @since 1.0.3
	 */
	public function debug_info( $info ){
		$link_att = get_option( 'mas_gallery_link' );

		$info['mas_version'] = array(
			'label' => __( 'Modula Advanced Shortcodes versions', 'modula-advanced-shortcodes' ),
			'value' => esc_html( MODULA_ADVANCED_SHORTCODES_VERSION ),
			'debug' => esc_html( 'MAS version' . MODULA_ADVANCED_SHORTCODES_VERSION ),
		);

		$info['mas_version'] = array(
			'label' => __( 'Link attribute', 'modula-advanced-shortcodes' ),
			'value' => ( $link_att && '' != $link_att ) ? esc_html( $link_att ) : __( 'No attribute specified', 'modula-advanced-shortcodes' ),
			'debug' => ( $link_att && '' != $link_att ) ? esc_html( $link_att ) : 'No attribute specified',
		);

		return $info;
	}

}