<?php

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Modula_Albums_Admin {

	/**
	 * Primary class constructor.
	 *
	 * @since 1.0.0
	 */
	function __construct() {

		add_action( 'wp_ajax_modula_albums_save_galleries', array( $this, 'save_galleries' ) );
		add_action( 'wp_ajax_modula_albums_save_gallery', array( $this, 'save_gallery' ) );

		// Custom Field
		add_filter( 'modula_field_type_album-lightbox-info_format', array( $this, 'field_format' ), 10, 2 );
		add_filter( 'modula_render_album-lightbox-info_field_type', array( $this, 'render_field' ) );

		// Add Modula Albums debug information
		add_filter( 'modula_debug_information', array( $this, 'debug_information' ), 35 );
	}

	private function sanitize_gallery( $image ) {

		$new_image = array();

		// This list will not contain id because we save our images based on image id.
		$image_attributes = apply_filters( 'modula_album_gallery_attributes', array(
			'id',
			'alt',
			'title',
			'caption',
			'width',
			'height',
			'cover',
			'customurl',
			'shuffleCover'
		) );

		foreach ( $image_attributes as $attribute ) {
			if ( isset( $image[ $attribute ] ) ) {

				switch ( $attribute ) {
					case 'alt':
						$new_image[ $attribute ] = sanitize_text_field( $image[ $attribute ] );
						break;
					case 'customurl':
						$new_image[ $attribute ] = esc_url( $image[ $attribute ] );
						break;
					case 'cover':
					case 'shuffleCover' :
						$new_image[ $attribute ] = sanitize_text_field( $image[ $attribute ] );
						break;
					case 'id':
					case 'width':
					case 'height':
						$new_image[ $attribute ] = absint( $image[ $attribute ] );
						break;
					case 'title':
					case 'caption' :
						$new_image[ $attribute ] = wp_filter_post_kses( $image[ $attribute ] );
						break;
					default:
						$new_image[ $attribute ] = apply_filters( 'modula_album_gallery_field_sanitization', sanitize_text_field( $image[ $attribute ] ), $image[ $attribute ], $attribute );
						break;
				}

			} else {
				$new_image[ $attribute ] = '';
			}
		}

		return $new_image;

	}

	/**
	 * Save selected galleries in Modula Albums
	 */
	public function save_galleries() {

		$nonce = $_POST['_wpnonce'];
		if ( ! wp_verify_nonce( $nonce, 'modula-ajax-save' ) ) {
			wp_send_json( array( 'status' => 'failed' ) );
		}

		if ( ! isset( $_POST['album'] ) ) {
			wp_send_json( array( 'status' => 'failed' ) );
		}

		$album_id = absint( $_POST['album'] );

		if ( 'modula-album' != get_post_type( $album_id ) ) {
			wp_send_json( array( 'status' => 'failed' ) );
		}

		if ( ! isset( $_POST['galleries'] ) ) {
			wp_send_json( array( 'status' => 'failed' ) );
		}

		$old_galleries = get_post_meta( $album_id, 'modula-album-galleries', true );
		$galleries     = json_decode( stripslashes( $_POST['galleries'] ), true );
		$new_galleries = array();

		if ( is_array( $galleries ) ) {
			foreach ( $galleries as $gallery ) {
				$new_galleries[] = $this->sanitize_gallery( $gallery );
			}
		}

		update_post_meta( $album_id, 'modula-album-galleries', $new_galleries );
		wp_send_json( array( 'status' => 'succes' ) );

	}

	/**
	 * Save gallery in Modula Album
	 */
	public function save_gallery() {

		$nonce = $_POST['_wpnonce'];
		if ( ! wp_verify_nonce( $nonce, 'modula-ajax-save' ) ) {
			wp_send_json( array( 'status' => 'failed' ) );
		}

		if ( ! isset( $_POST['album'] ) ) {
			wp_send_json( array( 'status' => 'failed' ) );
		}

		$album_id = absint( $_POST['album'] );

		if ( 'modula-album' != get_post_type( $album_id ) ) {
			wp_send_json( array( 'status' => 'failed' ) );
		}

		if ( ! isset( $_POST['gallery'] ) ) {
			wp_send_json( array( 'status' => 'failed' ) );
		}

		$gallery          = json_decode( stripslashes( $_POST['gallery'] ), true );
		$update_galleries = array();

		$old_galleries = get_post_meta( $album_id, 'modula-album-galleries', true );

		if ( $old_galleries && is_array( $old_galleries ) ) {

			foreach ( $old_galleries as $key => $old_gallery ) {
				if ( $old_gallery['id'] == $gallery['id'] ) {
					$old_galleries[ $key ] = $this->sanitize_gallery( $gallery );
				}
			}
			$update_galleries = $old_galleries;
		} else {
			$update_galleries[ $gallery['id'] ] = $this->sanitize_gallery( $gallery );
		}


		update_post_meta( $album_id, 'modula-album-galleries', $update_galleries );
		wp_send_json( array( 'status' => 'succes' ) );

	}

	public function field_format( $format, $field ) {
		$format = '<tr data-container="' . esc_attr( $field['id'] ) . '"><td colspan="2" style="padding-left:0;"><label class="th-label">%s</label>%s<div>%s</div></td></tr>';

		return $format;
	}

	public function render_field() {

		$html = '<div class="modula-album-url-settings">';
		$html .= '<ol>';
		$html .= '<li><p>Will redirect to a custom URL if one is set in Gallery Settings.</p></li>';
		$html .= '<li><p>Will redirect to a gallery URL if <a href="' . admin_url( 'edit.php?post_type=modula-gallery&page=modula' ) . '">standalone</a> is active.</p></li>';
		$html .= '</ol>';
		$html .= '</div>';

		return $html;
	}

	/**
	 * Add debug information
	 *
	 * @param $info
	 *
	 * @return mixed
	 * @since 1.0.8
	 */
	public function debug_information( $info ) {

		$standalone = get_option( 'modula_standalone' );

		$info['modula_albums_version'] = array(
			'label' => __( 'Modula Albums Version', 'modula-albums' ),
			'value' => MODULA_ALBUMS_VERSION,
			'debug' => esc_html( 'Albums version ' . MODULA_ALBUMS_VERSION ),
		);

		$info['total_albums'] = array(
			'label' => __( 'Total albums', 'modula-albums' ),
			'value' => esc_html( count( modula_get_albums() ) ),
			'debug' => esc_html( 'Total albums ' . count( modula_get_albums() ) ),
		);

		$info['albums_standalone'] = array(
			'label' => __( 'Albums standalone', 'modula-albums' ),
			'value' => ( $standalone && isset( $standalone['album']['enable_rewrite'] ) && 'enabled' == $standalone['album']['enable_rewrite'] ) ? esc_html__( 'Yes', 'modula-albums' ) : esc_html__( 'No', 'modula-albums' ),
			'debug' => ( $standalone && isset( $standalone['album']['enable_rewrite'] ) && 'enabled' == $standalone['album']['enable_rewrite'] ) ? esc_html( 'yes' ) : esc_html( 'no' ),
		);

		$info['albums_standalone_slug'] = array(
			'label' => __( 'Albums standalone slug', 'modula-albums' ),
			'value' => ( $standalone && isset( $standalone['album']['slug'] ) && '' != $standalone['album']['slug'] ) ? esc_html( $standalone['album']['slug'] ) : esc_html__( 'None', 'modula-albums' ),
			'debug' => ( $standalone && isset( $standalone['album']['slug'] ) && '' != $standalone['album']['slug'] ) ? esc_html( $standalone['album']['slug'] ) : esc_html( 'None' ),
		);

		$info['galleries_standalone'] = array(
			'label' => __( 'Galleries standalone', 'modula-albums' ),
			'value' => ( $standalone && isset( $standalone['gallery']['enable_rewrite'] ) && 'enabled' == $standalone['gallery']['enable_rewrite'] ) ? esc_html__( 'Yes', 'modula-albums' ) : esc_html__( 'No', 'modula-albums' ),
			'debug' => ( $standalone && isset( $standalone['gallery']['enable_rewrite'] ) && 'enabled' == $standalone['gallery']['enable_rewrite'] ) ? esc_html( 'yes' ) : esc_html( 'no' )
		);

		$info['galleries_standalone_slug'] = array(
			'label' => __( 'Galleries standalone slug', 'modula-albums' ),
			'value' => ( $standalone && isset( $standalone['gallery']['slug'] ) && '' != $standalone['gallery']['slug'] ) ? esc_html( $standalone['gallery']['slug'] ) : esc_html__( 'None', 'modula-albums' ),
			'debug' => ( $standalone && isset( $standalone['gallery']['slug'] ) && '' != $standalone['gallery']['slug'] ) ? esc_html( $standalone['gallery']['slug'] ) : esc_html( 'None' ),
		);

		return $info;
	}
}

new Modula_Albums_Admin();