<?php
// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Modula_Albums_Builder {

    /**
     * Primary class constructor.
     *
     * @since 1.0.0
     */
	function __construct() {}

	/**
	 * Get an instance of the field builder
	 */
	public static function get_instance() {
		static $inst;
		if ( ! $inst ) {
			$inst = new Modula_Albums_Builder();
		}
		return $inst;
	}

    /**
     * @return int
     *
     * Get Post ID
     */
	public function get_id(){
		global $id, $post;

        // Get the current post ID. If ajax, grab it from the $_POST variable.
        if ( defined( 'DOING_AJAX' ) && DOING_AJAX && array_key_exists( 'post_id', $_POST ) ) {
            $post_id = absint( $_POST['post_id'] );
        } else {
            $post_id = isset( $post->ID ) ? $post->ID : (int) $id;
        }

        return $post_id;
	}

	/**
     * Helper method for retrieving settings values.
     *
     * @since 2.0.0
     *
     * @global int $id        The current post ID.
     * @global object $post   The current post object.
     * @param string $key     The setting key to retrieve.
     * @param string $default A default value to use.
     * @return string         Key value on success, empty string on failure.
     */
    public function get_setting( $key, $default = false ) {

        // Get config
        $settings = get_post_meta( $this->get_id(), 'modula-album-settings', true );

        // Check config key exists
        if ( isset( $settings[ $key ] ) ) {
	        $value = $settings[ $key ];
        } else {
	        $value = $default ? $default : '';
        }

	    return apply_filters( 'modula_albums_admin_field_value', $value, $key, $settings );

    }

	public function render( $metabox, $post = false ) {

		switch ( $metabox ) {
			case 'gallery':
				$this->_render_gallery_metabox();
				break;
			case 'settings':
				$this->_render_settings_metabox();
				break;
			case 'shortcode':
				$this->_render_shortcode_metabox( $post );
				break;
			default:
				do_action( "modula_metabox_fields_{$metabox}" );
				break;
		}

	}

    /**
     * Create HTML for gallery metabox
     */
	private function _render_gallery_metabox() {

		$galleries = get_post_meta( $this->get_id(), 'modula-album-galleries', true );
		$helper_guidelines = $this->get_setting( 'helpergrid' );

		$max_upload_size = wp_max_upload_size();
	    if ( ! $max_upload_size ) {
	        $max_upload_size = 0;
	    }

		echo '<div class="modula-uploader-container">';
		echo '<div class="modula-upload-actions">';
		echo '<div class="upload-info-container">';
		echo '<div class="upload-info">';
		echo __( 'Galleries in the album', 'modula-albums' );
		echo '</div>';
		echo '</div>';
		echo '<div class="buttons">';
		echo '<a href="#" id="modula-wp-gallery" class="button button-primary">' . esc_html__( 'Select galleries', 'modula-albums' ) . '</a>';
		echo '</div>';
		echo '</div>';
		echo '<div id="modula-uploader-container" class="modula-uploader-inline modula-album">';
			echo '<div class="modula-error-container"></div>';
			echo '<div class="modula-uploader-inline-content modula-row">';
			echo '<div id="modula-grid" style="display:none"></div>';
			echo '</div>';
		echo '</div>';

		// Helper Guildelines Toggle
		echo '<div class="modula-helper-guidelines-container">';

			do_action( 'modula_albums_before_helper_grid' );

			echo '<div class="modula-toggle modula-helper-guidelines-wrapper">';
				echo '<input class="modula-toggle__input" type="checkbox" id="modula-helper-guidelines" name="modula-settings[helpergrid]" data-setting="modula-helper-guidelines" value="1" ' . checked( 1, $helper_guidelines, false ) . '>';
				echo '<div class="modula-toggle__items">';
					echo '<span class="modula-toggle__track"></span>';
					echo '<span class="modula-toggle__thumb"></span>';
					echo '<svg class="modula-toggle__off" width="6" height="6" aria-hidden="true" role="img" focusable="false" viewBox="0 0 6 6"><path d="M3 1.5c.8 0 1.5.7 1.5 1.5S3.8 4.5 3 4.5 1.5 3.8 1.5 3 2.2 1.5 3 1.5M3 0C1.3 0 0 1.3 0 3s1.3 3 3 3 3-1.3 3-3-1.3-3-3-3z"></path></svg>';
					echo '<svg class="modula-toggle__on" width="2" height="6" aria-hidden="true" role="img" focusable="false" viewBox="0 0 2 6"><path d="M0 0h2v6H0z"></path></svg>';
				echo '</div>';
				echo '<strong class="modula-helper-guidelines-label">' . esc_html__( 'Disable Helper Grid', 'modula-albums' ) . '</strong>';
			echo '</div>';

		do_action( 'modula_albums_after_helper_grid' );

		echo '</div>';

		echo '</div>';
	}

    /**
     * Create HTML for settings metabox
     */
	private function _render_settings_metabox() {
		$tabs = Modula_Album_Fields_Helper::get_tabs();

		// Sort tabs based on priority.
		uasort( $tabs, array( 'Modula_Helper', 'sort_data_by_priority' ) );

		$tabs_html = '';
		$tabs_content_html = '';
		$first = true;

		// Generate HTML for each tab.
		foreach ( $tabs as $tab_id => $tab ) {
			$tab['id'] = $tab_id;
			$tabs_html .= $this->_render_tab( $tab, $first );

			$fields = Modula_Album_Fields_Helper::get_fields( $tab_id );
			// Sort fields based on priority.
			uasort( $fields, array( 'Modula_Helper', 'sort_data_by_priority' ) );

			$current_tab_content = '<div id="modula-' . esc_attr( $tab['id'] ) . '" class="' . ( $first ? 'active-tab' : '' ) . '">';

			// Check if our tab have title & description
			if ( isset( $tab['title'] ) || isset( $tab['description'] ) ) {
				$current_tab_content .= '<div class="tab-content-header">';
				$current_tab_content .= '<div class="tab-content-header-title">';
				if ( isset( $tab['title'] ) && '' != $tab['title'] ) {
					$current_tab_content .= '<h2>' . esc_html( $tab['title'] ) . '</h2>';
				}
				if ( isset( $tab['description'] ) && '' != $tab['description'] ) {
					$current_tab_content .= '<div class="tab-header-tooltip-container modula-tooltip"><span>[?]</span>';
					$current_tab_content .= '<div class="tab-header-description modula-tooltip-content">' . wp_kses_post( $tab['description'] ) . '</div>';
					$current_tab_content .= '</div>';
				}
				$current_tab_content .= '</div>';

				$current_tab_content .= '<div class="tab-content-header-actions">';
				$current_tab_content .= apply_filters('modula_admin_documentation_link', '<a href="https://modula.helpscoutdocs.com/" target="_blank" class="">' . esc_html__( 'Documentation', 'modula-albums' ) . '</a>');
				$current_tab_content .= '<span> - or - </span>';
				$current_tab_content .= apply_filters('modula_admin_contact_link', '<a href="https://wp-modula.com/contact-us/" target="_blank" class="">' . esc_html__( 'Get in touch', 'modula-albums' ) . '</a>');
				$current_tab_content .= '</div>';

				$current_tab_content .= '</div>';
			}

			// Generate all fields for current tab
			$current_tab_content .= '<div class="form-table-wrapper">';
			$current_tab_content .= '<table class="form-table"><tbody>';
			foreach ( $fields as $field_id => $field ) {
				$field['id'] = $field_id;
				$current_tab_content .= $this->_render_row( $field );
			}
			$current_tab_content .= '</tbody></table>';
			// Filter to add extra content to a specific tab
			$current_tab_content .= apply_filters( 'modula_' . $tab_id . '_tab_content', '' );
			$current_tab_content .= '</div>';
			$current_tab_content .= '</div>';

			$tabs_content_html .= $current_tab_content;

			if ( $first ) {
				$first = false;
			}

		}

		$html = '<div class="modula-settings-container"><div class="modula-tabs">%s</div><div class="modula-tabs-content">%s</div>';
		printf( $html, $tabs_html, $tabs_content_html );
	}

    /**
     * Create HTML for shortcode metabox
     */
	private function _render_shortcode_metabox( $post ) {

		do_action( 'modula_albums_admin_before_shortcode_metabox', $post );

		$shortcode = '[modula-album id="' . $post->ID . '"]';
        echo '<div class="modula-copy-shortcode">';
        echo '<input type="text" value="' . esc_attr($shortcode) . '"  onclick="select()" readonly>';
        echo '<a href="#" title="'.esc_attr__('Copy shortcode','modula-albums').'" class="copy-modula-shortcode button button-primary dashicons dashicons-format-gallery" style="width:40px;"></a><span></span>';
		echo '<p class="shortcode-description">' . sprintf( esc_html__( 'You can use this to display your newly created album inside a %s post or a page %s', 'modula-albums'), '<u>', '</u>' ) .  '</p>';
        echo '</div>';

		do_action( 'modula_albums_admin_after_shortcode_metabox', $post );
	}

    /**
     * Create HTML for a tab
     */
	private function _render_tab( $tab, $first = false ) {
		$icon = '';
		$badge = '';

		if ( isset( $tab['icon'] ) ) {
			$icon = '<i class="' . esc_attr( $tab['icon'] ) . '"></i>';
		}

		if ( isset( $tab['badge'] ) ) {
			$badge = '<sup>' . esc_html( $tab['badge'] ) . '</sup>';
		}
		return '<div class="modula-tab' . ( $first ? ' active-tab' : '' ) . ' modula-' . esc_attr( $tab['id'] ) . '" data-tab="modula-' . esc_attr( $tab['id'] ) . '">' . $icon . wp_kses_post( $tab['label'] ) . $badge . '</div>';
	}

    /**
     * Create HTML for a row
     */
	private function _render_row( $field ) {

		$child = '';

		if(isset($field['is_child']) && $field['is_child'] && is_bool($field['is_child'])){
			$child = 'child_setting';
		}

		if(isset($field['is_child']) && $field['is_child'] && is_string($field['is_child'])){
			$child = $field['is_child'].'_child_setting';
		}

		$format = '<tr data-container="' . esc_attr( $field['id'] ) . '"><th scope="row" class="'.$child.'"><label>%s</label>%s</th><td>%s</td></tr>';

		if ( 'textarea' == $field['type'] || 'custom_code' == $field['type'] || 'hover-effect' == $field['type'] ) {
			$format = '<tr data-container="' . esc_attr( $field['id'] ) . '"><td colspan="2" class="'.$child.'"><label class="th-label">%s</label>%s<div>%s</div></td></tr>';
		}
		
		if( isset( $field['children'] ) && is_array( $field['children'] ) && 0 < count( $field['children'] ) ){

			$children = htmlspecialchars(json_encode( $field['children'] ), ENT_QUOTES, 'UTF-8');
			
			$parent = '';
			if( isset( $field['parent'] ) && '' != $field['parent']  ){
				$parent = 'data-parent="' . $field['parent'] . '"';
			}
			$format = '<tr data-container="' . esc_attr( $field['id'] ) . '" data-children=\'' . $children . '\' ' . $parent . '><th scope="row" class="' . esc_attr( $child ) . '"><label>%s</label>%s</th><td>%s</td></tr>';
		}

		$format = apply_filters( "modula_field_type_{$field['type']}_format", $format, $field );

		$default = '';

		// Check if our field have a default value
		if ( isset( $field['default'] ) ) {
			$default = $field['default'];
		}

		// Generate tooltip
		$tooltip = '';
		if ( isset( $field['description'] ) && '' != $field['description'] ) {
			$tooltip .= '<div class="modula-tooltip"><span>[?]</span>';
			$tooltip .= '<div class="modula-tooltip-content">' . wp_kses_post( $field['description'] ) . '</div>';
			$tooltip .= '</div>';
		}

		// Get the current value of the field
		$value = $this->get_setting( $field['id'], $default );
		return sprintf( $format, wp_kses_post( $field['name'] ), $tooltip, $this->_render_field( $field, $value ) );
	}

    /**
     * Create HTML for a field
     */
	private function _render_field( $field, $value = '' ) {
		$html = '';

		switch ( $field['type'] ) {
			case 'text':
				$html = '<input type="text" class="regular-text" name="modula-settings[' . esc_attr( $field['id'] ) . ']" data-setting="' . esc_attr( $field['id'] ) . '" value="' . esc_attr( $value ) . '">';
				break;
			case 'image-size':
				$html = '<div class="modula-image-size">';
				$html .= '<input type="text" name="modula-settings[' . esc_attr( $field['id'] ) . '][width]" data-setting="' . esc_attr( $field['id'] ) . '" value="' . ((is_array($value) && isset($value['width'])) ? esc_attr( $value['width'] ) : '') . '">';
				$html .= '<span class="modila-image-size-spacer">x</span>';
				$html .= '<input type="text" name="modula-settings[' . esc_attr( $field['id'] ) . '][height]" data-setting="' . esc_attr( $field['id'] ) . '" value="' . ((is_array($value) && isset($value['height'])) ? esc_attr( $value['height'] ) : '') . '">';
				$html .= '<span class="modila-image-size-spacer">px</span>';
				$html .= '</div>';
				break;
			case 'select' :
				$html = '<select name="modula-settings[' . esc_attr( $field['id'] ) . ']" data-setting="' . esc_attr( $field['id'] ) . '" class="regular-text">';
				foreach ( $field['values'] as $key => $option ) {
					if ( is_array( $option ) ) {
						$html .= '<optgroup label="' . esc_attr( $key ) . '">';
						foreach ( $option as $key_subvalue => $subvalue ) {
							$html .= '<option value="' . esc_attr( $key_subvalue ) . '" ' . selected( $key_subvalue, $value, false ) . '>' . esc_html( $subvalue ) . '</option>';
						}
						$html .= '</optgroup>';
					}else{
						$html .= '<option value="' . esc_attr( $key ) . '" ' . selected( $key, $value, false ) . '>' . esc_html( $option ) . '</option>';
					}
				}
				if ( isset( $field['disabled'] ) && is_array( $field['disabled'] ) ) {
					$html .= '<optgroup label="' . esc_attr( $field['disabled']['title'] ) . '">';
					foreach ( $field['disabled']['values'] as $key => $disabled ) {
						$html .= '<option value="' . esc_attr( $key ) . '" disabled >' . esc_html( $disabled ) . '</option>';
					}
					$html .= '</optgroup>';
				}
				$html .= '</select>';
				break;
			case 'dimensions-select' :
				$sizes = Modula_Album_Fields_Helper::get_image_sizes();
				$html = '<select name="modula-settings[' . esc_attr( $field['id'] ) . ']" data-setting="' . esc_attr( $field['id'] ) . '" class="regular-text">';
				$infos = '';

				foreach ( $sizes as $key => $size ) {
					$html .= '<option value="' . esc_attr( $key ) . '" ' . selected( $key, $value, false ) . '>' . esc_html( $size['label'] ) . '</option>';

					$infos .= '<div class="modula-imagesize-info" data-size="' . esc_attr( $key ) . '"><span>' . esc_html__( 'Image Size', 'modula-albums' ) . '</span>: ' . $size['dimensions']['width'] . 'x' . $size['dimensions']['height'];
					$infos .= '. <span>'.esc_html__('Crop','modula-albums').'</span>: ' . ( boolval( $size['crop'] ) ? 'true' : 'false' ) . '</div>';
				}

				$html .= '<option value="full" '. selected( 'full', $value, false ).'>' . __( 'Full', 'modula-albums' ) . '</option>';
				$html .= '<option value="custom" '.selected( 'custom', $value, false ).'>' . __( 'Custom', 'modula-albums' ) . '</option>';
				$html .= '</select>';
				if ( '' != $infos ) {
					$html .= '<div class="modula-imagesizes-infos">' . $infos . '</div>';
				}
				break;
			case 'ui-slider':
				$min  = isset( $field['min'] ) ? $field['min'] : 0;
				$max  = isset( $field['max'] ) ? $field['max'] : 100;
				$step = isset( $field['step'] ) ? $field['step'] : 1;
				if ( '' === $value ) {
					if ( isset( $field['default'] ) ) {
						$value = $field['default'];
					}else{
						$value = $min;
					}
				}
				$attributes = 'data-min="' . esc_attr( $min ) . '" data-max="' . esc_attr( $max ) . '" data-step="' . esc_attr( $step ) . '"';
				$html .= '<div class="slider-container modula-ui-slider-container">';
					$html .= '<input contenteditable="true" data-setting="' . esc_attr( $field['id'] ) . '"  name="modula-settings[' . esc_attr( $field['id'] ) . ']" type="text" class="rl-slider modula-ui-slider-input" id="input_' . esc_attr( $field['id'] ) . '" value="' . $value . '" ' . $attributes . '/>';
					$html .= '<div id="slider_' . esc_attr( $field['id'] ) . '" class="ss-slider modula-ui-slider"></div>';
				$html .= '</div>';
				break;
			case 'color' :
				$html .= '<div class="modula-colorpickers">';
				$html .= '<input id="' . esc_attr( $field['id'] ) . '" class="modula-color" data-setting="' . esc_attr( $field['id'] ) . '" name="modula-settings[' . esc_attr( $field['id'] ) . ']" value="' . esc_attr( $value ) . '">';
				$html .= '</div>';
				break;
			case "toggle":
				$html .= '<div class="modula-toggle">';
					$html .= '<input class="modula-toggle__input" type="checkbox" data-setting="' . esc_attr( $field['id'] ) . '" id="' . esc_attr( $field['id'] ) . '" name="modula-settings[' . esc_attr( $field['id'] ) . ']" value="1" ' . checked( 1, $value, false ) . '>';
					$html .= '<div class="modula-toggle__items">';
						$html .= '<span class="modula-toggle__track"></span>';
						$html .= '<span class="modula-toggle__thumb"></span>';
						$html .= '<svg class="modula-toggle__off" width="6" height="6" aria-hidden="true" role="img" focusable="false" viewBox="0 0 6 6"><path d="M3 1.5c.8 0 1.5.7 1.5 1.5S3.8 4.5 3 4.5 1.5 3.8 1.5 3 2.2 1.5 3 1.5M3 0C1.3 0 0 1.3 0 3s1.3 3 3 3 3-1.3 3-3-1.3-3-3-3z"></path></svg>';
						$html .= '<svg class="modula-toggle__on" width="2" height="6" aria-hidden="true" role="img" focusable="false" viewBox="0 0 2 6"><path d="M0 0h2v6H0z"></path></svg>';
					$html .= '</div>';
				$html .= '</div>';
				break;
			case "custom_code":
				$html = '<div class="modula-code-editor" data-syntax="' . esc_attr( $field['syntax'] ) . '">';
				$html .= '<textarea data-setting="' . esc_attr( $field['id'] ) . '" name="modula-settings[' . esc_attr( $field['id'] ) . ']" id="modula-' . esc_attr( $field['id'] ) . '" class="large-text code"  rows="10" cols="50">' . wp_kses_post($value) . '</textarea>';
				$html .= '</div>';
				break;
			case "hover-effect":

				$hovers = Modula_Albums_Hover_Effects::get_hover_effects();
				$html .= '<p class="description">' . esc_html__( 'Select one of the below hover effects.', 'modula-best-grid-gallery' ) . '</p>';


				$html .= '<div class="modula-effects-preview modula modula-gallery">';

				$html .= '<div class="modula-effects-wrapper">';

				$effect_array  = array( 'tilt_1', 'tilt_3', 'tilt_7', );
				$overlay_array = array( 'tilt_2', 'tilt_3', 'tilt_7' );
				$svg_array     = array( 'tilt_1', 'tilt_7' );
				$jtg_body      = array( 'lily', 'sadie', 'ruby', 'bubba', 'dexter', 'chico', 'ming' );

				foreach ( $hovers as $key => $name ) {

					$class = array( 'modula-item' );
					$class[] = 'effect-' . $key;

					$effect_elements = Modula_Helper::hover_effects_elements( $key );
					$effect          = '';
					$effect          .= '<div class="clearfix panel-pro-preview modula-hover-effect-item modula-items">';


						$effect .= '<input type="radio" name="modula-settings[effect]" data-setting="effect" value="' . esc_attr( $key ) . '" ' . checked( $key, $value, false ) . '>';
					

					$effect .= '<div class="modula-preview-item-container">';
					$effect .= '<div class="' . esc_attr( implode( ' ', $class ) ) . '">';

					if ( 'under' == $key ) {
						$effect .= '<div class="modula-item-image-continer"><img src="' . esc_url( MODULA_URL . '/assets/images/effect.jpg' ) . '" class="pic"></div>';
					} else {
						$effect .= '<img src="' . esc_url( MODULA_URL . '/assets/images/effect.jpg' ) . '" class="pic">';
					}

					if ( in_array( $key, $effect_array ) ) {
						$effect .= '<div class="tilter__deco tilter__deco--shine"><div></div></div>';
						if ( in_array( $key, $overlay_array ) ) {
							$effect .= '<div class="tilter__deco tilter__deco--overlay"></div>';
						}
						if ( in_array( $key, $svg_array ) ) {
							$effect .= '<div class="tilter__deco tilter__deco--lines"></div>';
						}

					}

					if ( 'none' != $key ) {

						$effect .= '<div class="figc"><div class="figc-inner">';

						if ( $effect_elements[ 'title' ] ) {
							$effect .= '<div class="jtg-title">Lorem ipsum</div>';
						}

						if ( in_array( $key, $jtg_body ) ) {
							$effect .= '<div class="jtg-body">';
						}

						if ( $effect_elements[ 'description' ] ) {
							$effect .= '<p class="description">Quisque diam erat, mollisvitae enim eget</p>';
						} else {
							$effect .= '<p class="description"></p>';
						}

						if ( $effect_elements[ 'social' ] ) {
							$effect .= '<div class="jtg-social">';
							$effect .= '<a href="#">' . Modula_Helper::get_icon( 'twitter' ) . '</a>';
							$effect .= '<a href="#">' . Modula_Helper::get_icon( 'facebook' ) . '</a>';
							$effect .= '<a href="#">' . Modula_Helper::get_icon( 'pinterest' ) . '</a>';
							$effect .= '<a href="#">' . Modula_Helper::get_icon( 'linkedin' ) . '</a>';
							$effect .= '</div>';
						}

						if ( in_array( $key, $jtg_body ) ) {
							$effect .= '</div>';
						}

						$effect .= '</div></div>';
					}

					$effect .= '</div>';
					$effect .= '<div class="modula-preview-item-content">';
					$effect .= '<h4>' . $name . '</h4>';
					if ( $effect_elements[ 'title' ] || $effect_elements[ 'description' ] || $effect_elements[ 'social' ] || $effect_elements[ 'scripts' ] ) {
						
						$effect .= '<div class="effect-compatibility">';
						$effect .= '<p class="description">' . esc_html__( 'This effect is compatible with:', 'modula-pro' );

						if ( $effect_elements[ 'title' ] ) {
							$effect .= '<span><strong> ' . esc_html__( 'Title', 'modula-pro' ) . '</strong></span>,';
						}

						if ( $effect_elements[ 'description' ] ) {
							$effect .= '<span><strong> ' . esc_html__( 'Description', 'modula-pro' ) . '</strong></span>,';
						}

						if ( $effect_elements[ 'social' ] ) {
							$effect .= '<span><strong> ' . esc_html__( 'Social Icons', 'modula-pro' ) . '</strong></span>';
						}
						$effect .= '</p>';

						if ( $effect_elements[ 'scripts' ] ) {
							$effect .= '<p class="description">' . esc_html__( 'This effect will add an extra js script to your gallery', 'modula-pro' ) . '</p>';
						} else {
							$effect .= '<p class="description">&nbsp;</p>';
						}

						$effect .= '</div>';

					}
					
					$effect .= '</div>';
					$effect .= '</div>';
					$effect .= '</div>';

					$html .= $effect;
				}

				$html .= '</div></div>';

				// Hook to change how hover effects field is rendered
				$html = apply_filters( "modula_render_hover_effect_field_type", $html, $field );

				if(isset($field['afterrow'])){
					$html .= '<p class="description '.esc_attr($field['id']).'-afterrow">'.esc_html($field['afterrow']).'</p>';
				}
				break;
			case "cursor_upload":
				$style = array(
					'upload'  => '',
					'replace' => 'display:none;',
					'delete'  => 'display:none;',
				);

				if ( 0 != absint( $value ) ) {
					$style['upload']  = 'display:none;';
					$style['replace'] = '';
					$style['delete']  = '';
				}

				$html = '<input type="hidden" name="modula-settings[' . esc_attr( $field['id'] ) . ']" id="' . esc_attr( $field['id'] ) . '" value="' . absint( $value ) . '">';
				$html .= '<div class="modula_cursor_preview">';
				if ( $value ) {
					$image = wp_get_attachment_image_src( $value );
					if ( $image ) {
						$html .= '<img src="' . esc_url( $image[0] ) . '" id="modula_cursor_preview">';
					}
				}
				$html .= '</div>';
				$html .= '<input type="button" style="' . esc_attr( $style['upload'] ) . '" class="button button-primary" id="upload_cursor_file" value="' . esc_attr__( 'Upload', 'modula-albums' ) . '">';
				$html .= '<input type="button" style="' . esc_attr( $style['replace'] ) . '" class="button button-primary" id="replace_cursor_file" value="' . esc_attr__( 'Replace', 'modula-albums' ) . '">';
				$html .= '<input type="button" style="' . esc_attr( $style['delete'] ) . '" class="button" id="delete_cursor_file" value="' . esc_attr__( 'Delete', 'modula-albums' ) . '">';
				break;
			case 'textarea-placeholder' :
				$html = '<div class="modula-textarea-placeholder">';
				$html .= '<textarea data-setting="' . esc_attr( $field['id'] ) . '" name="modula-settings[' . esc_attr( $field['id'] ) . ']" id="modula-' . esc_attr( $field['id'] ) . '" rows="5" cols="50">' . wp_kses_post( $value ) . '</textarea>';
				$html .= '</div>';
				$html .= '<div class="modula-placeholders">';
				foreach ( $field['values'] as $key => $value ) {

					$html .= "<span data-placeholder='" . esc_attr( $key ) . "' class='modula-placeholder-value'>";
					$html .= esc_html__( $value, 'modula-albums' );
					$html .= '</span>';
				}
				$html .= '</div>';
				break;
			case 'placeholder' :
				$html = '<input class="modula-placeholder-input" type="text" name="modula-settings[' . esc_attr( $field['id'] ) . ']" data-setting="' . esc_attr( $field['id'] ) . '" value="' . esc_attr( $value ) . '">';
				$html .= '<div class="modula-placeholders">';
				foreach ( $field['values'] as $key => $value ) {

					$html .= "<span data-placeholder='" . esc_attr( $key ) . "' class='modula-placeholder-value'>";
					$html .= esc_html__( $value, 'modula-albums' );
					$html .= '</span>';
				}
				$html .= '</div>';
				break;

			case 'icon-radio' :
				$wpchill_upsell = false;
				if ( class_exists( 'WPChill_Upsells' ) ) {
					// Initialize WPChill upsell class
					$args = apply_filters( 'modula_upsells_args', array(
						'shop_url' => 'https://wp-modula.com',
						'slug'     => 'modula',
					) );

					$wpchill_upsell = WPChill_Upsells::get_instance( $args );
				}
				$html .= '<div class="modula-icons-radio-wrapper">';

				if ( empty( $value ) ) {
					$value = 'grid';
				}

				foreach ( $field['values'] as $key => $name ) {

					$icon = MODULA_URL . 'assets/images/settings/' . $key . '.png';
					$html .= '<input id="modula-icon-' . esc_attr( $key ) . '" type="radio" name="modula-settings[album_type]"  data-setting="' . esc_attr( $field['id'] ) . '" value="' . esc_attr( $key ) . '" ' . checked( $key, $value, false ) . '>';
					$html .= '<label class="modula-radio-icon" for="modula-icon-' . esc_attr( $key ) . '"><img src="' . esc_url( $icon ) . '" alt="' . esc_attr( $name ) . '" title="' . esc_attr( $name ) . '" class="modula-icon-radio" /><span class="modula-icon-radio-name">' . esc_html( $name ) . '</span></label>';

				}

				if ( isset( $field['disabled'] ) ) {
					foreach ( $field['disabled']['values'] as $key => $name ) {

						if ( $wpchill_upsell && ! $wpchill_upsell->is_upgradable_addon( 'modula-' . $key ) ) {
							$class = 'modula-radio-icon-install';
						} else {
							$class = 'modula-radio-icon-disabled';
						}

						$icon = MODULA_URL . 'assets/images/settings/' . $key . '-disabled.png';
						$html .= '<label class="modula-radio-icon ' . esc_attr( $class ) . '" ><img src="' . esc_url( $icon ) . '" alt="' . esc_attr( $name ) . '" title="' . esc_attr( $name ) . '" class="modula-icon-radio" /><span class="modula-icon-radio-name">' . esc_html__( $name ) . '</span></label>';
					}
				}

				$html .= '</div>';

				if ( isset( $field['afterrow'] ) ) {
					$html .= '<p class="description ' . esc_attr( $field['id'] ) . '-afterrow">' . esc_html( $field['afterrow'] ) . '</p>';
				}
				break;
			default:
				/* Filter for render custom field types */
				$html = apply_filters( "modula_render_{$field['type']}_field_type", $html, $field, $value );
				break;
		}
		if( isset( $field['children'] ) && is_array( $field['children'] ) && 0 < count( $field['children'] ) ){

			$children = htmlspecialchars(json_encode( $field['children'] ), ENT_QUOTES, 'UTF-8');
			
			$html .= '<span class="modula_settings_accordion">' . absint( count( $field['children'] ) ) . esc_html__(' other settings') . ' </span>';
		}
		return $html;

	}

}