<?php
// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Modula_Albums_CPT {

	private $cpt_name;
	private $metaboxes = array();
	private $labels    = array();
	private $args      = array();


	/**
	 * Primary class constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		// In the case Modula PRO is not event active we don't need to show the notice to activate the license if there is none.
		if ( ! class_exists( 'Modula_PRO' ) ) {
			return;
		}

		if ( method_exists( 'Modula_PRO', 'check_for_license' ) && null === Modula_PRO::check_for_license() ) {

			add_action( 'admin_enqueue_scripts', array( $this, 'nopro_scripts' ) );
			add_action( 'admin_notices', array( $this, 'modula_pro_license_check' ) );

			return;
		}

		add_action( 'init', array( $this, 'register_cpt' ) );
		if ( class_exists( 'WPChill_Upsells' ) && is_admin() ) {

			$args           = apply_filters(
				'modula_upsells_args',
				array(
					'shop_url' => 'https://wp-modula.com',
					'slug'     => 'modula',
				)
			);
			$wpchill_upsell = WPChill_Upsells::get_instance( $args );

			add_action( 'wp_ajax_dismiss_albums_notice', array( $this, 'dismiss_albums_notice' ) );

			if ( $wpchill_upsell && ! $wpchill_upsell->is_upgradable_addon( 'modula-albums' ) ) {

				/* Fire our meta box setup function on the post editor screen. */
				add_action( 'load-post.php', array( $this, 'meta_boxes_setup' ) );
				add_action( 'load-post-new.php', array( $this, 'meta_boxes_setup' ) );

				/* Add templates for our plugin */
				add_action( 'admin_footer', array( $this, 'print_modula_templates' ) );

				$this->cpt_name = apply_filters( 'modula_albums_cpt_name', 'modula-album' );

				// Post Table Columns
				add_filter( "manage_{$this->cpt_name}_posts_columns", array( $this, 'add_column' ) );
				add_action( "manage_{$this->cpt_name}_posts_custom_column", array( $this, 'output_column' ), 10, 2 );

				// Ajax to get all galleries
				add_action( 'wp_ajax_modula_get_galleries', array( $this, 'get_galleries' ) );

				// Scripts & styles
				add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ), 20 );

				// Include necessary fieles
				require_once MODULA_ALBUMS_PATH . 'includes/admin/class-modula-albums-admin.php';
				$this->builder = new Modula_Albums_Builder();

			} else {

				add_filter( 'modula_album_cpt_args', array( $this, 'remove_albums_menu_item' ) );
			}

		}

	}

	/**
	 * Register CPT
	 */
	public function register_cpt() {
		if ( ! class_exists( 'Modula_Pro' ) ) {
			return;
		}

		$this->labels = apply_filters(
			'modula_album_cpt_labels',
			array(
				'name'                  => esc_html__( 'Albums', 'modula-albums' ),
				'singular_name'         => esc_html__( 'Album', 'modula-albums' ),
				'menu_name'             => esc_html__( 'Modula', 'modula-albums' ),
				'name_admin_bar'        => esc_html__( 'Modula Albums', 'modula-albums' ),
				'archives'              => esc_html__( 'Item Archives', 'modula-albums' ),
				'attributes'            => esc_html__( 'Item Attributes', 'modula-albums' ),
				'parent_item_colon'     => esc_html__( 'Parent Item:', 'modula-albums' ),
				'all_items'             => esc_html__( 'Albums', 'modula-albums' ),
				'add_new_item'          => esc_html__( 'Add New Item', 'modula-albums' ),
				'add_new'               => esc_html__( 'Add New', 'modula-albums' ),
				'new_item'              => esc_html__( 'New Item', 'modula-albums' ),
				'edit_item'             => esc_html__( 'Edit Item', 'modula-albums' ),
				'update_item'           => esc_html__( 'Update Item', 'modula-albums' ),
				'view_item'             => esc_html__( 'View Item', 'modula-albums' ),
				'view_items'            => esc_html__( 'View Items', 'modula-albums' ),
				'search_items'          => esc_html__( 'Search Item', 'modula-albums' ),
				'not_found'             => '<a href="' . admin_url( 'post-new.php?post_type=modula-album' ) . '">' . esc_html__( 'Add your first album ', 'modula-albums' ) . '</a>' . esc_html__( 'now or check out our ', 'modula-albums' ) . '<a href="' . esc_url( 'https://modula.helpscoutdocs.com/' ) . '" target="_blank">' . esc_html__( 'documentation ', 'modula-albums' ) . '</a>' . esc_html__( 'if you need help figuring things out.', 'modula-albums' ),
				'not_found_in_trash'    => esc_html__( 'Not found in Trash', 'modula-albums' ),
				'featured_image'        => esc_html__( 'Featured Image', 'modula-albums' ),
				'set_featured_image'    => esc_html__( 'Set featured image', 'modula-albums' ),
				'remove_featured_image' => esc_html__( 'Remove featured image', 'modula-albums' ),
				'use_featured_image'    => esc_html__( 'Use as featured image', 'modula-albums' ),
				'insert_into_item'      => esc_html__( 'Insert into item', 'modula-albums' ),
				'uploaded_to_this_item' => esc_html__( 'Uploaded to this item', 'modula-albums' ),
				'items_list'            => esc_html__( 'Items list', 'modula-albums' ),
				'items_list_navigation' => esc_html__( 'Items list navigation', 'modula-albums' ),
				'filter_items_list'     => esc_html__( 'Filter items list', 'modula-albums' ),
			),
			$this->labels
		);

		$this->args = apply_filters(
			'modula_album_cpt_args',
			array(
				'label'               => esc_html__( 'Modula Album', 'modula-albums' ),
				'description'         => esc_html__( 'Modula is the most powerful, user-friendly WordPress gallery plugin. Add galleries, masonry grids and more in a few clicks.', 'modula-albums' ),
				'supports'            => array( 'title' ),
				'public'              => false,
				'show_ui'             => true,
				'show_in_menu'        => 'edit.php?post_type=modula-gallery',
				'menu_position'       => 25,
				'menu_icon'           => 'data:image/svg+xml;base64,' . base64_encode( '<svg xmlns="http://www.w3.org/2000/svg" width="20" viewBox="0 0 32 32"><path fill="#f0f5fa" d="M9.3 25.3c-2.4-0.7-4.7-1.4-7.1-2.1 2.4-3.5 4.7-7 7-10.5C9.3 12.9 9.3 24.9 9.3 25.3z"/><path fill="#f0f5fa" d="M9.6 20.1c3.7 2 7.4 3.9 11.1 5.9 -0.1 0.1-5 5-5.2 5.2C13.6 27.5 11.6 23.9 9.6 20.1 9.6 20.2 9.6 20.2 9.6 20.1z"/><path fill="#f0f5fa" d="M22.3 11.9c-3.7-2-7.4-4-11-6 0 0 0 0 0 0 0 0 0 0 0 0 1.7-1.7 3.4-3.3 5.1-5 0 0 0 0 0.1-0.1C18.5 4.5 20.4 8.2 22.3 11.9 22.4 11.9 22.3 11.9 22.3 11.9z"/><path fill="#f0f5fa" d="M4.7 15c-0.6-2.4-1.2-4.7-1.8-7 0.2 0 11.9 0.6 12.7 0.6 0 0 0 0 0 0 0 0 0 0 0 0 -3.6 2.1-7.2 4.2-10.7 6.3C4.8 15 4.8 15 4.7 15z"/><path fill="#f0f5fa" d="M22.9 19.6c-0.2-4.2-0.3-8.3-0.5-12.5 2.4 0.6 4.8 1.2 7.1 1.8C27.4 12.4 25.1 16 22.9 19.6 22.9 19.6 22.9 19.6 22.9 19.6z"/><path fill="#f0f5fa" d="M27.7 16.8c0.6 2.4 1.2 4.7 1.9 7.1 -4.2-0.2-8.5-0.4-12.7-0.5 0 0 0 0 0 0C20.5 21.2 24.1 19 27.7 16.8z"/></svg>' ),
				'show_in_admin_bar'   => true,
				'show_in_nav_menus'   => false,
				'can_export'          => true,
				'has_archive'         => false,
				'exclude_from_search' => true,
				'rewrite'             => false,
				'show_in_rest'        => true,
			)
		);

		$this->metaboxes = apply_filters(
			'modula_album_cpt_metaboxes',
			array(
				'modula-album-galleries' => array(
					'title'    => esc_html__( 'Galleries', 'modula-albums' ),
					'callback' => 'output_album_galleries',
					'context'  => 'normal',
				),
				'modula-album-settings'  => array(
					'title'    => esc_html__( 'Settings', 'modula-albums' ),
					'callback' => 'output_album_settings',
					'context'  => 'normal',
				),
				'modula-album-shortcode' => array(
					'title'    => esc_html__( 'Shortcode', 'modula-albums' ),
					'callback' => 'output_album_shortcode',
					'context'  => 'side',
				),
			)
		);

		$this->cpt_name = apply_filters( 'modula_album_cpt_name', 'modula-album' );
		$args           = $this->args;
		$args['labels'] = $this->labels;

		register_post_type( $this->cpt_name, $args );

	}

	public function meta_boxes_setup() {

		/* Add meta boxes on the 'add_meta_boxes' hook. */
		add_action( 'add_meta_boxes', array( $this, 'add_meta_boxes' ) );

		/* Save post meta on the 'save_post' hook. */
		add_action( 'save_post', array( $this, 'save_meta_boxes' ), 10, 2 );
	}

	public function admin_scripts( $hook ) {

		global $id, $post;

		// Get current screen.
		$screen = get_current_screen();

		// Enqueue tooltip on standalone tab
		if ( 'modula-gallery_page_modula' == $screen->base ) {
			wp_enqueue_style( 'modula-cpt-style', MODULA_URL . 'assets/css/admin/modula-cpt.css', null, MODULA_LITE_VERSION );
		}

		// Check if is modula custom post type
		if ( 'modula-album' !== $screen->post_type ) {
			return;
		}

		// Set the post_id
		$post_id = isset( $post->ID ) ? $post->ID : (int) $id;

		// Enqueue Shrotcode Saving script
		wp_enqueue_style( 'modula-albums-shortcode-copy', MODULA_ALBUMS_URL . 'assets/css/admin/modula-albums-shortcode-copy.css', null, MODULA_ALBUMS_VERSION );
		wp_enqueue_script( 'modula-albums-shortcode-copy', MODULA_ALBUMS_URL . 'assets/js/admin/modula-albums-shortcode-copy.js', array(), MODULA_ALBUMS_VERSION, true );

		if ( 'post-new.php' == $hook || 'post.php' == $hook ) {

			// Media Scripts
			wp_enqueue_media(
				array(
					'post' => $post_id,
				)
			);

			$modula_helper = array(
				'items'            => array(),
				'settings'         => array(),
				'strings'          => array(),
				'id'               => $post_id,
				'_wpnonce'         => wp_create_nonce( 'modula-ajax-save' ),
				'ajax_url'         => admin_url( 'admin-ajax.php' ),
				'gallery_settings' => apply_filters(
					'modula-album-item-settings',
					array(
						'id',
						'title',
						'caption',
						'alt',
						'cover',
					)
				),
			);

			// Get all items from current gallery.
			$galleries = get_post_meta( $post_id, 'modula-album-galleries', true );
			if ( is_array( $galleries ) && ! empty( $galleries ) ) {

				foreach ( $galleries as $gallery ) {

					if ( ! is_numeric( $gallery['id'] ) ) {
						continue;
					}

					$gallery_obj = get_post( $gallery['id'] );

					if ( ! $gallery_obj ) {
						continue;
					}

					if ( 'modula-gallery' != $gallery_obj->post_type ) {
						continue;
					}

					$images = get_post_meta( $gallery['id'], 'modula-images', true );

					if ( empty( $images ) ) {
						continue;
					}

					$image = $images[0];

					$gallery_args = array(
						'id'           => '',
						'title'        => '',
						'caption'      => '',
						'alt'          => '',
						'cover'        => '',
						'coverURL'     => '',
						'customurl'    => '',
						'shuffleCover' => '',
						'thumb'        => array(),
					);

					$gallery_args = wp_parse_args( $gallery, $gallery_args );

					if ( '' == $gallery_args['title'] ) {
						$gallery_args['title'] = get_the_title( $gallery['id'] );
					}

					$attachment            = wp_prepare_attachment_for_js( $image['id'] );
					$image_url             = wp_get_attachment_image_src( $image['id'], 'large' );
					$image_full            = wp_get_attachment_image_src( $image['id'], 'full' );
					$gallery_args['thumb'] = array(
						'id'          => $image['id'],
						'orientation' => $attachment['orientation'],
						'full'        => $image_full[0],
						'thumbnail'   => $image_url[0],
					);

					if ( '' != $gallery_args['cover'] ) {
						$image_large = wp_get_attachment_image_src( absint( $gallery_args['cover'] ), 'large' );
						$image_thumb = wp_get_attachment_image_src( absint( $gallery_args['cover'] ), 'thumbnail' );
						if ( $image_large ) {
							$gallery_args['coverURL'] = $image_large[0];
						}
						if ( $image_thumb ) {
							$gallery_args['coverThumb'] = $image_thumb[0];
						}
					}

					// @todo: let's see what settings we will have for each gallery
					$modula_helper['items'][] = $gallery_args;

				}
			}

			// Get current album settings.
			$settings = get_post_meta( $post_id, 'modula-album-settings', true );
			if ( is_array( $settings ) ) {
				$modula_helper['settings'] = wp_parse_args( $settings, Modula_Album_Fields_Helper::get_defaults() );
			} else {
				$modula_helper['settings'] = Modula_Album_Fields_Helper::get_defaults();
			}

			// Styles
			wp_enqueue_style( 'wp-color-picker' );
			wp_enqueue_style( 'jquery-ui', MODULA_URL . 'assets/css/admin/jquery-ui.min.css', null, MODULA_ALBUMS_VERSION );
			wp_enqueue_style( 'modula-cpt-style', MODULA_URL . 'assets/css/admin/modula-cpt.css', null, MODULA_LITE_VERSION );
			wp_enqueue_style( 'admin-modula-album-grid-system', MODULA_ALBUMS_URL . 'assets/css/admin/modula-albums-admin.css', null, MODULA_ALBUMS_VERSION );
			wp_enqueue_style( 'modula-notices-style', MODULA_URL . 'assets/css/admin/modula-notices.css', null, MODULA_LITE_VERSION );
			wp_enqueue_style( 'modula-pro-effects', MODULA_PRO_URL . 'assets/css/effects.min.css' );

			// Scripts
			wp_enqueue_style( 'underscore' );
			wp_enqueue_style( 'wp-backbone' );

			// Modula scripts used for Tilt Hover Effects
			wp_register_script( 'modula-pro-tilt', MODULA_PRO_URL . 'assets/js/modula-pro-tilt.min.js', array( 'jquery' ), MODULA_PRO_VERSION, true );
			wp_enqueue_script( 'modula-pro-tilt' );

			wp_enqueue_script( 'modula-resize-senzor', MODULA_URL . 'assets/js/admin/resizesensor.js', array( 'jquery' ), MODULA_ALBUMS_VERSION, true );
			wp_enqueue_script( 'modula-packery', MODULA_URL . 'assets/js/admin/packery.min.js', array( 'jquery', 'jquery-ui-core', 'jquery-ui-widget', 'jquery-ui-droppable', 'jquery-ui-resizable', 'jquery-ui-draggable' ), MODULA_ALBUMS_VERSION, true );
			wp_register_script( 'modula-albums-cursor', MODULA_ALBUMS_URL . 'assets/js/admin/modula-albums-cursor.js', array( 'jquery' ), MODULA_ALBUMS_VERSION, true );
			wp_enqueue_script( 'modula-albums-cursor' );

			wp_enqueue_script( 'modula-settings', MODULA_ALBUMS_URL . 'assets/js/admin/wp-modula-settings.js', array( 'jquery', 'jquery-ui-slider', 'wp-color-picker', 'jquery-ui-sortable', 'wp-backbone' ), MODULA_ALBUMS_VERSION, true );
			wp_enqueue_script( 'modula-save', MODULA_ALBUMS_URL . 'assets/js/admin/wp-modula-save.js', array(), MODULA_ALBUMS_VERSION, true );
			wp_enqueue_script( 'modula-items', MODULA_ALBUMS_URL . 'assets/js/admin/wp-modula-items.js', array(), MODULA_ALBUMS_VERSION, true );
			wp_enqueue_script( 'modula-modal', MODULA_ALBUMS_URL . 'assets/js/admin/wp-modula-modal.js', array(), MODULA_ALBUMS_VERSION, true );
			wp_enqueue_script( 'modula-editmodal', MODULA_ALBUMS_URL . 'assets/js/admin/wp-modula-edit-modal.js', array(), MODULA_ALBUMS_VERSION, true );
			wp_enqueue_script( 'modula-gallery', MODULA_ALBUMS_URL . 'assets/js/admin/wp-modula-gallery.js', array(), MODULA_ALBUMS_VERSION, true );
			wp_enqueue_script( 'modula-galleries', MODULA_ALBUMS_URL . 'assets/js/admin/wp-modula-galleries.js', array(), MODULA_ALBUMS_VERSION, true );
			wp_add_inline_script( 'modula-galleries', 'modulaAlbumsAdmin = ' . wp_json_encode( array( '_wpnonce' => wp_create_nonce( 'modula_albums_admin' ) ) ) . ';', 'before' );
			wp_enqueue_script( 'modula-albums-conditions', MODULA_ALBUMS_URL . 'assets/js/admin/wp-modula-conditions.js', array(), MODULA_ALBUMS_VERSION, true );
			wp_enqueue_script( 'modula-hovereffects', MODULA_ALBUMS_URL . 'assets/js/admin/wp-modula-hovereffects.js', array(), MODULA_ALBUMS_VERSION, true );

			do_action( 'modula_album_scripts_before_wp_modula' );

			wp_enqueue_script( 'modula', MODULA_ALBUMS_URL . 'assets/js/admin/wp-modula.js', array(), MODULA_ALBUMS_VERSION, true );
			wp_localize_script( 'modula', 'modulaHelper', $modula_helper );

			do_action( 'modula_album_scripts_after_wp_modula' );

		}

	}

	public function add_column( $columns ) {

		$date = $columns['date'];
		unset( $columns['date'] );
		$columns['shortcode'] = esc_html__( 'Album Shortcode', 'modula-albums' );
		$columns['date']      = $date;
		return $columns;

	}

	public function output_column( $column, $post_id ) {

		if ( 'shortcode' == $column ) {
			$shortcode = '[modula-album id="' . $post_id . '"]';
			echo '<div class="modula-copy-shortcode">';
			echo '<input type="text" value="' . esc_attr( $shortcode ) . '"  onclick="select()" readonly>';
			echo '<a href="#" title="' . esc_attr__( 'Copy shortcode', 'modula-albums' ) . '" class="copy-modula-shortcode button button-primary dashicons dashicons-format-gallery" style="width:40px;"></a><span></span>';
			echo '</div>';
		}

	}

	public function add_meta_boxes() {

		global $post;

		if ( $this->metaboxes ) {

			foreach ( $this->metaboxes as $metabox_id => $metabox ) {

				if ( 'modula-shortcode' == $metabox_id && 'auto-draft' == $post->post_status ) {
					break;
				}

				add_meta_box(
					$metabox_id,      // Unique ID
					$metabox['title'],    // Title
					array( $this, $metabox['callback'] ),   // Callback function
					$this->cpt_name,         // Admin page (or post type)
					$metabox['context'],         // Context
					'high'         // Priority
				);
			}
		}

	}

	public function output_album_galleries() {
		$this->builder->render( 'gallery' );
	}

	public function output_album_settings() {
		$this->builder->render( 'settings' );
	}

	public function output_album_shortcode( $post ) {
		$this->builder->render( 'shortcode', $post );
	}

	public function save_meta_boxes( $post_id, $post ) {

		$resizer = new Modula_Image();

		/* Get the post type object. */
		$post_type = get_post_type_object( $post->post_type );

		/* Check if the current user has permission to edit the post. */
		if ( ! current_user_can( $post_type->cap->edit_post, $post_id ) || 'modula-album' != $post_type->name ) {
			return $post_id;
		}

		// We need to resize our images
		$images = get_post_meta( $post_id, 'modula-album-galleries', true );
		if ( $images && is_array( $images ) ) {
			if ( isset( $_POST['modula-settings']['image_dimensions'] ) && apply_filters( 'modula_album_resize_images', true, $_POST['modula-settings'] ) ) {

				$gallery_type = 'custom-grid' == $_POST['modula-settings']['type'] ? 'custom-grid' : 'creative-gallery';
				$img_size     = absint( $_POST['modula-settings']['image_dimensions'] );

				if ( 'custom' == $_POST['modula-settings']['image_size'] ) {

					foreach ( $images as $image ) {

						$grid_sizes = array(
							'width'  => isset( $image['width'] ) ? absint( $image['width'] ) : 1,
							'height' => isset( $image['height'] ) ? absint( $image['height'] ) : 1,
						);
						$sizes      = $resizer->get_image_size( $image['id'], $img_size, $gallery_type, $grid_sizes );
						if ( ! is_wp_error( $sizes ) ) {
							$resizer->resize_image( $sizes['url'], $sizes['width'], $sizes['height'] );
						}
					}
				}
			}
		}

		if ( isset( $_POST['modula-settings'] ) ) {

			$fields_with_tabs = Modula_Album_Fields_Helper::get_fields( 'all' );

			// Here we will save all our settings
			$modula_settings = array();

			// Values for selects
			$effect_values = Modula_Albums_Hover_Effects::get_hover_effects_keys();

			// We will save only our settings.
			foreach ( $fields_with_tabs as $tab => $fields ) {

				// We will iterate throught all fields of current tab
				foreach ( $fields as $field_id => $field ) {
					if ( isset( $_POST['modula-settings'][ $field_id ] ) ) {

						switch ( $field_id ) {

							// leave commented lines for future references. if not commented and not with break switch malfunctions
							/*
							 case 'type':
							case 'gutter':
							case 'width':
							case 'display_image_count':
							case 'image_dimensions':
							case 'crop_images':
							case 'enable_lightbox':
							case 'animation_effect':
							case 'animation_duration':
							case 'transition_effect':
							case 'transition_duration':
							case 'enable_slideshow':
							case 'slideshow_duration':
							case 'enable_thumbs':
							case 'thumbs_axis':*/
							case 'effect':
								if ( in_array( $_POST['modula-settings'][ $field_id ], $effect_values ) ) {
									$modula_settings[ $field_id ] = $_POST['modula-settings'][ $field_id ];
								} else {
									$modula_settings[ $field_id ] = 'pufrobo';
								}
								break;
							/*
							case 'titleColor':
							case 'captionColor':
							case 'hide_title':
							case 'hide_description':
							case 'titleFontSize':
							case 'captionFontSize':
							case 'lazy_load':
							case 'enable_responsive':
							case 'tablet_columns':
							case 'mobile_columns':
							case 'mobileTitleFontSize':
							case 'mobileCaptionFontSize':*/
							default:
								if ( is_array( $_POST['modula-settings'][ $field_id ] ) ) {
									$sanitized                    = array_map( 'sanitize_text_field', $_POST['modula-settings'][ $field_id ] );
									$modula_settings[ $field_id ] = apply_filters( 'modula_settings_field_sanitization', $sanitized, $_POST['modula-settings'][ $field_id ], $field_id, $field );
								} else {
									$modula_settings[ $field_id ] = apply_filters( 'modula_settings_field_sanitization', sanitize_text_field( $_POST['modula-settings'][ $field_id ] ), $_POST['modula-settings'][ $field_id ], $field_id, $field );
								}

								break;
						}
					} else {
						if ( 'toggle' == $field['type'] ) {
							$modula_settings[ $field_id ] = '0';
						} else {
							$modula_settings[ $field_id ] = '';
						}
					}
				}
			}

			// Save the value of helpergrid
			if ( isset( $_POST['modula-settings']['helpergrid'] ) ) {
				$modula_settings['helpergrid'] = 1;
			} else {
				$modula_settings['helpergrid'] = 0;
			}

			// Add settings to album meta
			update_post_meta( $post_id, 'modula-album-settings', $modula_settings );

		}

	}

	public function print_modula_templates() {
		// Get current screen.
		$screen = get_current_screen();

		// Check if is modula custom post type
		if ( 'modula-album' !== $screen->post_type ) {
			return;
		}

		include 'modula-albums-js-templates.php';
	}

	public function get_galleries() {

		check_ajax_referer( 'modula_albums_admin', '_wpnonce' );

		$return = array();

		$galleries = get_posts(
			array(
				'post_type'      => 'modula-gallery',
				'posts_per_page' => -1,
				'post_status'    => array( 'publish' ),
			)
		);

		foreach ( $galleries as $gallery ) {

			$images = get_post_meta( $gallery->ID, 'modula-images', true );

			if ( empty( $images ) ) {
				continue;
			}

			$image                        = $images['0'];
			$attachment                   = wp_prepare_attachment_for_js( $image['id'] );
			$image_url                    = wp_get_attachment_image_src( $image['id'], 'large' );
			$image_full                   = wp_get_attachment_image_src( $image['id'], 'full' );
			$gallery_args[ $gallery->ID ] = array(
				'id'      => $gallery->ID,
				'title'   => $gallery->post_title,
				'caption' => '',
				'alt'     => '',
				'cover'   => array(),
				'thumb'   => array(
					'id'          => $image['id'],
					'orientation' => $attachment['orientation'],
					'full'        => $image_full[0],
					'thumbnail'   => $image_url[0],
				),
			);
				
			$return[] = $gallery_args[ $gallery->ID ];

		}

		wp_send_json_success( $return );
		die();

	}

	/**
	 * Replace submit meta box
	 *
	 * @since 1.0.5
	 */
	public function replace_submit_meta_box() {
		global $post;
		$post_type = 'modula-album';
		remove_meta_box( 'submitdiv', $post_type, 'side' );
		add_meta_box( 'submitdiv', __( 'Publish' ), array( $this, 'post_submit_meta_box' ), $post_type, 'side', 'high' );

	}

	/**
	 * New submit meta box
	 *
	 * @since 1.0.5
	 */
	public function post_submit_meta_box() {
		global $action, $post;
		$post_type        = $post->post_type; // get current post_type
		$post_type_object = get_post_type_object( $post_type );
		$can_publish      = current_user_can( $post_type_object->cap->publish_posts );
		?>
		<div class="submitbox" id="submitpost">

			<div id="minor-publishing">

				<?php // Hidden submit button early on so that the browser chooses the right button when form is submitted with Return key ?>
				<div style="display:none;">
					<?php submit_button( __( 'Save' ), '', 'save' ); ?>
				</div>

				<div id="minor-publishing-actions">
					<div id="save-action">
						<?php
						if ( 'publish' != $post->post_status && 'future' != $post->post_status && 'pending' != $post->post_status ) {
							$private_style = '';
							if ( 'private' == $post->post_status ) {
								$private_style = 'style="display:none"';
							}
							?>
							<input <?php echo $private_style; ?> type="submit" name="save" id="save-post" value="<?php esc_attr_e( 'Save Draft' ); ?>" class="button" />
							<span class="spinner"></span>
						<?php } elseif ( 'pending' == $post->post_status && $can_publish ) { ?>
							<input type="submit" name="save" id="save-post" value="<?php esc_attr_e( 'Save as Pending' ); ?>" class="button" />
							<span class="spinner"></span>
						<?php } ?>
					</div>
					<?php if ( is_post_type_viewable( $post_type_object ) ) : ?>
						<div id="preview-action">
							<?php
							$preview_link = esc_url( get_preview_post_link( $post ) );
							if ( 'publish' == $post->post_status ) {
								$preview_button_text = esc_html__( 'Preview Changes', 'modula-albums' );
							} else {
								$preview_button_text = esc_html__( 'Preview', 'modula-albums' );
							}

							$preview_button = sprintf(
								'%1$s<span class="screen-reader-text"> %2$s</span>',
								$preview_button_text,
								/* translators: Accessibility text. */
								esc_html__( '(opens in a new tab)', 'modula-albums' )
							);
							?>
							<a class="preview button" href="<?php echo $preview_link; ?>" target="wp-preview-<?php echo (int) $post->ID; ?>" id="post-preview"><?php echo $preview_button; ?></a>
							<input type="hidden" name="wp-preview" id="wp-preview" value="" />
						</div>
					<?php endif; // public post type ?>
					<?php
					/**
					 * Fires before the post time/date setting in the Publish meta box.
					 *
					 * @since 4.4.0
					 *
					 * @param WP_Post $post WP_Post object for the current post.
					 */
					do_action( 'post_submitbox_minor_actions', $post );
					?>
					<div class="clear"></div>
				</div><!-- #minor-publishing-actions -->

				<div id="misc-publishing-actions">

					<div class="misc-pub-section misc-pub-post-status">
						<?php _e( 'Status:' ); ?> <span id="post-status-display">
			<?php

			switch ( $post->post_status ) {
				case 'private':
					_e( 'Privately Published' );
					break;
				case 'publish':
					_e( 'Published' );
					break;
				case 'future':
					_e( 'Scheduled' );
					break;
				case 'pending':
					_e( 'Pending Review' );
					break;
				case 'draft':
				case 'auto-draft':
					_e( 'Draft' );
					break;
			}
			?>
</span>
						<?php
						if ( 'publish' == $post->post_status || 'private' == $post->post_status || $can_publish ) {
							$private_style = '';
							if ( 'private' == $post->post_status ) {
								$private_style = 'style="display:none"';
							}
							?>
							<a href="#post_status" <?php echo $private_style; ?> class="edit-post-status hide-if-no-js" role="button"><span aria-hidden="true"><?php _e( 'Edit' ); ?></span> <span class="screen-reader-text"><?php _e( 'Edit status' ); ?></span></a>

							<div id="post-status-select" class="hide-if-js">
								<input type="hidden" name="hidden_post_status" id="hidden_post_status" value="<?php echo esc_attr( ( 'auto-draft' == $post->post_status ) ? 'draft' : $post->post_status ); ?>" />
								<label for="post_status" class="screen-reader-text"><?php _e( 'Set status' ); ?></label>
								<select name="post_status" id="post_status">
									<?php if ( 'publish' == $post->post_status ) : ?>
										<option<?php selected( $post->post_status, 'publish' ); ?> value='publish'><?php _e( 'Published' ); ?></option>
									<?php elseif ( 'private' == $post->post_status ) : ?>
										<option<?php selected( $post->post_status, 'private' ); ?> value='publish'><?php _e( 'Privately Published' ); ?></option>
									<?php elseif ( 'future' == $post->post_status ) : ?>
										<option<?php selected( $post->post_status, 'future' ); ?> value='future'><?php _e( 'Scheduled' ); ?></option>
									<?php endif; ?>
									<option<?php selected( $post->post_status, 'pending' ); ?> value='pending'><?php _e( 'Pending Review' ); ?></option>
									<?php if ( 'auto-draft' == $post->post_status ) : ?>
										<option<?php selected( $post->post_status, 'auto-draft' ); ?> value='draft'><?php _e( 'Draft' ); ?></option>
									<?php else : ?>
										<option<?php selected( $post->post_status, 'draft' ); ?> value='draft'><?php _e( 'Draft' ); ?></option>
									<?php endif; ?>
								</select>
								<a href="#post_status" class="save-post-status hide-if-no-js button"><?php _e( 'OK' ); ?></a>
								<a href="#post_status" class="cancel-post-status hide-if-no-js button-cancel"><?php _e( 'Cancel' ); ?></a>
							</div>

						<?php } ?>
					</div><!-- .misc-pub-section -->

					<div class="misc-pub-section misc-pub-visibility" id="visibility">
						<?php _e( 'Visibility:' ); ?> <span id="post-visibility-display">
							<?php

							if ( 'private' == $post->post_status ) {
								$post->post_password = '';
								$visibility          = 'private';
								$visibility_trans    = __( 'Private' );
							} elseif ( ! empty( $post->post_password ) ) {
								$visibility       = 'password';
								$visibility_trans = __( 'Password protected' );
							} elseif ( $post_type == 'post' && is_sticky( $post->ID ) ) {
								$visibility       = 'public';
								$visibility_trans = __( 'Public, Sticky' );
							} else {
								$visibility       = 'public';
								$visibility_trans = __( 'Public' );
							}

							echo esc_html( $visibility_trans );
							?>
</span>
						<?php if ( $can_publish ) { ?>
							<a href="#visibility" class="edit-visibility hide-if-no-js" role="button"><span aria-hidden="true"><?php _e( 'Edit' ); ?></span> <span class="screen-reader-text"><?php _e( 'Edit visibility' ); ?></span></a>

							<div id="post-visibility-select" class="hide-if-js">
								<input type="hidden" name="hidden_post_password" id="hidden-post-password" value="<?php echo esc_attr( $post->post_password ); ?>" />
								<?php if ( $post_type == 'post' ) : ?>
									<input type="checkbox" style="display:none" name="hidden_post_sticky" id="hidden-post-sticky" value="sticky" <?php checked( is_sticky( $post->ID ) ); ?> />
								<?php endif; ?>
								<input type="hidden" name="hidden_post_visibility" id="hidden-post-visibility" value="<?php echo esc_attr( $visibility ); ?>" />
								<input type="radio" name="visibility" id="visibility-radio-public" value="public" <?php checked( $visibility, 'public' ); ?> /> <label for="visibility-radio-public" class="selectit"><?php _e( 'Public' ); ?></label><br />
								<?php if ( $post_type == 'post' && current_user_can( 'edit_others_posts' ) ) : ?>
									<span id="sticky-span"><input id="sticky" name="sticky" type="checkbox" value="sticky" <?php checked( is_sticky( $post->ID ) ); ?> /> <label for="sticky" class="selectit"><?php _e( 'Stick this post to the front page' ); ?></label><br /></span>
								<?php endif; ?>
								<input type="radio" name="visibility" id="visibility-radio-password" value="password" <?php checked( $visibility, 'password' ); ?> /> <label for="visibility-radio-password" class="selectit"><?php _e( 'Password protected' ); ?></label><br />
								<span id="password-span"><label for="post_password"><?php _e( 'Password:' ); ?></label> <input type="text" name="post_password" id="post_password" value="<?php echo esc_attr( $post->post_password ); ?>"  maxlength="255" /><br /></span>
								<input type="radio" name="visibility" id="visibility-radio-private" value="private" <?php checked( $visibility, 'private' ); ?> /> <label for="visibility-radio-private" class="selectit"><?php _e( 'Private' ); ?></label><br />

								<p>
									<a href="#visibility" class="save-post-visibility hide-if-no-js button"><?php _e( 'OK' ); ?></a>
									<a href="#visibility" class="cancel-post-visibility hide-if-no-js button-cancel"><?php _e( 'Cancel' ); ?></a>
								</p>
							</div>
						<?php } ?>

					</div><!-- .misc-pub-section -->

					<?php
					/* translators: Publish box date string. 1: Date, 2: Time. See https://secure.php.net/date */
					$date_string = __( '%1$s at %2$s' );
					/* translators: Publish box date format, see https://secure.php.net/date */
					$date_format = _x( 'M j, Y', 'publish box date format' );
					/* translators: Publish box time format, see https://secure.php.net/date */
					$time_format = _x( 'H:i', 'publish box time format' );

					if ( 0 != $post->ID ) {
						if ( 'future' == $post->post_status ) { // scheduled for publishing at a future date
							/* translators: Post date information. %s: Date on which the post is currently scheduled to be published. */
							$stamp = __( 'Scheduled for: %s' );
						} elseif ( 'publish' == $post->post_status || 'private' == $post->post_status ) { // already published
							/* translators: Post date information. %s: Date on which the post was published. */
							$stamp = __( 'Published on: %s' );
						} elseif ( '0000-00-00 00:00:00' == $post->post_date_gmt ) { // draft, 1 or more saves, no date specified
							$stamp = __( 'Publish <b>immediately</b>' );
						} elseif ( time() < strtotime( $post->post_date_gmt . ' +0000' ) ) { // draft, 1 or more saves, future date specified
							/* translators: Post date information. %s: Date on which the post is to be published. */
							$stamp = __( 'Schedule for: %s' );
						} else { // draft, 1 or more saves, date specified
							/* translators: Post date information. %s: Date on which the post is to be published. */
							$stamp = __( 'Publish on: %s' );
						}
						$date = sprintf(
							$date_string,
							date_i18n( $date_format, strtotime( $post->post_date ) ),
							date_i18n( $time_format, strtotime( $post->post_date ) )
						);
					} else { // draft (no saves, and thus no date specified)
						$stamp = __( 'Publish <b>immediately</b>' );
						$date  = sprintf(
							$date_string,
							date_i18n( $date_format, strtotime( current_time( 'mysql' ) ) ),
							date_i18n( $time_format, strtotime( current_time( 'mysql' ) ) )
						);
					}

					if ( ! empty( $args['args']['revisions_count'] ) ) :
						?>
						<div class="misc-pub-section misc-pub-revisions">
							<?php
							/* translators: Post revisions heading. %s: The number of available revisions. */
							printf( __( 'Revisions: %s' ), '<b>' . number_format_i18n( $args['args']['revisions_count'] ) . '</b>' );
							?>
							<a class="hide-if-no-js" href="<?php echo esc_url( get_edit_post_link( $args['args']['revision_id'] ) ); ?>"><span aria-hidden="true"><?php _ex( 'Browse', 'revisions' ); ?></span> <span class="screen-reader-text"><?php _e( 'Browse revisions' ); ?></span></a>
						</div>
						<?php
					endif;

					if ( $can_publish ) : // Contributors don't get to choose the date of publish
						?>
						<div class="misc-pub-section curtime misc-pub-curtime">
						<span id="timestamp">
						<?php printf( $stamp, '<b>' . $date . '</b>' ); ?>
	</span>
						<a href="#edit_timestamp" class="edit-timestamp hide-if-no-js" role="button">
							<span aria-hidden="true"><?php _e( 'Edit' ); ?></span>
							<span class="screen-reader-text"><?php _e( 'Edit date and time' ); ?></span>
						</a>
						<fieldset id="timestampdiv" class="hide-if-js">
							<legend class="screen-reader-text"><?php _e( 'Date and time' ); ?></legend>
							<?php touch_time( ( $action === 'edit' ), 1 ); ?>
						</fieldset>
						</div><?php // /misc-pub-section ?>
					<?php endif; ?>

					<?php if ( 'draft' === $post->post_status && get_post_meta( $post->ID, '_customize_changeset_uuid', true ) ) : ?>
						<div class="notice notice-info notice-alt inline">
							<p>
								<?php
								echo sprintf(
								/* translators: %s: URL to the Customizer. */
									__( 'This draft comes from your <a href="%s">unpublished customization changes</a>. You can edit, but there&#8217;s no need to publish now. It will be published automatically with those changes.' ),
									esc_url(
										add_query_arg(
											'changeset_uuid',
											rawurlencode( get_post_meta( $post->ID, '_customize_changeset_uuid', true ) ),
											admin_url( 'customize.php' )
										)
									)
								);
								?>
							</p>
						</div>
					<?php endif; ?>

					<?php
					/**
					 * Fires after the post time/date setting in the Publish meta box.
					 *
					 * @since 2.9.0
					 * @since 4.4.0 Added the `$post` parameter.
					 *
					 * @param WP_Post $post WP_Post object for the current post.
					 */
					do_action( 'post_submitbox_misc_actions', $post );
					?>
				</div>
				<div class="clear"></div>
			</div>
			<?php do_action( 'modula_albums_cpt_publish_actions' ); ?>
			<div class="modula-shortcuts">
				<?php esc_html_e( 'Want a faster and easier way to save albums? Use our Keyboard shortcut:', 'modula-albums' ); ?> <strong>CTRL/CMD + S</strong>
			</div>
			<div id="major-publishing-actions">
				<?php
				/**
				 * Fires at the beginning of the publishing actions section of the Publish meta box.
				 *
				 * @since 2.7.0
				 * @since 4.9.0 Added the `$post` parameter.
				 *
				 * @param WP_Post|null $post WP_Post object for the current post on Edit Post screen,
				 *                           null on Edit Link screen.
				 */
				do_action( 'post_submitbox_start', $post );
				?>
				<div id="delete-action">
					<?php
					if ( current_user_can( 'delete_post', $post->ID ) ) {
						if ( ! EMPTY_TRASH_DAYS ) {
							$delete_text = __( 'Delete Permanently' );
						} else {
							$delete_text = __( 'Move to Trash' );
						}
						?>
						<a class="submitdelete deletion" href="<?php echo get_delete_post_link( $post->ID ); ?>"><?php echo $delete_text; ?></a>
						<?php
					}
					?>
				</div>
				<div id="publishing-action">
					<span class="spinner"></span>

					<?php
					if ( ! in_array( $post->post_status, array( 'publish', 'future', 'private' ) ) || 0 == $post->ID ) {
						if ( $can_publish ) :
							if ( ! empty( $post->post_date_gmt ) && time() < strtotime( $post->post_date_gmt . ' +0000' ) ) :
								?>
								<input name="original_publish" type="hidden" id="original_publish" value="<?php echo esc_attr_x( 'Schedule', 'post action/button label' ); ?>" />
								<?php submit_button( _x( 'Schedule', 'post action/button label' ), 'primary large', 'publish', false ); ?>
							<?php	else : ?>
								<input name="original_publish" type="hidden" id="original_publish" value="
								<?php
								esc_attr_e( 'Update ' ) . 'modula-album';
								?>
 " />
								<?php submit_button( __( 'Save Album', 'modula-albums' ), 'primary large', 'publish', false ); ?>
								<?php
							endif;
						else :
							?>
							<input name="original_publish" type="hidden" id="original_publish" value="<?php esc_attr_e( 'Submit for Review' ); ?>" />
							<?php submit_button( __( 'Submit for Review' ), 'primary large', 'publish', false ); ?>
							<?php
						endif;
					} else {
						?>

						<input name="save" type="submit" class="button button-primary button-large" id="publish" value="<?php esc_attr_e( 'Update Album', 'modula-albums' ) . 'modula-album'; ?>" />
						<?php
					}
					?>
				</div>
				<div class="clear"></div>
			</div>
		</div> 
		<?php
	}

	/**
	 * PRO license check function
	 *
	 * @since 1.0.0
	 */
	public function modula_pro_license_check() {

		if ( get_option( 'modula_albums_pro_check' ) ) {
			return;
		}
		// If no license is available let the user know
		$current_screen = get_current_screen();
		echo '<div id="modula_albums_pro_check" class="notice notice-warning is-dismissible">
             <p>' . esc_html__( 'In order to use Modula Albums addon with your gallery you\'ll need to activate your license ', 'modula-albums' ) . '<a href="' . admin_url( 'edit.php?post_type=modula-gallery&page=modula&modula-tab=licenses' ) . '">' . esc_html__( 'here', 'modula-albums' ) . '</a></p>'
			 . ' </div>';

	}

		/**
		 * Dismiss notice function
		 *
		 * @return void
		 * @since 1.2.0
		 */
	public function dismiss_albums_notice() {

		check_ajax_referer( 'dismiss_albums_nonce', 'nonce' );
		update_option( 'modula_albums_pro_check', '1' );
		wp_send_json_success();
		wp_die();
	}

	/**
	 * Enqueue PRO check script
	 *
	 * @since 1.0.0
	 */
	public function nopro_scripts() {

		wp_enqueue_script( 'modula-albums-pro-check', MODULA_ALBUMS_URL . '/assets/js/admin/modula-albums-pro-check.js', array(), MODULA_ALBUMS_VERSION, true );
		wp_add_inline_script( 'modula-albums-pro-check', 'modulaAlbumsNonce = "' . wp_create_nonce( 'dismiss_albums_nonce' ) . '";', 'before' );
	}

	/**
	 * Remove Albums menu item from Modula if license expired or non-existent.
	 * 
	 * @since 1.0.11
	 */
	public function remove_albums_menu_item( $args ) {

		$args['show_in_menu'] = false;

		return $args;
	}

}

new Modula_Albums_CPT();
