<?php
// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
class Modula_Standalone {
	/**
	 * Holds the class object.
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public static $instance;
	/**
	 * Primary class constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		// Set admin requirements on admin_init so that we won't trigger get_packages from WPChill_Upsells on frontend
		add_action( 'admin_init', array( $this, 'set_admin_requirements' ), 25 );
		add_action( 'modula_cpt_args', array( $this, 'alterate_cpt_args' ) );
		add_action( 'modula_album_cpt_args', array( $this, 'alterate_album_args' ) );
		add_action( 'wp_head', array( $this, 'standalone_insert_shortcode' ) );
		add_filter( 'modula_uninstall_db_options', array( $this, 'standalone_uninstall' ), 17, 1 );

	}

	/**
	 * Set admin requirements and check for license.
	 *
	 * @return void
	 * @since 1.0.7
	 */
	public function set_admin_requirements() {

		if ( class_exists( 'WPChill_Upsells' ) ) {

			$args           = apply_filters(
				'modula_upsells_args',
				array(
					'shop_url' => 'https://wp-modula.com',
					'slug'     => 'modula',
				)
			);
			$wpchill_upsell = WPChill_Upsells::get_instance( $args );

			if ( $wpchill_upsell && ! $wpchill_upsell->is_upgradable_addon( 'modula-albums' ) ) {

				// Add Setings Tab
				add_filter( 'modula_admin_page_tabs', array( $this, 'add_settings_tab' ) );
				add_action( 'modula_admin_tab_standalone', array( $this, 'render_settings_tab' ) );
				$this->register_settings();
				// Add script for installing addons
				add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );

			}
		}
	}

	/**
	 * @param $tabs
	 * @return mixed
	 *
	 * Add Modula Albums Standalone settings tab
	 */
	public function add_settings_tab( $tabs ) {

		if ( ! class_exists( 'Modula_Pro' ) ) {
			return $tabs;
		}

		$tabs['standalone'] = array(
			'label'    => esc_html__( 'Standalone', 'modula-albums' ),
			'priority' => 20,
		);

		return $tabs;
	}

	/**
	 * Render settings tab
	 */
	public function render_settings_tab() {

		include 'tabs/standalone.php';
	}

	/**
	 * Register standalone settings
	 */
	public function register_settings() {
		// creates our settings in the options table
		register_setting( 'modula_standalone', 'modula_standalone', array( $this, 'sanitize_option' ) );
	}

	public function sanitize_option( $current ) {

		if ( ! isset( $current['gallery']['enable_rewrite'] ) || 'disabled' == $current['gallery']['enable_rewrite'] ) {
			$current['gallery']['enable_rewrite'] = 'disabled';
		} else {
			$current['gallery']['enable_rewrite'] = 'enabled';
		}

		if ( ! isset( $current['album']['enable_rewrite'] ) || 'disabled' == $current['album']['enable_rewrite'] ) {
			$current['album']['enable_rewrite'] = 'disabled';
		} else {
			$current['album']['enable_rewrite'] = 'enabled';
		}

		if ( ! isset( $current['gallery']['slug'] ) || '' == $current['gallery']['slug'] ) {
			$current['gallery']['slug'] = 'modula-gallery';
		} else {
			$current['gallery']['slug'] = sanitize_text_field( $current['gallery']['slug'] );
		}

		if ( ! isset( $current['album']['slug'] ) || '' == $current['album']['slug'] ) {
			$current['album']['slug'] = 'modula-album';
		} else {
			$current['album']['slug'] = sanitize_text_field( $current['album']['slug'] );
		}

		return $current;
	}

	/**
	 * Returns the singleton instance of the class.
	 *
	 * @since 1.0.0
	 *
	 * @return object The Modula_Standalone object.
	 */
	public static function get_instance() {
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Modula_Standalone ) ) {
			self::$instance = new Modula_Standalone();
		}
		return self::$instance;
	}

	/**
	 * @param $cpt_args
	 * @return mixed
	 */
	public function alterate_cpt_args( $cpt_args ) {

		$defaults = apply_filters(
			'modula_standalone_default_settings',
			array(
				'gallery' => array(
					'enable_rewrite' => 'disabled',
					'slug'           => 'modula-gallery',
				),
				'album'   => array(
					'enable_rewrite' => 'disabled',
					'slug'           => 'modula-album',
				),
			)
		);

		$settings = get_option( 'modula_standalone', array() );
		$settings = wp_parse_args( $settings, $defaults );

		if ( 'enabled' == $settings['gallery']['enable_rewrite'] ) {
			$cpt_args['rewrite'] = array( 'slug' => sanitize_text_field( $settings['gallery']['slug'] ) );
			$cpt_args['public']  = true;
		}

		return $cpt_args;
	}

	/**
	 * @param $cpt_args
	 * @return mixed
	 */
	public function alterate_album_args( $cpt_args ) {

		$defaults = apply_filters(
			'modula_standalone_default_settings',
			array(
				'gallery' => array(
					'enable_rewrite' => 'disabled',
					'slug'           => 'modula-gallery',
				),
				'album'   => array(
					'enable_rewrite' => 'disabled',
					'slug'           => 'modula-album',
				),
			)
		);
		$settings = get_option( 'modula_standalone', array() );
		$settings = wp_parse_args( $settings, $defaults );

		if ( 'enabled' == $settings['album']['enable_rewrite'] ) {
			$cpt_args['rewrite'] = array( 'slug' => sanitize_text_field( $settings['album']['slug'] ) );
			$cpt_args['public']  = true;
		}

		return $cpt_args;
	}


	public function standalone_insert_shortcode() {

		// Check if we are on a single Post.
		if ( ! is_singular() ) {
			return;
		}

		// If not the proper post type (Modula), return early.
		$post_type = get_query_var( 'post_type' );

		if ( 'modula-gallery' == $post_type ) {
			add_filter( 'the_content', array( $this, 'insert_gallery_shortcode' ) );
		}

		if ( 'modula-album' == $post_type ) {
			add_filter( 'the_content', array( $this, 'insert_album_shortcode' ) );
		}

	}

	public function insert_gallery_shortcode( $content ) {

		// Display the gallery based on the query var available.
		$id = get_query_var( 'p' );
		if ( empty( $id ) ) {
			global $post;
			$id = $post->ID;
		}

		$shortcode = '[modula id="' . $id . '"]';

		return $shortcode . $content;

	}

	public function insert_album_shortcode( $content ) {

		// Display the gallery based on the query var available.
		$id = get_query_var( 'p' );
		if ( empty( $id ) ) {
			global $post;
			$id = $post->ID;
		}

		$shortcode = '[modula-album id="' . $id . '"]';

		return $shortcode . $content;

	}

	public function admin_scripts( $hook ) {

		if ( 'modula-gallery_page_modula' == $hook ) {
			wp_enqueue_script( 'modula-standalone', MODULA_ALBUMS_URL . 'assets/js/admin/wp-modula-standalone.js', array( 'jquery' ), '2.0.0', true );

			$defaults = apply_filters(
				'modula_standalone_default_settings',
				array(
					'gallery' => array(
						'enable_rewrite' => 'disabled',
						'slug'           => 'modula-gallery',
					),
					'album'   => array(
						'enable_rewrite' => 'disabled',
						'slug'           => 'modula-album',
					),
				)
			);
			$settings = get_option( 'modula_standalone', array() );
			$settings = wp_parse_args( $settings, $defaults );
			wp_localize_script( 'modula-standalone', 'modulaStandalone', $settings );
		}

	}

	/**
	 * Uninstall filter
	 *
	 * @param $db_options
	 *
	 * @return mixed
	 *
	 * @since 1.0.3
	 */
	public function standalone_uninstall( $db_options ) {

		$db_options[] = 'modula_standalone';

		return $db_options;
	}

}
