<?php

/**
 *
 */
class Modula_Deeplink {

	/**
	 * Holds the class object.
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public static $instance;

	/**
	 * The name of the plugin.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $plugin_name = 'Modula Deeplink';

	/**
	 * Unique plugin slug identifier.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $plugin_slug = 'modula-deeplink';

	/**
	 * Primary class constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		// Set admin requirements on admin_init so that we won't trigger get_packages from WPChill_Upsells on frontend
		add_action( 'admin_init', array( $this, 'set_admin_requirements' ), 25 );
		// Load the plugin textdomain.
		add_action( 'init', array( $this, 'set_locale' ) );

		add_filter( 'modula_gallery_settings', array( $this, 'modula_deeplink_js_config' ), 10, 2 );
		add_filter( 'modula_link_gallery_settings', array( $this, 'modula_link_deeplink_js_config' ), 10, 2 );

		// Field to register image id
		add_filter( 'modula_shortcode_item_data', array( $this, 'modula_pro_attachment_id' ), 50, 2 );

		// Sanitize custom url
		add_filter( 'modula_settings_field_sanitization', array( $this, 'sanitize_custom_link' ), 40, 4 );

		// Apply deeplink/hash options to fancybox
		add_filter( 'modula_fancybox_options', array( $this, 'modula_deeplink_fancybox_options' ), 5, 2 );

		add_action( 'modula_extra_scripts', array( $this, 'deeplink_extra_scripts' ) );

		add_action( 'modula_link_extra_scripts', array( $this, 'modula_link_deeplink_scripts' ), 30, 1 );

		// Enqueue admin required scripts
		add_action( 'modula_scripts_after_wp_modula', array( $this, 'modula_deeplink_admin_scripts' ), 40 );
		add_action( 'modula_defaults_scripts_after_wp_modula', array( $this, 'modula_deeplink_admin_scripts' ), 40 );

		add_filter( 'modula_migrate_gallery_data', array( $this, 'modula_deeplink_migrate' ), 25, 3 );

		// Add defaults
		add_filter( 'modula_lite_default_settings', array( $this, 'set_defaults' ) );

		// Load the plugin.
		$this->init();

		register_activation_hook( MODULA_DEEPLINK_FILE, array( $this, 'plugin_activation' ) );

	}

	/**
	 * Set admin requirements and check for license.
	 *
	 * @return void
	 * @since 1.0.7
	 */
	public function set_admin_requirements() {

		if ( class_exists( 'WPChill_Upsells' ) ) {

			$args           = apply_filters(
				'modula_upsells_args',
				array(
					'shop_url' => 'https://wp-modula.com',
					'slug'     => 'modula',
				)
			);
			$wpchill_upsell = WPChill_Upsells::get_instance( $args );

			if ( class_exists( 'WPChill_Upsells' ) ) {
				if ( $wpchill_upsell && ! $wpchill_upsell->is_upgradable_addon( 'modula-deeplink' ) ) {

					// Filter Modula Deeplink Tab
					add_filter( 'modula_gallery_tabs', array( $this, 'modula_deeplink_tabs' ), 99 );

					// Filter Modula Deeplink Fields
					add_filter( 'modula_gallery_fields', array( $this, 'modula_deeplink_fields' ) );
				}
			}
		}
	}

	/**
	 * Prevent plugin activation if Modula Pro is not installed and activated
	 *
	 * @since 1.0.1
	 */
	public function plugin_activation() {
		if ( ! class_exists( 'Modula_PRO' ) ) {
			deactivate_plugins( plugin_basename( MODULA_DEEPLINK_FILE ) );
			wp_die( __( 'Please install and activate Modula Pro before using Modula Deeplink add-on.', 'modula-deeplink' ) );

		}

	}

	/**
	 * Loads the plugin textdomain for translation.
	 *
	 * @since 1.0.0
	 */
	public function set_locale() {
		load_plugin_textdomain( $this->plugin_slug, false, dirname( plugin_basename( MODULA_DEEPLINK_FILE ) ) . '/languages' );
	}

	/**
	 * Loads the plugin into WordPress.
	 *
	 * @since 1.0.0
	 */
	public function init() {

		// Load admin only components.
		if ( is_admin() ) {
			add_action( 'modula_pro_updater', array( $this, 'addon_updater' ), 15, 2 );
		}
	}

	public function addon_updater( $license_key, $store_url ) {

		if ( class_exists( 'Modula_Pro_Base_Updater' ) ) {
			$modula_addon_updater = new Modula_Pro_Base_Updater(
				$store_url,
				MODULA_DEEPLINK_FILE,
				array(
					'version' => MODULA_DEEPLINK_VERSION,        // current version number
					'license' => $license_key,               // license key (used get_option above to retrieve from DB)
					'item_id' => 294024,                      // ID of the product
					'author'  => 'MachoThemes',            // author of this plugin
					'beta'    => false,
				)
			);
		}
	}


	/**
	 * Returns the singleton instance of the class.
	 *
	 * @since 1.0.0
	 *
	 * @return object The Modula_Deeplink object.
	 */
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Modula_Deeplink ) ) {
			self::$instance = new Modula_Deeplink();
		}

		return self::$instance;

	}



	/**
	 * @param $tabs
	 *
	 * @return mixed
	 *
	 * Add Modula Deeplink Tab
	 */
	public function modula_deeplink_tabs( $tabs ) {

		if ( ! isset( $tabs['misc'] ) ) {
			$tabs['misc'] = array(
				'label'    => esc_html__( 'Miscellaneous', 'modula-deeplink' ),
				'icon'     => 'dashicons dashicons-image-filter',
				'priority' => 100,
			);
		}

		unset( $tabs['misc']['badge'] );

		return $tabs;
	}


	/**
	 * @param $fields
	 *
	 * @return mixed
	 *
	 * Add Modula Deeplink field
	 */
	public function modula_deeplink_fields( $fields ) {

		if ( ! isset( $fields['misc'] ) || ! is_array( $fields['misc'] ) ) {
			$fields['misc'] = array();
		}

		$fields['misc']['modula_deeplink'] = array(
			'name'        => __( 'Enable Deeplinking?', 'modula-deeplink' ),
			'type'        => 'toggle',
			'default'     => 0,
			'description' => __( 'Enable creation of unique and indexable URLs for each Modula gallery items that can benefit SEO.', 'modula-deeplink' ),
			'priority'    => 15,
			'children'    => array( 'customLinkName' ), 
		);
		$fields['misc']['customLinkName']  = array(
			'name'        => esc_html__( 'Custom Link Name', 'modula-deeplink' ),
			'type'        => 'text',
			'default'     => 'modulagallery',
			'description' => esc_html__( 'Enter a custom prefix for your gallery URL that will be used to open the lightbox.', 'modula-deeplink' ),
			'priority'    => 20,
			'is_child'    => true,
			'afterrow'    => esc_html__( 'If using filters, in order for deeplink to work properly please either set `All` as default filter or enable `Reload Page On Filter Click ` setting.', 'modula-deeplink' ),
		);

		return $fields;
	}

	/**
	 * Add deeplink settings to Modula's defaults
	 *
	 * @param $defaults
	 *
	 * @return mixed
	 */
	public function set_defaults( $defaults ) {

		$defaults['modula_deeplink'] = 0;
		$defaults['customLinkName']  = 'modulagallery';

		return $defaults;
	}


	public function modula_deeplink_js_config( $js_config, $settings ) {

		if ( isset( $settings['modula_deeplink'] ) && '1' == $settings['modula_deeplink'] ) {
			$js_config['modula_deeplink'] = $settings['modula_deeplink'];
			if ( isset( $settings['gallery_id'] ) ) {
				$js_config['galleryID'] = str_replace( 'jtg-', '', $settings['gallery_id'] );
			}
			if ( isset( $settings['customLinkName'] ) ) {
				$js_config['customLinkName'] = $settings['customLinkName'];
			}
		} else {
			$js_config['modula_deeplink'] = 0;
		}

		return $js_config;
	}

	/**
	 * Modula Link Shortcode js config fitlering
	 *
	 * @param $js_config
	 * @param $settings
	 *
	 * @since 1.0.2
	 *
	 * @return mixed
	 */
	public function modula_link_deeplink_js_config( $js_config, $settings ) {

		if ( isset( $settings['modula_deeplink'] ) && '1' == $settings['modula_deeplink'] ) {

			$js_config['options']['modula_deeplink'] = $settings['modula_deeplink'];
			if ( isset( $settings['gallery_id'] ) ) {

				$js_config['options']['galleryID'] = str_replace( 'jtg-', '', $settings['gallery_id'] );
			}

			if ( isset( $settings['customLinkName'] ) ) {

				$js_config['options']['customLinkName'] = $settings['customLinkName'];
			}
		} else {
			$js_config['options']['modula_deeplink'] = 0;
		}

		return $js_config;
	}

	public function modula_pro_attachment_id( $item_data, $image ) {

		if ( isset( $item_data['link_attributes']['data-fancybox'] ) ) {
			$item_data['link_attributes']['data-fancybox'] = str_replace( 'jtg', '!modulagallery', $item_data['link_attributes']['data-fancybox'] );
		}

		return $item_data;
	}


	public function modula_deeplink_fancybox_options( $fancybox_options, $settings ) {

		if ( isset( $settings['modula_deeplink'] ) && '1' == $settings['modula_deeplink'] ) {

			if ( isset( $settings['gallery_id'] ) ) {
				if ( isset( $settings['customLinkName'] ) && '' != $settings['customLinkName'] ) {
					$fancybox_options['hash'] = $settings['customLinkName'] . str_replace( 'jtg', '', $settings['gallery_id'] );
				} else {
					$fancybox_options['hash'] = '!modulagallery' . str_replace( 'jtg', '', $settings['gallery_id'] );
				}
			}
		}

		return $fancybox_options;
	}


	/**
	 * Enqueue extra scripts required for deeplinking
	 *
	 * @param $settings
	 *
	 * @since 1.0.2
	 */
	public function deeplink_extra_scripts( $settings ) {

		$not_included = array( 'fancybox', 'no-link', 'direct', 'attachment-page' );

		if ( ! in_array( $settings['lightbox'], $not_included ) && isset( $settings['modula_deeplink'] ) && '1' == $settings['modula_deeplink'] ) {
			wp_enqueue_script( 'modula-deeplink', MODULA_DEEPLINK_URL . 'assets/js/modula-deeplink.js', array( 'jquery', 'modula-wfl', 'modula-lightgallery', 'modula-lightboxes' ), MODULA_DEEPLINK_VERSION, true );

			if ( 'lightgallery' == $settings['lightbox'] ) {
				wp_enqueue_script( 'modula-lightbox2-deeplink', MODULA_DEEPLINK_URL . 'assets/js/lg-hash.js', array( 'jquery' ), MODULA_DEEPLINK_VERSION, true );
			}
		} elseif ( 'fancybox' == $settings['lightbox'] ) {
			wp_enqueue_script( 'modula-fancybox-deeplink', MODULA_DEEPLINK_URL . 'assets/js/modula-fancybox-deeplink.js', array( 'jquery', 'modula-wfl' ), MODULA_DEEPLINK_VERSION, true );
		}
	}

	/**
	 * Enqueue admin Modula Deeplink scripts
	 *
	 * @since 1.0.2
	 */
	public function modula_deeplink_admin_scripts() {

		wp_enqueue_script( 'modula-deeplink-conditions', MODULA_DEEPLINK_URL . 'assets/js/wp-modula-deeplink-conditions.js', array( 'modula-conditions' ), '', true );
	}

	/**
	 * Add deeplink scripts for modula link shortcode
	 *
	 * @param $settings
	 */
	public function modula_link_deeplink_scripts( $settings ) {

		wp_enqueue_script( 'modula-fancybox-deeplink', MODULA_DEEPLINK_URL . 'assets/js/modula-fancybox-deeplink.js', array( 'jquery' ), MODULA_DEEPLINK_VERSION, true );
	}

	/**
	 * Function for sanitization of custom linl
	 *
	 * @since 1.0.4
	 *
	 * @param string $sanitized_value
	 * @param string $value
	 * @param string $field_id
	 * @param array  $field
	 *
	 * @return string $sanitized_value
	 */
	public function sanitize_custom_link( $sanitized_value, $value, $field_id, $field ) {

		if ( 'customLinkName' == $field_id ) {
			$sanitized_value = sanitize_title( $value );
		}
		return $sanitized_value;
	}

	/**
	 * Add deeplink options to Modula migrator
	 *
	 * @param $modula_settings
	 * @param $guest_settings
	 * @param $source
	 *
	 * @return mixed
	 *
	 * @since 1.0.4
	 */
	public function modula_deeplink_migrate( $modula_settings, $guest_settings, $source ) {

		if ( $source ) {
			switch ( $source ) {
				case 'envira':
					if ( isset( $guest_settings['config']['deeplinking'] ) && 1 == $guest_settings['config']['deeplinking'] ) {
						$modula_settings['modula_deeplink'] = 1;
					}
					break;
			}
		}

		return $modula_settings;
	}

}
