<?php

/**
 * Holds the admin object regarding settings
 */
class Modula_Download_Album_Settings {

	/**
	 * Holds the admin class object for albums
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public static $instance;

	/**
	 * Constructor for Modula_Download_Album_Settings class
	 *
	 * @return void
	 * @since 1.0.0
	 *
	 */
	public function __construct() {
		// Set admin requirements on admin_init so that we won't trigger get_packages from WPChill_Upsells on frontend
		add_action( 'admin_init', array( $this, 'set_admin_requirements' ), 50 );

		add_action( 'modula_album_scripts_after_wp_modula', array( $this, 'modula_download_album_admin_scripts' ) );
		add_action( 'modula_defaults_album_scripts_after_wp_modula', array( $this, 'modula_download_album_admin_scripts' ) );
	}

	/**
	 * Set admin requirements and check for license.
	 *
	 * @return void
	 * @since 1.0.4
	 */
	public function set_admin_requirements() {

		if ( class_exists( 'WPChill_Upsells' ) ) {

			$args           = apply_filters(
				'modula_upsells_args',
				array(
					'shop_url' => 'https://wp-modula.com',
					'slug'     => 'modula',
				)
			);
			$wpchill_upsell = WPChill_Upsells::get_instance( $args );

			if ( $wpchill_upsell && ! $wpchill_upsell->is_upgradable_addon( 'modula-download' ) ) {
				add_filter( 'modula_album_fields', array( $this, 'modula_download_album_field' ) );
				add_filter( 'modula_album_tabs', array( $this, 'modula_download_album_tabs' ) );
			}
		}
	}


	/**
	 * Returns the singleton instance of the class
	 *
	 * @return object Modula Download Album Settings object
	 * @since 1.0.0
	 *
	 */
	public static function get_instance() {
		if ( !isset( self::$instance ) && !(self::$instance instanceof Modula_Download_Album_Settings) ) {
			self::$instance = new Modula_Download_Album_Settings();
		}

		return self::$instance;
	}

	/**
	 * Enqueue admin conditions
	 *
	 * @since 1.0.0
	 */
	public function modula_download_album_admin_scripts() {
		wp_enqueue_script( 'modula-download-album-conditions', MODULA_DOWNLOAD_URL . 'assets/js/modula-download-album-conditions.js', array( 'jquery' ), MODULA_DOWNLOAD_VERSION, true );
		wp_enqueue_style( 'modula-download-admin-css', MODULA_DOWNLOAD_URL . 'assets/css/modula-download-admin.css', MODULA_DOWNLOAD_VERSION);
	}


	/**
	 * Download Tab for Albums
	 *
	 * @param array $tabs
	 *
	 * @return array $tabs
	 * @since 1.0.0
	 *
	 */
	public function modula_download_album_tabs( $tabs ) {
		$tabs['download'] = array(
			'label'       => esc_html__( 'Download', 'modula-download' ),
			'title'       => esc_html__( 'Download', 'modula-download' ),
			'description' => esc_html__( 'Downloading your entire album or just one gallery, is now one click away', 'modula-download' ),
			'icon'        => 'dashicons dashicons-download',
			'priority'    => 75,
		);
		return $tabs;
	}

	/**
	 * Download Tab for Albums
	 *
	 * @param array $fields
	 *
	 * @return array $fields
	 * @since 1.0.0
	 *
	 */
	public function modula_download_album_field( $fields ) {
		$fields['download'] = array(
			'enable_download_albums'                  => array(
				"name"        => esc_html__( 'Enable Download', 'modula-download' ),
				"type"        => "toggle",
				"default"     => 0,
				"description" => esc_html__( 'Enable this option to display a download button on gallery hover or add a download all button.', 'modula-download' ),
				"priority"    => 1,
				"children"    => array( 'download_image_sizes_albums','download_gallery_button_albums', 'download_all_lightbox_button_albums', 'download_all_gallery_button_albums', 'custom_zip_name_albums' ),
			),
			'download_image_sizes_albums'             => array(
				"name"        => esc_html__( 'Download Image size', 'modula-download' ),
				"type"        => "select",
				"description" => esc_html__( 'Select the image size to download from a selection of options.', 'modula-download' ),
				'values'      => array(
					'thumbnail' => esc_html__( 'Thumbnail', 'modula-download' ),
					'medium'    => esc_html__( 'Medium', 'modula-download' ),
					'large'     => esc_html__( 'Large', 'modula-download' ),
					'full'      => esc_html__( 'Full', 'modula-download' ),
				),
				'default'     => 'thumbnail',
				'priority'    => 3,
				'is_child'    => true,
			),
			'download_gallery_button_albums'          => array(
				"name"        => esc_html__( 'Download Button', 'modula-download' ),
				"type"        => "toggle",
				"default"     => 1,
				"description" => esc_html__( 'Enable this option to display the download button hovering over a gallery. On by default', 'modula-download' ),
				"priority"    => 5,
				'is_child'    => true,
				"children"    => array( 'download_gallery_button_color_albums', 'download_gallery_button_position_albums' ),
				'parent'	  => "enable_download_albums",
			),
			'download_gallery_button_color_albums'    => array(
				"name"        => esc_html__( 'Download Button Color', 'modula-download' ),
				"type"        => "color",
				"default"     => '#ffffff',
				"description" => esc_html__( 'Choose the download button color. Default #ffffff', 'modula-download' ),
				"priority"    => 7,
				'is_child'    => 'two',
			),
			'download_gallery_button_position_albums' => array(
				"name"     => esc_html__( 'Download Gallery Button Position', 'modula-download' ),
				"type"     => "select",
				"values"   => array(
					"top_left"     => esc_html__( 'Top Left', 'modula-download' ),
					"top_right"    => esc_html__( 'Top Right', 'modula-download' ),
					"bottom_left"  => esc_html__( 'Bottom Left', 'modula-download' ),
					"bottom_right" => esc_html__( 'Bottom Right', 'modula-download' ),
				),
				"default"  => "top_left",
				"priority" => 15,
				"is_child" => 'two',
			),
			'download_all_lightbox_button_albums'     => array(
				"name"        => esc_html__( 'Lightbox Download All Button', 'modula-download' ),
				"type"        => "toggle",
				"default"     => 1,
				"description" => esc_html__( 'Enable this option to display the download all button inside the lightbox. On by default', 'modula-download' ),
				"priority"    => 20,
				'is_child'    => true,
			),
			'download_all_gallery_button_albums'      => array(
				"name"        => esc_html__( 'Download All Button', 'modula-download' ),
				"type"        => "toggle",
				"default"     => 1,
				"description" => esc_html__( 'Enable this option to display the download all label above or bellow the gallery. On by default', 'modula-download' ),
				"priority"    => 25,
				'is_child'    => true,
				"children"    => array( "download_all_color_albums", 'download_all_size_albums', 'download_all_hposition_albums', 'download_all_position_albums', 'download_all_label_albums','download_all_gallery_button_icon_albums','download_all_gallery_icon_color_albums'  ),
				'parent'	  => "enable_download_albums",
			),
			"download_all_color_albums"     => array(
				"name"        => esc_html__( 'Button Color', 'modula-best-grid-gallery' ),
				"type"        => "color",
				"description" => esc_html__( 'Set the color of download all button.', 'modula-download' ),
				"default"     => "",
				'is_child'    => 'two',
				'priority'    => 26,
			),
			'download_all_size_albums'              => array(
				"name"        => esc_html__( 'Button Size', 'modula-download' ),
				"type"        => "select",
				"values"      => array(
					"small"   => esc_html__( 'Small', 'modula-download' ),
					"medium"  => esc_html__( 'Medium', 'modula-download' ),
					"large"   => esc_html__( 'Large', 'modula-download' ),
					"x-large" => esc_html__( 'X-Large', 'modula-download' ),
				),
				"default"     => "medium",
				"description" => esc_html__( 'Select the size of your "Download All" button.', 'modula-download' ),
				"priority"    => 27,
				"is_child"    => 'two',
			),
			'download_all_hposition_albums'         => array(
				"name"        => esc_html__( 'Button Horizontal Position', 'modula-download' ),
				"type"        => "select",
				"values"      => array(
					"left"    => esc_html__( 'Left', 'modula-download' ),
					"right"   => esc_html__( 'Right', 'modula-download' ),
					"center"  => esc_html__( 'Center', 'modula-download' ),
				),
				"default"     => "left",
				"description" => esc_html__( 'Select the horizontal alignment of the "Download All Button"', 'modula-download' ),
				"priority"    => 28,
				"is_child"    => 'two',
			),
			'download_all_position_albums'            => array(
				"name"        => esc_html__( 'Button Position', 'modula-download' ),
				"type"        => "select",
				"values"      => array(
					"above_gallery"       => esc_html__( 'Above Album', 'modula-download' ),
					"below_gallery"       => esc_html__( 'Below Album', 'modula-download' ),
					"above_below_gallery" => esc_html__( 'Above & Below Album', 'modula-download' ),
				),
				"default"     => "above_gallery",
				"description" => esc_html__( 'Select where you want to display the "Download All" text.', 'modula-download' ),
				"priority"    => 35,
				"is_child"    => 'two',
			),
			'download_all_label_albums'               => array(
				"name"        => esc_html__( 'Download All Button Label', 'modula-download' ),
				"type"        => "text",
				"default"     => esc_html__( 'Download All Images', 'modula-download' ),
				"description" => esc_html__( 'Enter a custom label for the Download All button', 'modula-download' ),
				"priority"    => 40,
				'is_child'    => 'two',
			),
			'download_all_gallery_button_icon_albums' => array(
				"name"        => esc_html__( 'Download all Icon', 'modula-download' ),
				"type"        => "toggle",
				"default"     => 1,
				"description" => esc_html__( 'Enable this option to display an icon near the "Download All" text. Applied to both the shortcode and the label. On by default', 'modula-download' ),
				"priority"    => 46,
				'is_child'    => 'two',
			),
			'download_all_gallery_icon_color_albums' => array(
				"name"        => esc_html__( 'Download all Icon Color', 'modula-download' ),
				"type"        => "color",
				"default"     => '#000',
				"description" => esc_html__( 'Choose the download all icon color. If left empty, the default color will be black. Applied to both the shortcode and the label.', 'modula-download' ),
				"priority"    => 47,
				'is_child'    => 'two',
			),
			'custom_zip_name_albums'                  => array(
				"name"        => esc_html__( 'Custom Zip Name', 'modula-download' ),
				"type"        => "placeholder",
				"values"      => Modula_Pro_Helper::modula_albums_placeholders(),
				"default"     => '%%album_title%%',
				"description" => esc_html__( 'Enter a custom zip name. If left empty the zip name will be under the format of : modula-album-id. Applied to both the shortcode and the label.', 'modula-download' ),
				"priority"    => 48,
				'afterrow'    => esc_html__( 'Each placeholder will automatically have space between them or you can manually insert a separator.', 'modula-download' ),
				'is_child'    => true,
			),
		);
		if ( !extension_loaded('zip') ) {
			$fields['download']['helper-message'] = array(
				"name"     => ' ',
				"type"     => "content",
				"content"  => esc_html__( 'In order for the "Download All" functionality to work you need to enable the ZIP PHP extension. Ask your hosting provider to enable it.', 'modula-download' ),
				'priority' => 0,
			);
		}
		
		return $fields;
	}

}

$modula_download_albums_settings = Modula_Download_Album_Settings::get_instance();