<?php

/**
 * Main class for Modula Download
 */

class Modula_Download {
	/**
	 * Holds the class object
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public static $instance;

	/**
	 * The name of the plugin
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $plugin_name = 'Modula Download';

	/**
	 * Unique plugin identifier
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $plugin_slug = 'modula-download';

	/**
	 * Primary class construct
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		// Set admin requirements on admin_init so that we won't trigger get_packages from WPChill_Upsells on frontend
		add_action( 'admin_init', array( $this, 'set_admin_requirements' ), 25 );

		if ( ! is_admin() ) {
			add_action( 'init', array( $this, 'modula_force_download' ) );
		}

		$this->require_helpers();

		add_action( 'modula_extra_scripts', array( $this, 'modula_download_scripts' ) );
		add_filter( 'modula_lite_default_settings', array( $this, 'modula_download_defaults' ) );
		add_filter( 'modula_shortcode_css', array( $this, 'modula_download_gallery_css' ), 10, 3 );
		add_filter( 'modula_shortcode_item_data', array( $this, 'modula_download_item_data' ), 35, 3 );
		add_action( 'modula_extra_socials', array( $this, 'modula_download_gallery_button' ), 10 );
		add_action( 'modula_item_after_image', array( $this, 'modula_download_no_socials' ), 10 );

		if( extension_loaded('zip') ){
			add_action( 'modula_shortcode_before_items', array( $this, 'modula_download_all_gallery_button_above' ), 10, 3 );
			add_action( 'modula_shortcode_after_items', array( $this, 'modula_download_all_gallery_button_below' ), 20, 3 );
			add_action( 'modula_album_shortcode_before_items', array( $this, 'modula_download_all_album_button_above' ) );
			add_action( 'modula_album_shortcode_after_items', array( $this, 'modula_download_all_album_button_below' ) );
			add_filter( 'modula_fancybox_options', array( $this, 'modula_download_fancybox_button' ), 15, 2 );
			add_filter( 'modula_album_template_data', array( $this, 'modula_download_album_fancybox_button' ) );
			add_action( 'modula_album_item_before_link', array( $this, 'modula_download_album_button' ) );
		}


		add_filter( 'modula_link_extra_scripts', array( $this, 'modula_download_link_scripts' ) );
		add_action( 'init', array( $this, 'modula_download_add_shortcode' ) );
		add_action( 'modula_album_extra_scripts', array( $this, 'modula_download_albums_scripts' ) );
		add_filter( 'modula_album_default_settings', array( $this, 'modula_download_albums_defaults' ) );
		add_filter( 'modula_album_shortcode_css', array( $this, 'modula_download_album_css' ), 10, 3 );
		add_filter( 'modula_album_shortcode_item_data', array( $this, 'modula_download_albums_item_data' ), 10, 5 );
		add_filter( 'modula_gallery_placeholders', array( $this, 'modula_download_placeholders' ), 15, 1 );
		add_filter( 'modula_albums_placeholders', array( $this, 'modula_albums_download_placeholders' ), 15, 1 );
		add_filter( 'modula_placeholder_values', array( $this, 'modula_download_placeholder_values' ), 15, 2 );
		add_filter( 'modula_albums_placeholder_values', array( $this, 'modula_albums_download_placeholder_values' ), 15, 2 );
		add_filter( 'modula_slider_item_after_image', array( $this, 'slider_download_button' ), 15, 1 );
		// filter for Modula migrator gallery settings
		add_filter( 'modula_migrate_gallery_data', array( $this, 'modula_download_migrate_data' ), 25, 3 );

		add_action( 'init', array( $this, 'set_locale' ) );

		register_activation_hook( MODULA_DOWNLOAD_FILE, array( $this, 'plugin_activation' ) );

		if ( is_admin() ) {
			add_action( 'modula_pro_updater', array( $this, 'addon_updater' ), 15, 2 );
		}
	}

	/**
	 * Set admin requirements and check for license.
	 *
	 * @return void
	 * @since 1.0.4
	 */
	public function set_admin_requirements() {

		if ( class_exists( 'WPChill_Upsells' ) ) {

			$args           = apply_filters(
				'modula_upsells_args',
				array(
					'shop_url' => 'https://wp-modula.com',
					'slug'     => 'modula',
				)
			);
			$wpchill_upsell = WPChill_Upsells::get_instance( $args );

			if ( $wpchill_upsell && ! $wpchill_upsell->is_upgradable_addon( 'modula-download' ) ) {
				$this->require_admin();
			}
		}
	}

	/**
	 * Prevent plugin activation if Modula Pro is not installed
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function plugin_activation() {
		if ( ! class_exists( 'Modula_PRO' ) ) {
			deactivate_plugins( plugin_basename( MODULA_DOWNLOAD_FILE ) );
			wp_die( __( 'Please install and activate Modula Pro before using Modula Download add-on.', 'modula-download' ) );
		}
	}

	public function addon_updater( $license_key, $store_url ) {

		if ( class_exists( 'Modula_Pro_Base_Updater' ) ) {
			$modula_addon_updater = new Modula_Pro_Base_Updater(
				$store_url,
				MODULA_DOWNLOAD_FILE,
				array(
					'version' => MODULA_DOWNLOAD_VERSION,        // current version number
					'license' => $license_key,               // license key (used get_option above to retrieve from DB)
					'item_id' => 447525,                      // ID of the product
					'author'  => 'MachoThemes',            // author of this plugin
					'beta'    => false,
				)
			);
		}
	}

	/**
	 * Returns the singleton instance of the class.
	 *
	 * @return object The Modula_Download object.
	 * @since 1.0.0
	 */
	public static function get_instance() {
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Modula_Download ) ) {
			self::$instance = new Modula_Download();
		}

		return self::$instance;
	}

	/**
	 * Loads all admin related files into scope.
	 *
	 * @since 1.0.0
	 */
	public function require_admin() {

			add_action( 'modula_admin_after_shortcode_metabox', array( $this, 'output_download_shortcode' ), 30 );
			add_action( 'modula_albums_admin_after_shortcode_metabox', array( $this, 'output_download_album_shortcode' ), 30 );
			add_action( 'manage_modula-gallery_posts_custom_column', array( $this, 'modula_download_gallery_column' ), 30, 2 );
			add_action( 'manage_modula-album_posts_custom_column', array( $this, 'modula_download_album_column' ), 20, 2 );

			require_once MODULA_DOWNLOAD_PATH . 'includes/admin/class-modula-download-settings.php';
			require_once MODULA_DOWNLOAD_PATH . 'includes/admin/class-modula-download-album-settings.php';
	}

	/**
	 * Loads all the helper functions into scope
	 *
	 * @since 1.0.0
	 */
	public function require_helpers() {

		require_once MODULA_DOWNLOAD_PATH . 'includes/modula-download-helper.php';
	}

	/**
	 * Add defaults settings for download
	 *
	 * @param array $defaults
	 *
	 * @return array $defaults
	 * @since 1.0.0
	 */
	public function modula_download_defaults( $defaults ) {
		$defaults['enable_download']                  = 0;
		$defaults['download_gallery_button']          = 1;
		$defaults['download_image_sizes']             = 'full';
		$defaults['download_all_gallery_button']      = 1;
		$defaults['download_all_color']               = '';
		$defaults['download_all_background_color']    = '';
		$defaults['download_all_size']                = 'medium';
		$defaults['download_all_hposition']           = 'center';
		$defaults['download_all_position']            = 'below_gallery';
		$defaults['download_all_label']               = 'Download All Images';
		$defaults['custom_zip_name']                  = '%%gallery_title%%';
		$defaults['download_all_lightbox_button']     = 1;
		$defaults['download_all_gallery_button_icon'] = 1;
		$defaults['download_all_gallery_icon_color']  = '';

		return $defaults;
	}

	/**
	 * Add defaults settings for download
	 *
	 * @param array $defaults
	 *
	 * @return array $defaults
	 * @since 1.0.0
	 */
	public function modula_download_albums_defaults( $defaults ) {
		$defaults['enable_download_albums']                  = 0;
		$defaults['download_gallery_button_albums']          = 1;
		$defaults['download_image_sizes_albums']             = 'full';
		$defaults['download_gallery_button_position_albums'] = 'top_left';
		$defaults['download_all_gallery_button_albums']      = 1;
		$defaults['download_all_color_albums']               = '';
		$defaults['download_all_size_albums']                = 'medium';
		$defaults['download_all_hposition_albums']           = 'center';
		$defaults['download_all_position_albums']            = 'below_gallery';
		$defaults['download_all_label_albums']               = 'Download All Images';
		$defaults['custom_zip_name_albums']                  = '%%album_title%%';
		$defaults['download_all_lightbox_button_albums']     = 1;
		$defaults['download_gallery_button_color_albums']    = '#ffffff';
		$defaults['download_all_gallery_button_icon_albums'] = 1;
		$defaults['download_all_gallery_icon_color_albums']  = '';
		$defaults['download_all_color_albums']               = '#000';
		$defaults['download_all_font_size_albums ']          = 0;
		$defaults['download_all_mobile_font_size_albums ']   = 0;

		return $defaults;
	}

	/**
	 * Add Modula Download placeholders
	 *
	 * @param $placeholders
	 *
	 * @return mixed
	 * @since 1.0.0
	 */
	public function modula_download_placeholders( $placeholders ) {

		$placeholders = array_merge(
			array(
				'%%gallery_title%%'          => esc_html__( 'Gallery Title', 'modula-download' ),
				'%%gallery_id%%'             => esc_html__( 'Gallery ID', 'modula-download' ),
				'%%gallery_author%%'         => esc_html__( 'Gallery Author', 'modula-download' ),
				'%%gallery_date%%'           => esc_html__( 'Gallery Date', 'modula-download' ),
				'%%archive_generated_time%%' => esc_html__( 'Zip Created Time', 'modula-download' ),
			),
			$placeholders
		);

		return $placeholders;
	}

	/**
	 * Add Modula Download placeholders
	 *
	 * @param $placeholders
	 *
	 * @return mixed
	 * @since 1.0.0
	 */
	public function modula_albums_download_placeholders( $placeholders ) {

		$placeholders = array_merge(
			array(
				'%%album_title%%'            => esc_html__( 'Album Title', 'modula-download' ),
				'%%album_id%%'               => esc_html__( 'Album ID', 'modula-download' ),
				'%%album_author%%'           => esc_html__( 'Album Author', 'modula-download' ),
				'%%album_date%%'             => esc_html__( 'Album Date', 'modula-download' ),
				'%%archive_generated_time%%' => esc_html__( 'Zip Created Time', 'modula-download' ),
			),
			$placeholders
		);

		return $placeholders;
	}


	/**
	 * Modula Download placeholder real values
	 *
	 * @param $values
	 * @param $object
	 *
	 * @return array
	 * @since 1.0.0
	 */
	public function modula_download_placeholder_values( $values, $object ) {

		if ( ! is_wp_error( $object ) ) {
			return array_merge(
				$values,
				array(
					'setting' => array(
						'%%gallery_title%%',
						'%%gallery_id%%',
						'%%gallery_author%%',
						'%%gallery_date%%',
						'%%archive_generated_time%%',
					),
					'front'   => array(
						$object->post_title . ' ',
						$object->ID . ' ',
						get_the_author_meta( 'nickname', $object->post_author ) . ' ',
						explode( ' ', $object->post_date )[0] . ' ',
						date( 'Y-m-d' ) . ' ',
					),
				)
			);
		}

		return $values;
	}

	/**
	 * Modula Download placeholder real values
	 *
	 * @param $values
	 * @param $object
	 *
	 * @return array
	 * @since 1.0.0
	 */
	public function modula_albums_download_placeholder_values( $values, $object ) {

		if ( ! is_wp_error( $object ) ) {
			return array_merge(
				$values,
				array(
					'setting' => array(
						'%%album_title%%',
						'%%album_id%%',
						'%%album_author%%',
						'%%album_date%%',
						'%%archive_generated_time%%',
					),
					'front'   => array(
						$object->post_title . ' ',
						$object->ID . ' ',
						get_the_author_meta( 'nickname', $object->post_author ) . ' ',
						explode( ' ', $object->post_date )[0] . ' ',
						date( 'Y-m-d' ) . ' ',
					),
				)
			);
		}

		return $values;
	}

	/**
	 * Enqueue necessary download scripts
	 *
	 * @param array $settings
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function modula_download_scripts( $settings ) {
		if ( isset( $settings['enable_download'] ) && 1 == $settings['enable_download'] ) {
			wp_enqueue_style( 'modula-download-css', MODULA_DOWNLOAD_URL . 'assets/css/modula-download-front.css', MODULA_DOWNLOAD_VERSION );
		}
	}

	/**
	 * Enqueue necessary link scripts
	 *
	 * @param array $settings
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function modula_download_link_scripts( $settings ) {
		if ( isset( $settings['enable_download'] ) && 1 == $settings['enable_download'] ) {
			wp_enqueue_style( 'modula-download-css', MODULA_DOWNLOAD_URL . 'assets/css/modula-download-front.css', MODULA_DOWNLOAD_VERSION );
		}
	}

	/**
	 * Enqueue necessary download albums scripts
	 *
	 * @param array $settings
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function modula_download_albums_scripts( $settings ) {
		if ( 1 == $settings['enable_download_albums'] ) {
			wp_enqueue_style( 'modula-download-css', MODULA_DOWNLOAD_URL . 'assets/css/modula-download-front.css', MODULA_DOWNLOAD_VERSION );
		}
	}

	/**
	 * Output download shortcode in admin metabox
	 *
	 * @param object $post
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function output_download_shortcode( $post ) {
		$shortcode = '[modula-download id="' . $post->ID . '"]' . esc_html__( 'Download Gallery', 'modula-download' ) . '[/modula-download]';
		echo '<div class="modula-copy-shortcode">';
		echo '<input type="text" value="' . esc_attr( $shortcode ) . '"  onclick="select()" readonly>';
		echo '<a href="#" title="' . esc_attr__( 'Copy shortcode', 'modula-download' ) . '" class="copy-modula-shortcode button button-primary dashicons dashicons-format-gallery" style="width:40px;"></a><span></span>';
		echo '<p>' . esc_html__( 'You can use this shortcode to create a download button for your newly created gallery ', 'modula-download' ) . '</p>';
		echo '</div>';
	}

	/**
	 * Output download shortcode for albums in admin metabox
	 *
	 * @param object $post
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function output_download_album_shortcode( $post ) {

		$shortcode = '[modula-download id="' . $post->ID . '"]' . esc_html__( 'Download Album', 'modula-download' ) . '[/modula-download]';
		echo '<div class="modula-copy-shortcode">';
		echo '<input type="text" value="' . esc_attr( $shortcode ) . '"  onclick="select()" readonly>';
		echo '<a href="#" title="' . esc_attr__( 'Copy shortcode', 'modula-download' ) . '" class="copy-modula-shortcode button button-primary dashicons dashicons-format-gallery" style="width:40px;"></a><span></span>';
		echo '<p>' . esc_html__( 'You can use this shortcode to create a download button for your newly created album ', 'modula-download' ) . '</p>';
		echo '</div>';

	}

	/**
	 * Force Download the browser to download the gallery
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function modula_force_download() {

		if ( ! isset( $_REQUEST['modula-id'] ) ) {
			return;
		}

		$id        = explode( '-', $_REQUEST['modula-id'] );
		$id        = end( $id );
		$post_type = get_post_type( $id );

		if ( ! in_array( $post_type, array( 'modula-gallery', 'modula-album' ) ) ) {
			return;
		}

		$settings     = array();
		$images       = array();
		$custom_name  = 'modula-download';
		$upload_dir   = wp_upload_dir();
		$images_paths = array();

		if ( 'modula-gallery' == $post_type ) {
			$settings = get_post_meta( $id, 'modula-settings', true );
			$settings = wp_parse_args( $settings, Modula_CPT_Fields_Helper::get_defaults() );
			$images   = get_post_meta( $id, 'modula-images', true );

			$custom_name = 'modula-gallery-' . $id;
			if ( ! empty( $settings['custom_zip_name'] ) ) {
				$gallery_info = get_post( $id );

				if ( class_exists( 'Modula_Pro_Helper' ) && ! empty( Modula_Helper::modula_placeholders_value( $settings['custom_zip_name'], $gallery_info ) ) && ' ' != Modula_Helper::modula_placeholders_value( $settings['custom_zip_name'], $gallery_info ) ) {

					$custom_name = trim( sanitize_text_field( Modula_Helper::modula_placeholders_value( $settings['custom_zip_name'], $gallery_info ) ), '/' );
				}
			}
		}

		if ( 'modula-album' == $post_type ) {
			$settings  = get_post_meta( $id, 'modula-album-settings', true );
			$settings  = wp_parse_args( $settings, Modula_Album_Fields_Helper::get_defaults() );
			$galleries = get_post_meta( $id, 'modula-album-galleries', true );
			$galleries = get_post_meta( $id, 'modula-album-galleries', true );

			foreach ( $galleries as $gallery ) {
				$gal_images = get_post_meta( $gallery['id'], 'modula-images', true );
				$images     = array_merge( $images, $gal_images );
			}

			$custom_name = 'modula-album-' . $id;

			if ( ! empty( $settings['custom_zip_name_albums'] ) ) {
				$album_info = get_post( $id );
				if ( class_exists( 'Modula_Pro_Helper' ) && ! empty( Modula_Pro_Helper::modula_albums_placeholders_value( $settings['custom_zip_name_albums'], $album_info ) ) && ' ' != Modula_Pro_Helper::modula_albums_placeholders_value( $settings['custom_zip_name_albums'], $album_info ) ) {

					$custom_name = trim( sanitize_text_field( Modula_Pro_Helper::modula_albums_placeholders_value( $settings['custom_zip_name_albums'], $album_info ) ), '/' );
				}
			}
		}

		if ( ! empty( $images ) ) {
			foreach ( $images as $image ) {

				$image_object = false;

				if ( isset( $settings['enable_download'] ) ) {
					if ( $settings['enable_download'] ) {
						if ( $settings['download_image_sizes'] != 'full' ) {
							$image_object = wp_get_attachment_image_src( $image['id'], $settings['download_image_sizes'] );
						} elseif ( $settings['download_image_sizes'] == 'full' ) {
							$image_url = wp_get_original_image_url( $image['id'] );
						}
					} else {
						$image_url = wp_get_original_image_url( $image['id'] );
					}
				}

				if ( isset( $settings['enable_download_albums'] ) ) {
					if ( $settings['enable_download_albums'] ) {
						if ( $settings['download_image_sizes_albums'] != 'full' ) {
							$image_object = wp_get_attachment_image_src( $image['id'], $settings['download_image_sizes_albums'] );
						} elseif ( $settings['download_image_sizes_albums'] == 'full' ) {
							$image_url = wp_get_original_image_url( $image['id'] );
						}
					} else {
						$image_url = wp_get_original_image_url( $image['id'] );
					}
				}

				if ( $image_object ) {
					$image_url = $image_object[0];
				}

				$image_path = str_replace( $upload_dir['baseurl'], $upload_dir['basedir'], $image_url );

				if ( $image_path ) {
					$images_paths[] = $image_path;
				}
			}
		}

		$requested_file_path = "{$upload_dir['path']}/{$custom_name}.zip";

		$result    = modula_download_zip( $images_paths, $requested_file_path );
		$file_path = str_replace( content_url(), WP_CONTENT_DIR, $requested_file_path );
		modula_download_file( $file_path, $requested_file_path );
	}

	/**
	 * Generate Gallery CSS for position of the dl button
	 *
	 * @param string $css
	 * @param string $gall_id
	 * @param array  $settings
	 *
	 * @return string $css
	 * @since 1.0.0
	 */
	public function modula_download_gallery_css( $css, $gall_id, $settings ) {

		if ( isset( $settings['enableSocial'] ) && 1 == $settings['enableSocial'] ) {

			if ( 1 == $settings['download_gallery_button'] ) {

				if ( isset( $settings['socialIconColor'] ) && '' != $settings['socialIconColor'] ) {
					$css .= "#{$gall_id} .modula-item .jtg-social a.modula-download-button,#{$gall_id} .modula-item .no-socials a.modula-download-button { color: " . Modula_Helper::sanitize_rgba_colour( $settings['socialIconColor'] ) . ' }';
				} else {
					$css .= "#{$gall_id}  .modula-item .jtg-social a.modula-download-button,#{$gall_id} .modula-item .no-socials a.modula-download-button { color: " . Modula_Helper::sanitize_rgba_colour( $settings['socialIconColor'] ) . ' }';
				}
				if ( 'under' == $settings['effect'] && 0 == $settings['enableSocial'] ) {
					$css .= "#{$gall_id} .no-socials { position: absolute; width: " . absint( $settings['socialIconSize'] ) . 'px; height: ' . absint( $settings['socialIconSize'] ) . 'px;margin-left: 25px; bottom: 25px; z-index:100 }';
				}
			}

			if ( 1 == $settings['download_all_gallery_button'] || 1 == $settings['download_all_gallery_button_icon'] ) {

				if ( isset( $settings['download_all_hposition'] ) && 'left' != $settings['download_all_hposition'] ) {

					if ( 'center' == $settings['download_all_hposition'] ) {

						$css .= "#{$gall_id} .modula-download-all-button { text-align: center }";
					} elseif ( 'right' == $settings['download_all_hposition'] ) {

						$css .= "#{$gall_id} .modula-download-all-button { text-align: right }";
					}
				}

				if ( isset( $settings['download_all_color'] ) && '' != $settings['download_all_color'] ) {

					$css .= "#{$gall_id} .modula-download-all-button a{";
					$css .= 'color:' . Modula_Helper::sanitize_rgba_colour( $settings['download_all_color'] ) . ';';
					$css .= '}';
				}

				if ( isset( $settings['download_all_background_color'] ) && '' != $settings['download_all_background_color'] ) {
					$css .= "#{$gall_id} .modula-download-all-button .modula-btn-" . esc_attr( $settings['download_all_size'] ) . "{";
					$css .= 'background-color:' . Modula_Helper::sanitize_rgba_colour( $settings['download_all_background_color'] ) . ';';
					$css .= '}';
				}

				if ( isset( $settings['download_all_gallery_icon_color'] ) && '' != $settings['download_all_gallery_icon_color'] ) {
					$css .= "#{$gall_id} .modula-download-all-button svg path { fill : " . Modula_Helper::sanitize_rgba_colour( $settings['download_all_gallery_icon_color'] ) . ' }';
				} else {
					$css .= "#{$gall_id} .modula-download-all-button svg path { fill: #000; }";
				}
			}
		}
		return $css;
	}

	/**
	 * Add download button and download all in $item_data
	 *
	 * @param array $item_data
	 * @param array $image
	 * @param array $settings
	 *
	 * @return array $item_data
	 * @since 1.0.0
	 */
	public function modula_download_item_data( $item_data, $image, $settings ) {

		if ( isset( $settings['enable_download'] ) && 1 == $settings['enable_download'] ) {
			if ( $settings['download_image_sizes'] != 'full' ) {
				$item_data['download'] = array(
					'download_button' => $settings['enable_download'] ? $settings['download_gallery_button'] : 0,
					'image_url'       => wp_get_attachment_image_src( $image['id'], $settings['download_image_sizes'] )[0],
				);
			} elseif ( $settings['download_image_sizes'] == 'full' ) {
				$item_data['download'] = array(
					'download_button' => $settings['enable_download'] ? $settings['download_gallery_button'] : 0,
					'image_url'       => wp_get_original_image_url( $image['id'] ),
				);
			}
		}
		return $item_data;
	}

	/**
	 * Display the download button on hover
	 *
	 * @param object $data
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function modula_download_gallery_button( $data ) {

		if ( isset( $data->download['download_button'] ) && $data->download['download_button'] ) {
			?>
			<a href="<?php echo esc_url( $data->download['image_url'] ); ?>" target="_self" download
			   class="modula-download-button" title="Download image">
				<svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="download"
					 class="svg-inline--fa fa-download fa-w-16" role="img" xmlns="http://www.w3.org/2000/svg"
					 viewBox="0 0 512 512">
					<path fill="currentColor"
						  d="M216 0h80c13.3 0 24 10.7 24 24v168h87.7c17.8 0 26.7 21.5 14.1 34.1L269.7 378.3c-7.5 7.5-19.8 7.5-27.3 0L90.1 226.1c-12.6-12.6-3.7-34.1 14.1-34.1H192V24c0-13.3 10.7-24 24-24zm296 376v112c0 13.3-10.7 24-24 24H24c-13.3 0-24-10.7-24-24V376c0-13.3 10.7-24 24-24h146.7l49 49c20.1 20.1 52.5 20.1 72.6 0l49-49H488c13.3 0 24 10.7 24 24zm-124 88c0-11-9-20-20-20s-20 9-20 20 9 20 20 20 20-9 20-20zm64 0c0-11-9-20-20-20s-20 9-20 20 9 20 20 20 20-9 20-20z"></path>
				</svg>
			</a>
			<?php
		}
	}

	/**
	 * Add download button to hover effects that don't have socials
	 *
	 * @param $data
	 */
	public function modula_download_no_socials( $data ) {

		if ( isset( $data->download['download_button'] ) && $data->download['download_button'] && $data->hide_socials ) {
			?>
			<div class="no-socials">
				<a href="<?php echo esc_url( $data->download['image_url'] ); ?>" target="_self" download
				   class="modula-download-button" title="Download image">
					<svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="download"
						 class="svg-inline--fa fa-download fa-w-16" role="img" xmlns="http://www.w3.org/2000/svg"
						 viewBox="0 0 512 512">
						<path fill="currentColor"
							  d="M216 0h80c13.3 0 24 10.7 24 24v168h87.7c17.8 0 26.7 21.5 14.1 34.1L269.7 378.3c-7.5 7.5-19.8 7.5-27.3 0L90.1 226.1c-12.6-12.6-3.7-34.1 14.1-34.1H192V24c0-13.3 10.7-24 24-24zm296 376v112c0 13.3-10.7 24-24 24H24c-13.3 0-24-10.7-24-24V376c0-13.3 10.7-24 24-24h146.7l49 49c20.1 20.1 52.5 20.1 72.6 0l49-49H488c13.3 0 24 10.7 24 24zm-124 88c0-11-9-20-20-20s-20 9-20 20 9 20 20 20 20-9 20-20zm64 0c0-11-9-20-20-20s-20 9-20 20 9 20 20 20 20-9 20-20z"></path>
					</svg>
				</a>
			</div>
			<?php
		}
	}

	/**
	 * Slider download button
	 *
	 * @param $data
	 * @since 1.0.0
	 */
	public function slider_download_button( $data ) {

		if ( isset( $data->download['download_button'] ) && '1' == $data->download['download_button'] ) {
			?>
			<div class="no-socials">
				<a href="<?php echo esc_url( $data->download['image_url'] ); ?>" target="_self" download
				   class="modula-download-button" title="Download image">
					<svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="download"
						 class="svg-inline--fa fa-download fa-w-16" role="img" xmlns="http://www.w3.org/2000/svg"
						 viewBox="0 0 512 512">
						<path fill="currentColor"
							  d="M216 0h80c13.3 0 24 10.7 24 24v168h87.7c17.8 0 26.7 21.5 14.1 34.1L269.7 378.3c-7.5 7.5-19.8 7.5-27.3 0L90.1 226.1c-12.6-12.6-3.7-34.1 14.1-34.1H192V24c0-13.3 10.7-24 24-24zm296 376v112c0 13.3-10.7 24-24 24H24c-13.3 0-24-10.7-24-24V376c0-13.3 10.7-24 24-24h146.7l49 49c20.1 20.1 52.5 20.1 72.6 0l49-49H488c13.3 0 24 10.7 24 24zm-124 88c0-11-9-20-20-20s-20 9-20 20 9 20 20 20 20-9 20-20zm64 0c0-11-9-20-20-20s-20 9-20 20 9 20 20 20 20-9 20-20z"></path>
					</svg>
				</a>
			</div>
			<?php
		}
	}

	/**
	 * Display the download all (ABOVE) && (ABOVE && BELLOW ) button
	 *
	 * @param array $settings
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function modula_download_all_gallery_button_above( $settings ) {

		if ( ! isset( $settings['enable_download'] ) || ! $settings['enable_download'] || ! $settings['download_all_gallery_button'] ) {
			return;
		}

		$accepted_positions = array( 'above_gallery', 'above_below_gallery' );

		if ( ! in_array( $settings['download_all_position'], $accepted_positions ) ) {
			return;
		}
		?>
		<div class="modula-download-all-button">
			<button class="modula-btn-<?php echo esc_attr( $settings['download_all_size'] ); ?>">
				<a href="<?php echo esc_url( modula_download_req_url( $settings['gallery_id'] ) ); ?>"
				target="_self"
				class="modula-download-all">
					<?php
					if ( isset( $settings['download_all_gallery_button_icon'] ) && '1' == $settings['download_all_gallery_button_icon'] ) {
						echo modula_download_all_svg();
					}
					echo ! empty( $settings['download_all_label'] ) ? esc_html( $settings['download_all_label'] ) : esc_html__( 'Download All Images', 'modula-download' )
					?>
				</a>
			</button>
		</div>
		<?php
	}

	/**
	 * Display the download all (BELOW) && (ABOVE && BELOW ) button
	 *
	 * @param array $settings
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function modula_download_all_gallery_button_below( $settings ) {

		if ( ! isset( $settings['enable_download'] ) || ! $settings['enable_download'] || ! $settings['download_all_gallery_button'] ) {
			return;
		}
		$accepted_positions = array( 'below_gallery', 'above_below_gallery' );

		if ( ! in_array( $settings['download_all_position'], $accepted_positions ) ) {
			return;
		}

		?>
		<div class="modula-download-all-button">
			<button class="modula-btn-<?php echo esc_attr( $settings['download_all_size'] ); ?>">
				<a href="<?php echo esc_url( modula_download_req_url( $settings['gallery_id'] ) ); ?>"
				target="_self"
				class="modula-download-all">
					<?php
					if ( isset( $settings['download_all_gallery_button_icon'] ) && '1' == $settings['download_all_gallery_button_icon'] ) {
						echo modula_download_all_svg();
					}
					echo ! empty( $settings['download_all_label'] ) ? esc_html( $settings['download_all_label'] ) : esc_html__( 'Download All Images', 'modula-download' )
					?>
				</a>
			</button>
		</div>
		<?php
	}

	/**
	 * Add a download all button in lightbox
	 *
	 * @param array $fancybox_options
	 * @param array $settings
	 *
	 * @return array $fancybox_options
	 * @since 1.0.0
	 */
	public function modula_download_fancybox_button( $fancybox_options, $settings ) {

		if ( ! isset( $settings['enable_download'] ) || ! $settings['enable_download'] || ! $settings['download_all_lightbox_button'] ) {
			return $fancybox_options;
		}

		$fancybox_options['btnTpl']['downloadAll'] = '<a class="modula-fancybox-button download-all" title="Download All" href="' . esc_url( modula_download_req_url( $settings['gallery_id'] ) ) . '"> <svg xmlns="http://www.w3.org/2000/svg" width="2480" height="3507" viewBox="0 0 2480 3507">
        <rect class="cls-1" x="5" y="2512" width="2475" height="996"/>
        <path id="ALL" class="cls-2" d="M627.528,2585.41L352.443,3406.16H487.6l64.474-199.75H848.647L917.9,3406.16H1054.01L759.34,2585.41H627.528Zm66.383,175.66,112.709,324.9H591.232Zm438.419-175.66v820.75h452.74V3282.19h-323.8V2585.41H1132.33Zm541.09,0v820.75h452.75V3282.19h-323.8V2585.41H1673.42Z"/>
        <rect class="cls-1" x="1001" y="6" width="453" height="1506"/>
        <path class="cls-3" d="M296,1399.6l295.125-296.49,406.3,408.19H1456.28L1869.66,1096,2185,1412.8,1253.13,2349Z"/>
        </svg></a>';
		array_unshift( $fancybox_options['buttons'], 'downloadAll' );

		return $fancybox_options;
	}

	/**
	 * Generate CSS for Albums
	 *
	 * @param string $css
	 * @param string $gall_id
	 * @param array  $settings
	 *
	 * @return string $css
	 * @since  1.0.0
	 */
	public function modula_download_album_css( $css, $gall_id, $settings ) {

		if ( isset( $settings['enable_download_albums'] ) && 1 == $settings['enable_download_albums'] ) {
			if ( 1 == $settings['download_gallery_button_albums'] ) {
				if ( 'top_left' == $settings['download_gallery_button_position_albums'] ) {
					$css .= "#{$gall_id} .modula-item a.modula-download-button { top:25px; left:25px; bottom:auto;right:auto;}";
				}
				if ( 'top_right' == $settings['download_gallery_button_position_albums'] ) {
					$css .= "#{$gall_id} .modula-item a.modula-download-button { top:25px; right:25px;bottom:auto; }";
				}
				if ( 'bottom_left' == $settings['download_gallery_button_position_albums'] ) {
					$css .= "#{$gall_id} .modula-item a.modula-download-button { bottom:25pxpx; left:25px;right:auto;  }";
				}
				if ( 'bottom_right' == $settings['download_gallery_button_position_albums'] ) {
					$css .= "#{$gall_id} .modula-item a.modula-download-button { bottom:25px; right:25px;  }";
				}

				if ( isset( $settings['download_gallery_button_color_albums'] ) && '' != $settings['download_gallery_button_color_albums'] ) {
					$css .= "#{$gall_id}  .modula-item a.modula-download-button { color: " . Modula_Helper::sanitize_rgba_colour( $settings['download_gallery_button_color_albums'] ) . ' }';
				}
			}

			if ( 1 == $settings['download_all_gallery_button_albums'] || 1 == $settings['download_all_gallery_button_icon_albums'] ) {

				if ( isset( $settings['download_all_hposition_albums'] ) && 'left' != $settings['download_all_hposition_albums'] ) {
					if ( 'center' == $settings['download_all_hposition_albums'] ) {
						$css .= "#{$gall_id} .modula-download-all-button { text-align: center }";
					} elseif ( 'right' == $settings['download_all_hposition_albums'] ) {
						$css .= "#{$gall_id} .modula-download-all-button { text-align: right }";
					}
				}
				if ( isset( $settings['download_all_color_albums'] ) && '' != $settings['download_all_color_albums'] ) {

					$css .= "#{$gall_id} .modula-download-all-button a{color:" . sanitize_hex_color( $settings['download_all_color_albums'] ) . ';}';
				}

				if ( isset( $settings['download_all_gallery_icon_color_albums'] ) && '' != $settings['download_all_gallery_icon_color_albums'] ) {
					$css .= "#{$gall_id} .modula-download-all-button svg path { fill : " . Modula_Helper::sanitize_rgba_colour( $settings['download_all_gallery_icon_color_albums'] ) . ' }';
				} else {
					$css .= "#{$gall_id} .modula-download-all-button svg path { fill : black }";
				}
			}
		}
		return $css;
	}

	/**
	 * Add shortcode on init
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function modula_download_add_shortcode() {
		add_shortcode( 'modula-download', array( $this, 'modula_download_shortcode' ) );
	}

	/**
	 * Render download all shortcode
	 *
	 * @param array  $atts
	 * @param string $content
	 *
	 * @return string $html
	 * @since 1.0.0
	 */
	public function modula_download_shortcode( $atts, $content = null ) {
		$defaults_atts = array(
			'id' => false,
		);

		$atts = wp_parse_args( $atts, $defaults_atts );

		if ( '' == $atts['id'] || ! $atts['id'] ) {
			return esc_html__( 'Gallery not found', 'modula-download' );
		}

		$gallery        = get_post( absint( $atts['id'] ) );
		$accepted_posts = array( 'modula-gallery', 'modula-album' );

		if ( ! in_array( get_post_type( $gallery ), $accepted_posts ) ) {
			return esc_html__( 'Given ID doesn\'t belong to a Modula Gallery!', 'modula-download' );
		}

		switch ( get_post_type( $gallery ) ) {
			case 'modula-gallery':
				$settings = get_post_meta( $atts['id'], 'modula-settings', true );
				$defaults = Modula_CPT_Fields_Helper::get_defaults();
				$settings = wp_parse_args( $settings, $defaults );

				ob_start();

				echo $this->generate_download_css( $atts['id'], $settings );
				echo '<div class="modula-download-all-button" id="jtg-' . esc_attr( $atts['id'] ) . '-download"><button class="modula-btn-' . esc_attr( $settings['download_all_size'] ) . '"><a href="' . esc_url( modula_download_req_url( $atts['id'] ) ) . '" target="_self" class="modula-download-all">' . modula_download_all_svg() . do_shortcode( $content ) . '</a></button></div>';

				$html = ob_get_clean();

				return $html;
				break;

			case 'modula-album':
				$settings = get_post_meta( $atts['id'], 'modula-album-settings', true );
				$defaults = Modula_Album_Fields_Helper::get_defaults();
				$settings = wp_parse_args( $settings, $defaults );

				ob_start();
				echo $this->generate_album_download_css( $atts['id'], $settings );
				echo '<div class="modula-download-all-button" id="jtg-album-' . esc_attr( $atts['id'] ) . '-download"><button class="modula-btn-' . esc_attr( $settings['download_all_size_albums'] ) . '"><a href="' . esc_url( modula_download_req_url( $atts['id'] ) ) . '" target="_self" class="modula-download-all">' . modula_download_all_svg() . do_shortcode( $content ) . '</a></button></div>';

				$html = ob_get_clean();

				return $html;
				break;
		}

	}

	/**
	 * Add download all shortcode in the columns
	 *
	 * @param string $column
	 * @param int    $post_id
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function modula_download_gallery_column( $column, $post_id ) {
		if ( 'shortcode' == $column ) {
			$shortcode = '[modula-download id="' . $post_id . '"]' . esc_html__( 'Download Gallery', 'modula-download' ) . '[/modula-download]';
			echo '<div class="modula-copy-shortcode">';
			echo '<input type="text" value="' . esc_attr( $shortcode ) . '"  onclick="select()" readonly>';
			echo '<a href="#" title="' . esc_attr__( 'Copy shortcode', 'modula-pro' ) . '" class="copy-modula-shortcode button button-primary dashicons dashicons-format-gallery" style="width:40px;"></a><span></span>';
			echo '</div>';
		}
	}

	/**
	 * Add download all shortcode in albums post type
	 *
	 * @param string $column
	 * @param int    $post_id
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function modula_download_album_column( $column, $post_id ) {
		if ( 'shortcode' == $column ) {
			$shortcode = '[modula-download id="' . $post_id . '"]' . esc_html__( 'Download Album', 'modula-download' ) . '[/modula-download]';
			echo '<div class="modula-copy-shortcode">';
			echo '<input type="text" value="' . esc_attr( $shortcode ) . '"  onclick="select()" readonly>';
			echo '<a href="#" title="' . esc_attr__( 'Copy shortcode', 'modula-pro' ) . '" class="copy-modula-shortcode button button-primary dashicons dashicons-format-gallery" style="width:40px;"></a><span></span>';
			echo '</div>';
		}
	}

	/**
	 * Add download settings in item data
	 *
	 * @param array $item_data
	 * @param array $image
	 * @param array $gall_settings
	 * @param array $settings
	 * @param array $images
	 *
	 * @return array $item_data
	 * @since 1.0.0
	 */
	public function modula_download_albums_item_data( $item_data, $image, $gall_settings, $settings, $images ) {

		if ( isset( $settings['enable_download_albums'] ) && 1 == $settings['enable_download_albums'] ) {
			$item_data['download'] = array(
				'download_button' => $settings['enable_download_albums'] ? $settings['download_gallery_button_albums'] : 0,
				'album_id'        => $settings['gallery_id'],
			);
		}
		return $item_data;
	}

	/**
	 * Add download all lightbox in albums
	 *
	 * @param $template_data
	 *
	 * @return array $js_config
	 * @since 1.0.0
	 */
	public function modula_download_album_fancybox_button( $template_data ) {

		if ( ! isset( $template_data['settings']['enable_download_albums'] ) || ! $template_data['settings']['enable_download_albums'] || ! $template_data['settings']['download_all_lightbox_button_albums'] ) {
			return $template_data;
		}

		$template_data['js_config']['lightbox_settings']['btnTpl']['downloadAll'] = '<a class="modula-fancybox-button download-all" title="Download All" href="' . esc_url( modula_download_req_url( $template_data['settings']['gallery_id'] ) ) . '"> <svg xmlns="http://www.w3.org/2000/svg" width="2480" height="3507" viewBox="0 0 2480 3507">
        <rect class="cls-1" x="5" y="2512" width="2475" height="996"/>
        <path id="ALL" class="cls-2" d="M627.528,2585.41L352.443,3406.16H487.6l64.474-199.75H848.647L917.9,3406.16H1054.01L759.34,2585.41H627.528Zm66.383,175.66,112.709,324.9H591.232Zm438.419-175.66v820.75h452.74V3282.19h-323.8V2585.41H1132.33Zm541.09,0v820.75h452.75V3282.19h-323.8V2585.41H1673.42Z"/>
        <rect class="cls-1" x="1001" y="6" width="453" height="1506"/>
        <path class="cls-3" d="M296,1399.6l295.125-296.49,406.3,408.19H1456.28L1869.66,1096,2185,1412.8,1253.13,2349Z"/>
        </svg></a>';

		array_unshift( $template_data['js_config']['lightbox_settings']['buttons'], 'downloadAll' );
		$template_data['album_container']['data-config'] = json_encode( $template_data['js_config'] );

		return $template_data;
	}

	/**
	 * Add a download button hovering over image in album
	 *
	 * @param object $data
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function modula_download_album_button( $data ) {
		if ( isset( $data->download['download_button'] ) && $data->download['download_button'] ) {
			?>
			<div class="no-socials">
				<a href="<?php echo esc_url( modula_download_req_url( $data->gallery_id ) ); ?>"
				   target="_self" download
				   class="modula-download-button" title="Download image">
					<svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="download"
						 class="svg-inline--fa fa-download fa-w-16" role="img" xmlns="http://www.w3.org/2000/svg"
						 viewBox="0 0 512 512">
						<path fill="currentColor"
							  d="M216 0h80c13.3 0 24 10.7 24 24v168h87.7c17.8 0 26.7 21.5 14.1 34.1L269.7 378.3c-7.5 7.5-19.8 7.5-27.3 0L90.1 226.1c-12.6-12.6-3.7-34.1 14.1-34.1H192V24c0-13.3 10.7-24 24-24zm296 376v112c0 13.3-10.7 24-24 24H24c-13.3 0-24-10.7-24-24V376c0-13.3 10.7-24 24-24h146.7l49 49c20.1 20.1 52.5 20.1 72.6 0l49-49H488c13.3 0 24 10.7 24 24zm-124 88c0-11-9-20-20-20s-20 9-20 20 9 20 20 20 20-9 20-20zm64 0c0-11-9-20-20-20s-20 9-20 20 9 20 20 20 20-9 20-20z"></path>
					</svg>
				</a>
			</div>
			<?php
		}
	}

	/**
	 * Display the download all (ABOVE) && (ABOVE && BELOW ) button
	 *
	 * @param array $settings
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function modula_download_all_album_button_above( $settings ) {

		if ( ! isset( $settings['enable_download_albums'] ) || ! $settings['enable_download_albums'] || ! $settings['download_all_gallery_button_albums'] ) {
			return;
		}

		$accepted_positions = array( 'above_gallery', 'above_below_gallery' );

		if ( ! in_array( $settings['download_all_position_albums'], $accepted_positions ) ) {
			return;
		}
		?>
		<div class="modula-download-all-button">
			<button class="modula-btn-<?php echo esc_attr( $settings['download_all_size_albums'] ); ?>">
				<a href="<?php echo esc_url( modula_download_req_url( $settings['gallery_id'] ) ); ?>"
				target="_self"
				class="modula-download-all">
					<?php
					if ( isset( $settings['download_all_gallery_button_icon_albums'] ) && '1' == $settings['download_all_gallery_button_icon_albums'] ) {
						echo modula_download_all_svg();
					}
					echo ! empty( $settings['download_all_label_albums'] ) ? esc_html( $settings['download_all_label_albums'] ) : esc_html__( 'Download All Images', 'modula-download' )
					?>
				</a>
			</button>
		</div>
		<?php
	}

	/**
	 * Display the download all (BELOW) && (ABOVE && BELOW ) button
	 *
	 * @param array $settings
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function modula_download_all_album_button_below( $settings ) {
		if ( ! isset( $settings['enable_download_albums'] ) || ! $settings['enable_download_albums'] || ! $settings['download_all_gallery_button_albums'] ) {
			return;
		}

		$accepted_positions = array( 'below_gallery', 'above_below_gallery' );

		if ( ! in_array( $settings['download_all_position_albums'], $accepted_positions ) ) {
			return;
		}
		?>
		<div class="modula-download-all-button">
			<button class="modula-btn-<?php echo esc_attr( $settings['download_all_size_albums'] ); ?>">
				<a href="<?php echo esc_url( modula_download_req_url( $settings['gallery_id'] ) ); ?>"
				target="_self"
				class="modula-download-all">
					<?php
					if ( isset( $settings['download_all_gallery_button_icon_albums'] ) && '1' == $settings['download_all_gallery_button_icon_albums'] ) {
						echo modula_download_all_svg();
					}
					echo ! empty( $settings['download_all_label_albums'] ) ? esc_html( $settings['download_all_label_albums'] ) : esc_html__( 'Download All Images', 'modula-download' )
					?>
				</a>
			</button>
		</div>
		<?php
	}

	/**
	 * Download all shortcode CSS
	 *
	 * @param $id
	 * @param $settings
	 *
	 * @return string
	 */
	public function generate_download_css( $id, $settings ) {

		$css  = '<style>';
		$css .= '#jtg-' . $id . '-download.modula-download-all-button .modula-download-all svg {fill:' . Modula_Helper::sanitize_rgba_colour( $settings['download_all_gallery_icon_color'] ) . '}';
		$css .= "#jtg-{$id}-download.modula-download-all-button svg { display:inline-block;vertical-align:middle;}";
		$css .= '</style>';

		return $css;
	}

	public function generate_album_download_css( $id, $settings ) {
		$css  = '<style>';
		$css .= '#jtg-album-' . $id . '-download.modula-download-all-button .modula-download-all svg {fill:' . Modula_Helper::sanitize_rgba_colour( $settings['download_all_gallery_icon_color_albums'] ) . '; display:inline-block;vertical-align:middle;}';
		$css .= '</style>';

		return $css;
	}

	/**
	 * Add download settings to Modula migrator
	 *
	 * @param $modula_settings
	 * @param $guest_settings
	 * @param $source
	 *
	 * @return mixed
	 *
	 * @since 1.0.1
	 */
	public function modula_download_migrate_data( $modula_settings, $guest_settings, $source ) {

		if ( isset( $source ) ) {
			switch ( $source ) {
				case 'envira':
					if ( isset( $guest_settings['config']['download'] ) && 1 == $guest_settings['config']['download'] ) {
						$modula_settings['enable_download'] = 1;
					}

					if ( isset( $guest_settings['config']['download_image_size'] ) ) {
						if ( 'thumbnail' == $guest_settings['config']['download_image_size'] || 'medium' == $guest_settings['config']['download_image_size'] || 'large' == $guest_settings['config']['download_image_size'] || 'full' == $guest_settings['config']['download_image_size'] ) {
							$modula_settings['download_image_sizes'] = $guest_settings['config']['download_image_size'];
						}
					}

					if ( isset( $guest_settings['config']['download_all'] ) && 1 == $guest_settings['config']['download_all'] ) {
						$modula_settings['download_all_gallery_button'] = 1;
					}

					if ( isset( $guest_settings['config']['download_custom_name'] ) && 'envira-gallery' != $guest_settings['config']['download_custom_name'] ) {
						$modula_settings['custom_zip_name'] = $guest_settings['config']['download_custom_name'];
					}

					if ( isset( $guest_settings['config']['download_all_position'] ) ) {
						if ( 'above' == $guest_settings['config']['download_all_position'] ) {
							$modula_settings['download_all_position'] = 'above_gallery';
						} elseif ( 'below' == $guest_settings['config']['download_all_position'] ) {
							$modula_settings['download_all_position'] = 'below_gallery';
						} else {
							$modula_settings['download_all_position'] = 'above_below_gallery';
						}
					}

					if ( isset( $guest_settings['config']['download_all_label'] ) && '' != $guest_settings['config']['download_all_label'] ) {
						$modula_settings['download_all_label'] = $guest_settings['config']['download_all_label'];
					}

					break;
			}
		}

		return $modula_settings;
	}

	/**
	 * Loads the plugin textdomain for translation.
	 *
	 * @since 1.0.3
	 */
	public function set_locale() {
		load_plugin_textdomain( $this->plugin_slug, false, dirname( plugin_basename( MODULA_DOWNLOAD_FILE ) ) . '/languages/' );
	}

}
