<?php

/**
 * Helper function to zip gallery
 *
 * @param array  $files
 * @param string $location
 *
 * @return bool
 * @since 1.0.0
 *
 */
function modula_download_zip( $files, $location ) {
	if ( !extension_loaded( 'zip' ) ) {
		return false;
	}

	if ( file_exists( $location ) ) {
		unlink( $location );
	}

	// If empty or not array we return false
	if ( empty( $files ) || !is_array( $files ) ) {
		return false;
	}

	$zip = new ZipArchive();
	if ( !$zip->open( $location, ZIPARCHIVE::CREATE ) ) {
		return false;
	}

	foreach ( $files as $file ) {
		if ( file_exists( $file ) ) {
			$zip->addFile( $file, basename( $file ) );
		}
	}

	$zip->close();

	return true;
}

/**
 * Helper function to download file
 *
 * @param string $file_path
 * @param string $location
 *
 * @return void
 * @since 1.0.0
 *
 */
function modula_download_file( $file_path, $location ) {

	session_write_close();

	if ( function_exists( 'apache_setenv' ) ) {
		@apache_setenv( 'no-gzip', 1 );
	}
	@ini_set( 'zlib.output_compression', 'Off' );

	nocache_headers();
	header( 'Pragma: no-cache' );
	header( 'Expires: 0' );
	header( 'Robots: none' );
	header( 'Content-Type: application/zip' );
	header( 'Content-Description: File Transfer' );
	header( 'Content-Disposition: attachment; filename="' . basename( $location ) . '"' );
	header( 'Content-Transfer-Encoding: binary' );
	header( 'Content-Length: ' . filesize( $file_path ) );

	if ( stristr( getenv( 'SERVER_SOFTWARE' ), 'lighttpd' ) ) {

		header( "X-LIGHTTPD-send-file: $file_path" );

	} elseif ( stristr( getenv( 'SERVER_SOFTWARE' ), 'nginx' ) || stristr( getenv( 'SERVER_SOFTWARE' ), 'cherokee' ) ) {

		// We need a path relative to the domain.
		$file_path = (isset( $_SERVER['DOCUMENT_ROOT'] )) ? str_ireplace( realpath( sanitize_text_field( wp_unslash( $_SERVER['DOCUMENT_ROOT'] ) ) ), '', $file_path ) : false;
		header( "X-Accel-Redirect: /$file_path" );

	}

	modula_download_readfile_chunked( $file_path );
	exit();
}

/**
 * Helper Function to read file in chuncks
 *
 * @param string $file
 * @param bool   $retbytes
 *
 * @return bool
 * @since 1.0.0
 */
function modula_download_readfile_chunked( $file, $retbytes = true ) {

	$chunksize = 1024 * 1024;
	$buffer    = '';
	$cnt       = 0;
	$handle    = fopen( $file, 'r' );

	$size = filesize( $file );

	if ( $size ) {
		header( 'Content-Length: ' . $size );
	}

	if ( false === $handle ) {
		return false;
	}

	while ( !feof( $handle ) ) {
		$buffer = fread( $handle, $chunksize );
		echo $buffer;

		if ( $retbytes ) {
			$cnt += strlen( $buffer );
		}
	}

	$status = fclose( $handle );

	if ( $retbytes && $status ) {
		return $cnt;
	}

	return $status;
}

/**
 * Helper function to get request url
 *
 * @param string $id
 *
 * @return string $url
 * @since 1.0.0
 */
function modula_download_req_url( $id ) {
	$first_part  = isset( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : false;
	$second_part = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : false;



	$url = add_query_arg(
		array(
			'modula-id' => $id,
		), (is_ssl() ? 'https' : 'http') . '://' . $first_part . $second_part );

	return $url;
}

/**
 * Function to echo the download icon + filter to customize
 * 
 * @return string $svg
 * 
 * @since 1.0.0 
 */
function modula_download_all_svg() {
	
	return apply_filters ('modula_download_all_icon', '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
	<path d="M18.62 17.09V19H5.38v-1.91zm-2.97-6.96L17 11.45l-5 4.87-5-4.87 1.36-1.32 2.68 2.64V5h1.92v7.77z"></path> </svg>' );
}