<?php 

class Modula_Exif_Settings {

    /**
     * Holds the admin class object
     * 
     * @since 1.0.0
     * 
     * @var object
     */
    public static $instance;

    function __construct() {
        //Filter Modula Fields
        add_filter( 'modula_gallery_fields', array( $this, 'modula_exif_field'));
	    add_filter( 'modula_album_fields', array( $this, 'modula_exif_albums_field'));

        //Filter Modula Tab
        add_filter( 'modula_gallery_tabs', array( $this, 'modula_exif_tab'),99);
	    add_filter( 'modula_album_tabs', array( $this, 'modula_exif_album_tab'));

        // Template for EXIF
        add_action( 'admin_footer', array( $this, 'print_modula_exif_templates' ) );

        // Save Filters for our items
        add_filter( 'modula_gallery_image_attributes', array( $this, 'exif_add_item_fields' ) );

        // Add meta to image model
        add_filter( 'modula_image_properties', array( $this, 'modula_exif_add_metadata'));

    }

    /**
     * Returns the singleton instance of the class
     *
     * @return object Modula Exif Settings object
     *
     *  @since 1.0.0
     */
    public static function get_instance() {
        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Modula_Exif_Settings ) ) {
            self::$instance = new Modula_Exif_Settings();
        }

        return self::$instance;
    }

	/**
	 * Add the exif tab in settings
	 *
	 * @param $tabs
	 *
	 * @return mixed $tabs
	 *
	 * @since 1.0.0
	 */
    public function modula_exif_tab( $tabs ) {
        $tabs['exif'] = array(
            'label'       => esc_html__( "EXIF", "modula-exif" ),
            'title'       => esc_html__( "Exif Settings", "modula-exif" ),
            'description' => esc_html__( "The settings bellow adjust what type of picture information is displayed in the lightbox.", "modula-exif" ),
            'icon'        => "dashicons dashicons-camera-alt",
            'priority'    => 85
        );

	    if ( isset( $tabs['exif']['badge'] ) ) {
		    unset( $tabs['exif']['badge'] );
	    }

        return $tabs;
    }

	/**
	 * Add the exif tab in settings
	 *
	 * @param $tabs
	 *
	 * @return mixed $tabs
	 *
	 * @since 1.0.0
	 */
	public function modula_exif_album_tab( $tabs ) {
		$tabs['exif'] = array(
			'label'       => esc_html__( "EXIF", "modula-exif" ),
			'title'       => esc_html__( "Exif Settings", "modula-exif" ),
			'description' => esc_html__( "Choose to override or keep default options from galleries", "modula-exif" ),
			'icon'        => "dashicons dashicons-camera-alt",
			'priority'    => 85
		);
		return $tabs;
	}

	/**
	 * Add exif fields in settings
	 *
	 * @param $fields
	 *
	 * @return mixed $fields
	 *
	 * @since 1.0.0
	 */
    public function modula_exif_field( $fields ) {
        $fields['exif'] = array(
            'enable_exif'  => array(
                "name"        => esc_html__( 'Enable Exif', 'modula-exif' ),
                "type"        => "toggle",
                "default"     => 0,
                "description" => esc_html__( 'Enable this option to display the EXIF information in the lightbox.', 'modula-exif' ),
                "priority"    => 1,
                'children'    => array( 'exif_camera', 'exif_lens', 'exif_shutter_speed','exif_aperture', 'exif_focal_length', 'exif_iso', 'exif_date' ),
            ),
            'exif_camera'  => array(
                "name"        => esc_html__( 'Camera Model', 'modula-exif' ),
                "type"        => "toggle",
                "default"     => 1,
                "description" => esc_html__( 'Enable this option to display the camera information', 'modula-exif' ),
                "priority"    => 3,
                'is_child'    => true,
            ),
            'exif_lens'   => array(
                "name"        => esc_html__( 'Lens', 'modula-exif' ),
                "type"        => "toggle",
                "default"     => 1,
                "description" => esc_html__( 'Enable this option to display the lens information', 'modula-exif' ),
                "priority"    => 4,
                'is_child'    => true,
            ),
            'exif_shutter_speed'   => array(
                "name"        => esc_html__( 'Shutter Speed', 'modula-exif' ),
                "type"        => "toggle",
                "default"     => 1,
                "description" => esc_html__( 'Enable this option to display the shutter speed information', 'modula-exif' ),
                "priority"    => 5,
                'is_child'    => true,
            ),
            'exif_aperture'   => array(
                "name"        => esc_html__( 'Aperture', 'modula-exif' ),
                "type"        => "toggle",
                "default"     => 1,
                "description" => esc_html__( 'Enable this option to display the aperture information', 'modula-exif' ),
                "priority"    => 6,
                'is_child'    => true,
            ),
            'exif_focal_length'   => array(
                "name"        => esc_html__( 'Focal Length', 'modula-exif' ),
                "type"        => "toggle",
                "default"     => 1,
                "description" => esc_html__( 'Enable this option to display the focal length information', 'modula-exif' ),
                "priority"    => 7,
                'is_child'    => true,
            ),
            'exif_iso'   => array(
                "name"        => esc_html__( 'ISO', 'modula-exif' ),
                "type"        => "toggle",
                "default"     => 1,
                "description" => esc_html__( 'Enable this option to display the iso information', 'modula-exif' ),
                "priority"    => 8,
                'is_child'    => true,
            ),
            'exif_date'   => array(
                "name"        => esc_html__( 'Date', 'modula-exif' ),
                "type"        => "toggle",
                "default"     => 1,
                "description" => esc_html__( 'Enable this option to display the date information', 'modula-exif' ),
                "priority"    => 9,
                'is_child'    => true,
            ),
        );

        return $fields;
    }

	/**
	 * Add exif fields in settings
	 *
	 * @param $fields
	 *
	 * @return mixed $fields
	 *
	 * @since 1.0.0
	 */
    public function modula_exif_albums_field( $fields ){
	    $fields['exif'] = array(
		    'displayExif' => array(
			    "name"        => esc_html__( 'Show EXIF data', 'modula-exif' ),
			    "type"        => "select",
			    "values"      => array(
				    "default" => esc_html__( 'Take Gallery setting','modula-exif' ),
				    "on"      => esc_html__( 'Display EXIF data', 'modula-exif' ),
				    "off"     => esc_html__( 'Hide EXIF data', 'modula-exif' ),
			    ),
			    "default"     => 'default',
			    "description" => esc_html__( 'Choose to keep or override the EXIF data settings from galleries.', 'modula-exif' ),
			    "priority"    => 1,
		    ),
	    );

	    return $fields;
    }

    /**
     * EXIF information in edit image 
     * 
     * @since 1.0.0
     * 
     */
    public function print_modula_exif_templates() {
        ?>
        <script type="text/html" id="tmpl-modula-exif">

                <span class="name"><?php _e( 'Exif Information', 'modula-exif' ); ?></span>

                <label class="exif">
	                <span> <?php echo Modula_Exif::modula_exif_get_icons( 'camera' ) ?> </span>
	                <p> <?php esc_html_e( 'Camera Model', 'modula_exif' ); ?> </p>
	                <input type="text" name="exif_camera" value="{{ data.exif_camera }}" placeholder="<?php echo esc_html('Ex.: SM-j105H'); ?>"/>
                </label>
                <label class="exif">
                    <span> <?php echo Modula_Exif::modula_exif_get_icons('lens') ?> </span>
                    <p> <?php esc_html_e( 'Lens', 'modula_exif' ); ?> </p>
                    <input type="text" name="exif_lens" value="{{ data.exif_lens }}" placeholder="<?php echo esc_html('Ex.: 85mm f/1.4'); ?>"/>
                </label>
                <label class="exif">
                    <span> <?php echo Modula_Exif::modula_exif_get_icons('eye') ?> </span>
                    <p> <?php esc_html_e( 'Focal Length', 'modula_exif' ); ?> </p>
                    <input type="text" name="exif_focal_length" value="{{ data.exif_focal_length }}" placeholder="<?php echo esc_html('Ex.: 3mm'); ?>" />
                </label>
                <label class="exif">
                    <span> <?php echo Modula_Exif::modula_exif_get_icons('shutter_speed') ?> </span>
                    <p> <?php esc_html_e( 'Shutter Speed', 'modula_exif' ); ?> </p>
                    <input type="text" name="exif_shutter_speed" value="{{ data.exif_shutter_speed }}" placeholder="<?php echo esc_html('Ex.: 1/25'); ?>" />
                </label>
                <label class="exif">
                    <span> <?php echo Modula_Exif::modula_exif_get_icons('aperture') ?> </span>
                    <p> <?php esc_html_e( 'Aperture', 'modula_exif' ); ?> </p>
                    <input type="text" name="exif_aperture" value="{{ data.exif_aperture }}" placeholder="<?php echo esc_html('Ex.: f/2.2'); ?>" />
                </label>
                <label class="exif">
                    <span> <?php echo Modula_Exif::modula_exif_get_icons('iso') ?> </span>
                    <p> <?php esc_html_e( 'Iso', 'modula_exif' ); ?> </p>
                    <input type="text" name="exif_iso" value="{{ data.exif_iso }}" placeholder="<?php echo esc_html('Ex.: ISO 125'); ?>" />
                </label>
                <label class="exif">
                    <span> <?php echo Modula_Exif::modula_exif_get_icons('calendar') ?> </span>
                    <p> <?php esc_html_e( 'Date', 'modula_exif' ); ?> </p>
                    <input type="text" name="exif_date" value="{{ data.exif_date }}" placeholder="<?php echo date('Y-m-d H:i'); ?>"/>
                </label>
		</script>
		<?php
    }


	/**
	 * Add exif information to edit modal
	 *
	 * @param $fields
	 *
	 * @return mixed $fields
	 *
	 * @since 1.0.0
	 */
    public function exif_add_item_fields( $fields ) {
        $fields[] = 'exif_camera';
        $fields[] = 'exif_lens';
        $fields[] = 'exif_focal_length';
        $fields[] = 'exif_shutter_speed';
        $fields[] = 'exif_aperture';
        $fields[] = 'exif_iso';
        $fields[] = 'exif_date';
		return $fields;
    }

	/**
	 * Add exif meta to image model
	 *
	 * @param $image
	 *
	 * @return mixed $image
	 *
	 * @since 1.0.0
	 */
    public function modula_exif_add_metadata( $image ) {

        if( !extension_loaded( 'exif' ) ){
            return $image;
        }

        $meta = wp_get_attachment_metadata( $image['id'] );
        $file = get_attached_file( $image['id'] );
        $pp = pathinfo( $file );
        $meta['image_meta']['lens'] = "";
        if ( in_array( strtolower( $pp['extension'] ), array( 'jpg', 'jpeg', 'tiff' ) ) ) {
            $exif = @exif_read_data( $file );
            if ( $exif && isset( $exif['UndefinedTag:0xA434'] ) )
            $meta['image_meta']['lens'] = empty( $exif['UndefinedTag:0xA434'] ) ? "" : $exif['UndefinedTag:0xA434'];
        }
        wp_update_attachment_metadata( $image['id'], $meta );

	    if ( isset( $image['exif_camera'] ) && '' == $image['exif_camera'] ) {
		    $image['exif_camera'] = modula_format_exif_data( 'camera', $meta['image_meta']['camera'] );
	    }

	    if ( isset( $image['exif_lens'] ) && empty( $image['exif_lens'] ) ) {
		    $image['exif_lens'] = modula_format_exif_data( 'lens', $meta['image_meta']['lens'] );
	    }

	    if ( isset( $image['exif_focal_length'] ) && '' == $image['exif_focal_length'] ) {
		    $image['exif_focal_length'] = modula_format_exif_data( 'focal_length', $meta['image_meta']['focal_length'] );
	    }

	    if ( isset( $image['exif_shutter_speed'] ) && '' == $image['exif_shutter_speed'] ) {
		    $image['exif_shutter_speed'] = modula_format_exif_data( 'shutter_speed', $meta['image_meta']['shutter_speed'] );
	    }

	    if ( isset( $image['exif_aperture'] ) && '' == $image['exif_aperture'] ) {
		    $image['exif_aperture'] = modula_format_exif_data( 'aperture', $meta['image_meta']['aperture'] );
	    }

	    if ( isset( $image['exif_iso'] ) && '' == $image['exif_iso'] ) {
		    $image['exif_iso'] = modula_format_exif_data( 'iso', $meta['image_meta']['iso'] );
	    }

	    if ( isset( $image['exif_date'] ) ) {
		    if ( empty( $image['exif_date'] ) && !empty( $meta['image_meta']['created_timestamp'] ) ) {
			    $image['exif_date'] = date( 'Y-m-d H:i', $meta['image_meta']['created_timestamp'] );
		    } else if ( 'N/A' == $image['exif_date'] || empty( $image['exif_date'] ) ) {
			    $image['exif_date'] = sanitize_text_field( 'N/A' );
		    }
	    }

        return $image;
    }

}

$modula_exif_settings = Modula_Exif_Settings::get_instance();