<?php
// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}
/**
 *
 */
class Modula_Albums_Password_Protect {

    /**
     * Holds the class object.
     *
     * @since 1.0.0
     *
     * @var object
     */
    public static $instance;

    /**
     * Primary class constructor.
     *
     * @since 1.0.2
     */
    public function __construct() {

        // Set admin requirements on admin_init so that we won't trigger get_packages from WPChill_Upsells on frontend
        add_action( 'admin_init', array( $this, 'set_admin_albums_requirements' ), 25 );
       
        // Add protect_textarea field type for Albums
        add_filter( 'modula_render_protect_textarea_field_type', array( $this, 'textarea_field_type' ), 10, 5 );

        // Add another format for our textarea for Albums
        add_filter( 'modula_field_type_protect_textarea_format', array( $this, 'password_protect_textarea_format' ), 15, 10 );

        // Load conditional js for Albums
	    add_action( 'modula_album_scripts_after_wp_modula', array( $this, 'admin_scripts' ), 25 );
	    add_action( 'modula_defaults_album_scripts_after_wp_modula', array( $this, 'admin_scripts' ), 25 );

	    add_filter( 'modula_album_default_settings', array( $this,'album_password_protect_defaults' ) );

        // Alter Save
        /* Fire our meta box setup function on the post editor screen. */
        add_action( 'load-post.php', array( $this, 'save_password' ) );
        add_action( 'load-post-new.php', array( $this, 'save_password' ) );

        // Require the password protection module
        require_once MODULA_PASSWORD_PROTECT_PATH.'includes/albums-password-protect/class-modula-albums-password-protect-module.php';

    }

    /**
	 * Set admin requirements and check for license.
	 *
	 * @return void
	 * @since 1.0.8
	 */
    public function set_admin_albums_requirements() {

        if ( class_exists( 'WPChill_Upsells' ) ) {

			$args           = apply_filters(
				'modula_upsells_args',
				array(
					'shop_url' => 'https://wp-modula.com',
					'slug'     => 'modula',
				)
			);
			$wpchill_upsell = WPChill_Upsells::get_instance( $args );

			if ( $wpchill_upsell && ! $wpchill_upsell->is_upgradable_addon( 'modula-password-protect' ) ) {
				 // Filter Modula Password Protect Tab for Albums
                 add_filter( 'modula_album_tabs', array( $this, 'modula_password_protect_tabs' ) );

                 // Filter Modula Password Protect Fields for Albums
                 add_filter( 'modula_album_fields', array( $this, 'modula_password_protect_fields' ) );

			}
		}
    }


    /**
     * Returns the singleton instance of the class.
     *
     * @since 1.0.2
     *
     * @return object The Modula_Password_Protect object.
     */
    public static function get_instance() {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Modula_Albums_Password_Protect ) ) {
            self::$instance = new Modula_Albums_Password_Protect();
        }

        return self::$instance;

    }


    /**
     * Add Password Protect Tab
     *
     * @param $tabs
     *
     * @return mixed
     */
    public function modula_password_protect_tabs( $tabs ) {

        if ( ! isset( $tabs['password_protect'] ) ) {
            $tabs['password_protect'] = array(
                'label'    => esc_html__( 'Password Protect', 'modula-password-protect' ),
                "icon"     => "dashicons dashicons-shield",
                'priority' => 100,
            );
        }

        return $tabs;
    }


    /**
     * Add Password Protect Fields
     *
     * @param $fields
     *
     * @return mixed
     */
    public function modula_password_protect_fields( $fields ) {
        if(class_exists('Modula_Albums') ) {
            $fields['password_protect'] = array(
                "enable_password"           => array(
                    "name"     => esc_html__('Enable Password Protect', 'modula-password-protect'),
                    "type"     => "toggle",
                    "default"  => 0,
                    'priority' => 10,
                    'children' => array( 'password', 'password_protect_username', 'password_protect_text' ),
                ),
                'password'                  => array(
                    "name"        => esc_html__('Password', 'modula-password-protect'),
                    "type"        => "text",
                    'description' => __('', 'modula-password-protect'),
                    'priority'    => 30,
                ),
                'password_protect_username' => array(
                    "name"        => esc_html__('Username or Email address', 'modula-password-protect'),
                    "type"        => "text",
                    'description' => __('This is an optional step. If this field is defined, upon access to Modula album this will also be required.', 'modula-password-protect'),
                    'priority'    => 30,
                ),
                'password_protect_text'     => array(
                    "name"        => esc_html__('Message', 'modula-password-protect'),
                    "type"        => "protect_textarea",
                    'description' => __('Text shown on front end, informing the visitors that a password is required for them to view the gallery.', 'modula-password-protect'),
                    'priority'    => 40,
                ),
            );
        }

        return $fields;
    }

    /***
     * Add protect_textarea field type
     *
     * @param $html
     * @param $field
     * @param $value
     *
     * @return string
     */
    public function textarea_field_type( $html, $field, $value ) {

        $html = '<div class="modula-textarea">';
        $html .= '<textarea data-setting="' . esc_attr( $field['id'] ) . '" name="modula-settings[' . esc_attr( $field['id'] ) . ']" id="modula-' . esc_attr( $field['id'] ) . '" rows="10" cols="50">' . wp_kses_post( $value ) . '</textarea>';
        $html .= '</div>';

        return $html;

    }

    /**
     * Custom format for protect_textarea field type
     *
     * @param $format
     * @param $field
     *
     * @return string
     */
    public function password_protect_textarea_format( $format, $field ) {

        $format = '<tr data-container="' . esc_attr( $field['id'] ) . '"><th scope="row"><label class="th-label">%s</label>%s</th><td>%s</td></tr>';

        return $format;
    }

    public function save_password(){

        /* Save post meta on the 'save_post' hook. */
        add_action( 'save_post', array( $this, 'save_extra_fields' ), 50, 2 );

    }

    public function save_extra_fields( $post_id, $post ){

        /* Get the post type object. */
        $post_type = get_post_type_object( $post->post_type );
        /* Check if the current user has permission to edit the post. */
        if ( !current_user_can( $post_type->cap->edit_post, $post_id ) ) {
            return $post_id;
        }

        if ( 'modula-album' != get_post_type( $post ) ) {
            return $post_id;
        }

        global $wpdb;

        if ( ! isset( $_POST['modula-settings']['enable_password'] ) && '' == $post->post_password ) {
            return $post_id;
        }elseif ( ! isset( $_POST['modula-settings']['enable_password'] ) && '' != $post->post_password ) {
            $wpdb->update( $wpdb->posts, array( 'post_password' => '' ), array( 'ID' => $post_id ) );
        }elseif ( isset( $_POST['modula-settings']['password'] ) && '' != $_POST['modula-settings']['password'] ) {
            $args = array(
                'post_password' => sanitize_text_field( $_POST['modula-settings']['password'] ),
                'post_status'   => 'publish',
            );
            $wpdb->update( $wpdb->posts, $args, array( 'ID' => $post_id ) );
        }

    }

	/* Enqueue Admin Scripts */
	public function admin_scripts( $hook ) {

		wp_enqueue_script( 'modula-albums-password-protect-conditions', MODULA_PASSWORD_PROTECT_URL . 'assets/js/wp-modula-albums-password-conditional.js', array(), MODULA_PASSWORD_PROTECT_VERSION, true );
	}

	/**
	 * Add albums defaults
	 *
	 * @param $defaults
	 *
	 * @return mixed
	 *
	 * @since 1.0.4
	 */
	public function album_password_protect_defaults($defaults){
		$defaults['enable_password ']           = 0;
		$defaults['password ']                  = '';
		$defaults['password_protect_username '] = '';
		$defaults['password_protect_text ']     = '';

		return $defaults;
	}
}