<?php
// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 *
 * @since 1.0.0
 *
 * @package Modula_Password_Protect
 * @author  WPChill
 */
class Modula_Password_Protect_Module {

	/**
	 * Holds the class object.
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public static $instance;

	/**
	 * Primary class constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		add_action( 'login_form_postpass', array( $this, 'username_validation' ) );
		add_filter( 'modula_pre_output_filter_check', array( $this, 'password_protect_gallery_check' ), 10, 3 );
		add_filter( 'modula_pre_output_filter', array( $this, 'password_protect_gallery' ), 10, 3 );
		add_filter( 'the_password_form', array( $this, 'modula_password_form' ), 10, 2 );
		add_filter( 'the_password_form', array( $this, 'modula_password_form_message' ), 30, 2 );
		add_filter( 'post_password_expires', array( $this, 'modula_gallery_password_expires' ) );

	}

	/**
	 * Customize cookie expire time for password.
	 *
	 * @since 1.0.0
	 *
	 * @param int $expires The expiry time, as passed to setcookie().
	 *
	 * @return  mixed|void
	 */
	public function modula_gallery_password_expires( $expires ) {
		// default is 10 days.
		$expires = apply_filters( 'modula_password_protect_time_limit', time() + 864000, false, false );

		return $expires;
	}

	/**
	 * Customize message for password.
	 *
	 * @since 1.0.0
	 */
	public function change_password_message_content( $output ) {

		// Grab the gallery ID from the form.
		$start     = 'id="pwbox-';
		$end       = '"';
		$start_pos = strpos( $output, $start ) + strlen( $start );
		$end_pos   = strpos( $output, $end, $start_pos );
		$id        = substr( $output, $start_pos, ( $end_pos - $start_pos ) );


		// Check we got a valid ID.
		if ( ! is_numeric( $id ) ) {
			return $output;
		}

		// Check if post is a Modula Gallery Post.
		global $post;

		if ( ! in_array( $post->post_type, array( 'modula-gallery' ) ) ) {
			return $output;
		}

        $settings = get_post_meta( $post->ID, 'modula-settings', true );


		$message = ( isset( $settings['password_protect_text'] ) && ! empty( $settings['password_protect_text'] ) ) ? $settings['password_protect_text'] : false;

		$message = apply_filters( 'modula_password_protect_message', $message, $post, $settings );

		if ( ! empty( $post->post_password ) && post_password_required( $post ) === false ) {

			$output = '
			<form action="' . get_option( 'siteurl' ) . '/wp-pass.php" method="post">
			  ' . $message . '
			    <label for="post_password">' . esc_html__( 'Password', 'modula-password-protect' ) . ':</label>
			    <input name="post_password" class="input" type="password" size="20" />
			    <input type="submit" name="Submit" class="button" value="' . esc_html__( 'Submit', 'modula-password-protect' ) . '" />
			</form>
			';

			return $output;

		} else {

			return $output;
		}

	}


	/**
	 * Checks if the given Modula Gallery requires a username as part of the validation
	 * process.
	 * If validation passes, we store the username in a cookie, and continue.
	 *
	 * @since 1.0.0
	 */
	public function username_validation() {

		// Check if Post ID and Username are specified in the POST request.
		if ( ! isset( $_POST['post_ID'] ) || ! isset( $_POST['password_protect_username'] ) ) {
			return;
		}

		$id               = absint( $_POST['post_ID'] );
		$default_settings = Modula_CPT_Fields_Helper::get_defaults();
		$modula_settings  = wp_parse_args( get_post_meta( $id, 'modula-settings', true ), $default_settings );

		// Check if the email address / username matches the one from the gallery.
		$email = $modula_settings['password_protect_username'];

		if ( isset( $email ) && ! empty( $email ) ) {
			if ( $email !== $_POST['password_protect_username'] ) {
				// Username doesn't match so redirect to referring page.
				wp_redirect( $_SERVER['HTTP_REFERER'] );
                die();
			}

			// If here, username matches.
			// Set cookie.
			$time_limit = apply_filters( 'modula_password_protect_time_limit', time() + 864000, $id, $modula_settings );
			$expire     = time() + intval( $time_limit );

			setcookie( 'modula_password_protect_email_' . $id, $email, intval( $expire ) );
		}
	}

	/**
	 * Password protect a gallery, if password protection is enabled
	 *
	 * @since 1.0.0
	 *
	 * @param array $modula_settings Gallery Data
	 * @param int $id Modula Gallery ID
	 *
	 * @return mixed false (if password required) or Gallery Form
	 */
	public function password_protect_gallery( $pre_output,$modula_settings, $gallery ) {

		// Assume username is valid
		$username_exists = true;

		// Check the email address for the Modula Gallery is set, and matches the cookie.
		$email = $modula_settings['password_protect_username'];

		if ( isset( $email ) && ! empty( $email ) ) {
			// Check cookie.
			if ( ! isset( $_COOKIE[ 'modula_password_protect_email_' . $gallery->ID ] ) ||
			     $_COOKIE[ 'modula_password_protect_email_' . $gallery->ID ] !== $email ) {

				// No cookie, or cookie exists and doesn't match username required.
				$username_exists = false;
			}
		}

		// Check if Post is password protected, and if so whether the password has been provided.
		if ( $username_exists && post_password_required( $gallery ) === false ) {
			return false;
		}

		// check current post type so that if standalone is active the password form won't be displayed twice
		if('modula-gallery' != get_post_type()){
            // Assign password form to variable.
            $pre_output = get_the_password_form( $gallery );
        }

		return $pre_output;

	}

    public function password_protect_gallery_check( $return,$modula_settings, $gallery ) {
		
        // Only do this on the frontend.
        if ( is_admin() ) {
            return false;
        }

        // Return if we couldn't get the Modula Gallery
        if ( ! $gallery ) {
            return false;
        }

        // Check if Post is password protected, and if so whether the password has been provided.
        if ( post_password_required( $gallery ) === false && $this->post_username_validation( $gallery ) ) {
	
            return false;
        }
        return true;

    }

	/**
	 * Checks if the username is required and if the username in cookie is still valid.
	 *
	 * @since 1.0.9
	 *
	 * @param object $gallery post_object
	 *
	 * @return bool
	 */
	public function post_username_validation( $gallery ){
		// Asume username is valid
		$username_exists = true;

		$modula_settings = get_post_meta( $gallery->ID, 'modula-settings', true );
		$email = $modula_settings['password_protect_username'];

		if ( isset( $email ) && ! empty( $email ) ) {
			// Check cookie.
			if ( ! isset( $_COOKIE[ 'modula_password_protect_email_' . $gallery->ID ] ) ||
					$_COOKIE[ 'modula_password_protect_email_' . $gallery->ID ] !== $email ) {

				// No cookie, or cookie exists and doesn't match username required.
				$username_exists = false;
			}
		}

		return $username_exists;
	}


	/**
	 * Checks if the given form belongs to an Modula Gallery, and display a custom message if Modula Gallery has one.
	 *
	 * @since 1.0.0
	 *
	 * @param string $output Output
	 *
	 * @return string Output
	 */
	public function modula_password_form_message( $output, $post = false) {

		if( ! $post ){
			// Backwards compatibility wp <5.8. There's no function var or public exposure, so grab the ID from the form
			$start     = 'id="pwbox-';
			$end       = '"';
			$start_pos = strpos( $output, $start ) + strlen( $start );
			$end_pos   = strpos( $output, $end, $start_pos );
			$id        = substr( $output, $start_pos, ( $end_pos - $start_pos ) );

			// Check we got a valid ID
			if ( ! is_numeric( $id ) ) {
				return $output;
			}

			$post = get_post( $id );
		}

		$modula_settings = get_post_meta( $post->ID, 'modula-settings', true );
		if ( ! in_array( $post->post_type, array( 'modula-gallery' ), true ) ) {
			return $output;
		}

		$message = ( ! empty( $modula_settings['password_protect_text'] ) ) ? $modula_settings['password_protect_text'] : false;
		$message = apply_filters( 'modula_password_protect_message', $message, $post, $modula_settings );

		if ( $message ) {
			
			$output = str_replace( 'This content is password protected. To view it please enter your password below:', $message, $output );

			$output = str_replace( 'This content is password protected. Please enter a password to view.', $message, $output );

			return $output;
		} else {
			return $output;
		}

	}

	/**
	 * Checks if the given form belongs to an Modula Gallery POST, and if that gallery
	 * requires a username as well, prepends the form with a username field
	 *
	 * @since 1.0.0
	 *
	 * @param string $output Output
	 *
	 * @return string Output
	 */
	public function modula_password_form( $output, $post = false ) {

		if( ! $post ){
			// Backwards compatibility wp <5.8. There's no function var or public exposure, so grab the ID from the form
			$start     = 'id="pwbox-';
			$end       = '"';
			$start_pos = strpos( $output, $start ) + strlen( $start );
			$end_pos   = strpos( $output, $end, $start_pos );
			$id        = substr( $output, $start_pos, ( $end_pos - $start_pos ) );

			// Check we got a valid ID
			if ( ! is_numeric( $id ) ) {
				return $output;
			}

			$post = get_post( $id );
		}else{
			$id = $post->ID;
		}

		if ( ! in_array( $post->post_type, array( 'modula-gallery' ), true ) ) {
			return $output;
		}

		$modula_settings = get_post_meta( $id, 'modula-settings', true );

		$username       = ( isset( $_COOKIE[ 'modula_password_protect_email_' . $id ] ) ? htmlspecialchars( $_COOKIE[ 'modula_password_protect_email_' . $id ] ) : '' );
		$username_field = '<p><label for="username-' . esc_attr( $id ) . '">' . esc_html__( 'Username', 'modula-password-protect' ) . ': <input type="text" name="password_protect_username" id="username-' . esc_attr( $id ) . '" value="" /></label></p>';

		$email = $modula_settings['password_protect_username'];

		if ( isset( $email ) && ! empty( $email ) ) {
			$output = str_replace( "</p>\n", $username_field, $output );
			$output = str_replace( '</form>', '<input type="hidden" name="post_ID" value="' . esc_attr( $id ) . '" /></form>', $output );

		}

		return $output;

	}

	/**
	 * Returns the singleton instance of the class.
	 *
	 * @since 1.0.0
	 *
	 * @return object The Modula_Password_Protect_Module object.
	 */
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Modula_Password_Protect_Module ) ) {
			self::$instance = new Modula_Password_Protect_Module();
		}

		return self::$instance;

	}

}

// Load the shortcode class.
$modula_password_protect_module = Modula_Password_Protect_Module::get_instance();
