<?php
// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
/**
 *
 */
class Modula_Password_Protect {

	/**
	 * Holds the class object.
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public static $instance;

	/**
	 * The name of the plugin.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $plugin_name = 'Modula Password Protect';

	/**
	 * Unique plugin slug identifier.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $plugin_slug = 'modula-password-protect';

	/**
	 * Primary class constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		// Set admin requirements on admin_init so that we won't trigger get_packages from WPChill_Upsells on frontend
		add_action( 'admin_init', array( $this, 'set_admin_requirements' ), 25 );

		// Load the plugin textdomain.
		add_action( 'plugins_loaded', array( $this, 'load_plugin_textdomain' ) );

		// Add protect_textarea field type
		add_filter( 'modula_render_protect_textarea_field_type', array( $this, 'textarea_field_type' ), 10, 5 );

		// Add another format for our textarea
		add_filter( 'modula_field_type_protect_textarea_format', array( $this, 'password_protect_textarea_format' ), 15, 10 );

		// Load the plugin.
		$this->init();

		// Require the password protection module
		require_once MODULA_PASSWORD_PROTECT_PATH . 'includes/class-modula-password-protect-module.php';

		register_activation_hook( MODULA_PASSWORD_PROTECT_FILE, array( $this, 'plugin_activation' ) );

	}

	/**
	 * Set admin requirements and check for license.
	 *
	 * @return void
	 * @since 1.0.8
	 */
	public function set_admin_requirements() {

		if ( class_exists( 'WPChill_Upsells' ) ) {

			$args           = apply_filters(
				'modula_upsells_args',
				array(
					'shop_url' => 'https://wp-modula.com',
					'slug'     => 'modula',
				)
			);
			$wpchill_upsell = WPChill_Upsells::get_instance( $args );

			if ( $wpchill_upsell && ! $wpchill_upsell->is_upgradable_addon( 'modula-password-protect' ) ) {
				// Filter Modula Password Protect Tab
				add_filter( 'modula_gallery_tabs', array( $this, 'modula_password_protect_tabs' ), 99 );

				// Filter Modula Password Protect Fields
				add_filter( 'modula_gallery_fields', array( $this, 'modula_password_protect_fields' ) );

				// Filter Modula Password Protect Default Field Values
				add_filter( 'modula_lite_default_settings', array( $this, 'modula_password_protect_defaults' ) );
				// Load conditional js
				add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ), 25 );

				// Alter Save
				/* Fire our meta box setup function on the post editor screen. */
				add_action( 'load-post.php', array( $this, 'save_password' ) );
				add_action( 'load-post-new.php', array( $this, 'save_password' ) );
			}
		}
	}

	/**
	 * Prevent plugin activation if Modula Pro is not installed and activated
	 *
	 * @since 1.0.3
	 */
	public function plugin_activation() {
		if ( ! class_exists( 'Modula_PRO' ) ) {
			deactivate_plugins( plugin_basename( MODULA_PASSWORD_PROTECT_FILE ) );
			wp_die( __( 'Please install and activate Modula Pro before using Modula Password Protect add-on.', 'modula-password-protect' ) );

		}

	}

	/**
	 * Loads the plugin textdomain for translation.
	 *
	 * @since 1.0.0
	 */
	public function load_plugin_textdomain() {
		load_plugin_textdomain( $this->plugin_slug, false, MODULA_PASSWORD_PROTECT_PATH . '/languages/' );
	}

	/**
	 * Loads the plugin into WordPress.
	 *
	 * @since 1.0.0
	 */
	public function init() {

		// Load admin only components.
		if ( is_admin() ) {
			add_action( 'modula_pro_updater', array( $this, 'addon_updater' ), 15, 2 );
		}

	}

	public function addon_updater( $license_key, $store_url ) {

		if ( class_exists( 'Modula_Pro_Base_Updater' ) ) {
			$modula_addon_updater = new Modula_Pro_Base_Updater(
				$store_url,
				MODULA_PASSWORD_PROTECT_FILE,
				array(
					'version' => MODULA_PASSWORD_PROTECT_VERSION,        // current version number
					'license' => $license_key,               // license key (used get_option above to retrieve from DB)
					'item_id' => 316395,                      // ID of the product
					'author'  => 'MachoThemes',            // author of this plugin
					'beta'    => false,
				)
			);
		}
	}


	/**
	 * Returns the singleton instance of the class.
	 *
	 * @since 1.0.0
	 *
	 * @return object The Modula_Password_Protect object.
	 */
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Modula_Password_Protect ) ) {
			self::$instance = new Modula_Password_Protect();
		}

		return self::$instance;

	}

	/**
	 * Add Password Protect Tab
	 *
	 * @param $tabs
	 *
	 * @return mixed
	 */
	public function modula_password_protect_tabs( $tabs ) {

		$tabs['password_protect'] = array(
			'label'    => esc_html__( 'Pass Protect', 'modula-password-protect' ),
			'title'    => esc_html__( 'Password Protect', 'modula-password-protect' ),
			'icon'     => 'dashicons dashicons-shield',
			'priority' => 120,
		);

		unset( $tabs['password_protect']['badge'] );

		return $tabs;

	}

	/**
	 * Add Password Protect defaults settings
	 *
	 * @param $defaults
	 *
	 * @return mixed $defaults;
	 *
	 * @since 1.0.8
	 */
	public function modula_password_protect_defaults( $defaults ) {
		$defaults['enable_password']    = 0;
		$defaults['password ']                  = '';
		$defaults['password_protect_username '] = '';
		$defaults['password_protect_text ']     = '';
		return $defaults;
	}


	/**
	 * Add Password Protect Fields
	 *
	 * @param $fields
	 *
	 * @return mixed
	 */
	public function modula_password_protect_fields( $fields ) {

		$fields['password_protect'] = array(
			'enable_password'           => array(
				'name'     => esc_html__( 'Enable Password Protect', 'modula-password-protect' ),
				'type'     => 'toggle',
				'default'  => 0,
				'priority' => 10,
				'children' => array( 'password', 'password_protect_username', 'password_protect_text' ),
			),
			'password'                  => array(
				'name'        => esc_html__( 'Password', 'modula-password-protect' ),
				'type'        => 'text',
				'description' => __( '', 'modula-password-protect' ),
				'priority'    => 30,
			),
			'password_protect_username' => array(
				'name'        => esc_html__( 'Username or Email address', 'modula-password-protect' ),
				'type'        => 'text',
				'description' => __( 'This is an optional step. If this field is defined, upon access to Modula gallery this will also be required.', 'modula-password-protect' ),
				'priority'    => 30,
			),
			'password_protect_text'     => array(
				'name'        => esc_html__( 'Message', 'modula-password-protect' ),
				'type'        => 'protect_textarea',
				'description' => __( 'Text shown on front end, informing the visitors that a password is required for them to view the gallery.', 'modula-password-protect' ),
				'priority'    => 40,
			),
		);

		return $fields;
	}

	/***
	 * Add protect_textarea field type
	 *
	 * @param $html
	 * @param $field
	 * @param $value
	 *
	 * @return string
	 */
	public function textarea_field_type( $html, $field, $value ) {

		$html  = '<div class="modula-textarea">';
		$html .= '<textarea data-setting="' . esc_attr( $field['id'] ) . '" name="modula-settings[' . esc_attr( $field['id'] ) . ']" id="modula-' . esc_attr( $field['id'] ) . '" rows="10" cols="50">' . wp_kses_post( $value ) . '</textarea>';
		$html .= '</div>';

		return $html;

	}

	/**
	 * Custom format for protect_textarea field type
	 *
	 * @param $format
	 * @param $field
	 *
	 * @return string
	 */
	public function password_protect_textarea_format( $format, $field ) {

		$format = '<tr data-container="' . esc_attr( $field['id'] ) . '"><th scope="row"><label class="th-label">%s</label>%s</th><td>%s</td></tr>';

		return $format;
	}

	public function save_password() {

		/* Save post meta on the 'save_post' hook. */
		add_action( 'save_post', array( $this, 'save_extra_fields' ), 50, 2 );

	}

	public function save_extra_fields( $post_id, $post ) {

		/* Get the post type object. */
		$post_type = get_post_type_object( $post->post_type );
		/* Check if the current user has permission to edit the post. */
		if ( ! current_user_can( $post_type->cap->edit_post, $post_id ) ) {
			return $post_id;
		}

		if ( 'modula-gallery' != get_post_type( $post ) ) {
			return $post_id;
		}

		global $wpdb;

		if ( ! isset( $_POST['modula-settings']['enable_password'] ) && '' == $post->post_password ) {
			return $post_id;
		} elseif ( ! isset( $_POST['modula-settings']['enable_password'] ) && '' != $post->post_password ) {
			$wpdb->update( $wpdb->posts, array( 'post_password' => '' ), array( 'ID' => $post_id ) );
		} elseif ( isset( $_POST['modula-settings']['password'] ) && '' != $_POST['modula-settings']['password'] ) {
			$args = array(
				'post_password' => sanitize_text_field( $_POST['modula-settings']['password'] ),
				'post_status'   => 'publish',
			);
			$wpdb->update( $wpdb->posts, $args, array( 'ID' => $post_id ) );
		}

	}

	/* Enqueue Admin Scripts */
	public function admin_scripts( $hook ) {
		global $id, $post;
		// Get current screen.
		$screen = get_current_screen();
		// Check if is modula custom post type
		if ( 'modula-gallery' !== $screen->post_type ) {
			return;
		}
		// Set the post_id
		$post_id = isset( $post->ID ) ? $post->ID : (int) $id;
		if ( 'post-new.php' == $hook || 'post.php' == $hook ) {
			/* CPT Styles & Scripts */

			wp_enqueue_script( 'modula-password-protect-conditions', MODULA_PASSWORD_PROTECT_URL . 'assets/js/wp-modula-password-conditional.js', array(), MODULA_PASSWORD_PROTECT_VERSION, true );

		}
	}

}
