<?php
// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 *
 */
class Modula_Albums_Slider {

	/**
	 * Holds the class object.
	 *
	 * @since 1.1.2
	 *
	 * @var object
	 */
	public static $instance;

	/**
	 * The name of the plugin.
	 *
	 * @since 1.1.2
	 *
	 * @var string
	 */
	public $plugin_name = 'Modula Slider';

	/**
	 * Unique plugin slug identifier.
	 *
	 * @since 1.1.2
	 *
	 * @var string
	 */
	public $plugin_slug = 'modula-slider';

	/**
	 * Primary class constructor.
	 *
	 * @since 1.1.2
	 */
	private function __construct() {

		// Filter Modula slider Scripts & Styles.
		add_filter( 'modula_album_necessary_scripts', array( $this, 'modula_slider_scripts' ), 15, 2 );
		add_filter( 'modula_album_necessary_styles', array( $this, 'modula_slider_styles' ), 15, 2 );

		// Set admin requirements on admin_init so that we won't trigger get_packages from WPChill_Upsells on frontend.
		add_action( 'admin_init', array( $this, 'set_admin_requirements' ), 25 );

		// Add defaults.
		add_filter( 'modula_album_lite_default_settings', array( $this, 'set_defaults' ) );

		add_filter( 'modula_album_template_data', array( $this, 'modula_album_gallery_classes' ), 40 );

		// Model output css.
		add_filter( 'modula_album_shortcode_css', array( $this, 'modula_album_slider_output_css' ), 17, 3 );

		// Add slider settings to js config.
		add_filter( 'modula_album_js_config', array( $this, 'js_slider_config' ), 999, 2 );

		// Add action for slider syncing.
		add_action( 'modula_album_shortcode_after_items', array( $this, 'modula_slider_syncing' ), 85 );

		// Add new path for templates.
		add_filter( 'modula_template_paths', array( $this, 'add_modula_slider_templates_path' ), 20 );

		// Enqueue block assets.
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_modula_block_styles' ) );

	}

	/**
	 * Set admin requirements and check for license.
	 *
	 * @return void
	 * @since 1.1.2
	 */
	public function set_admin_requirements() {

		if ( class_exists( 'WPChill_Upsells' ) ) {
			$args           = apply_filters(
				'modula_album_upsells_args',
				array(
					'shop_url' => 'https://wp-modula.com',
					'slug'     => 'modula',
				)
			);
			$wpchill_upsell = WPChill_Upsells::get_instance( $args );

			if ( $wpchill_upsell && ! $wpchill_upsell->is_upgradable_addon( 'modula-slider' ) ) {
				add_filter( 'modula_album_fields', array( $this, 'modula_album_slider_gallery_type' ), 15, 1 );

				// Enqueue admin required scripts Modula Albums.
				add_action( 'modula_album_scripts_before_wp_modula', array( $this, 'modula_slider_admin_scripts' ), 1 );
				add_action( 'modula_album_defaults_scripts_after_wp_modula', array(
					$this,
					'modula_slider_admin_scripts'
				),          40 );
			}
		}
	}

	/**
	 * Enqueue admin Modula Slider scripts
	 *
	 * @since 1.1.2
	 */
	public function modula_slider_admin_scripts() {

		wp_enqueue_script( 'modula-slider-conditions', MODULA_SLIDER_URL . 'assets/js/modula-slider-albums-conditions.js', array(), '', true );
		wp_enqueue_script( 'modula-slider-settings', MODULA_SLIDER_URL . 'assets/js/wp-modula-slider.js', array(), '', true );
	}


	/**
	 * Enqueue slider scripts
	 *
	 * @param array $scripts Array of scripts.
	 * @param array $settings Array of settings.
	 *
	 * @return array
	 * @since 1.1.2
	 */
	public function modula_slider_scripts( $scripts, $settings ) {

		if ( isset( $settings['album_type'] ) && 'slider' === $settings['album_type'] ) {
			$scripts[] = 'modula-slider-script';
		}

		return $scripts;
	}

	/**
	 * Add extra path for modula templates
	 *
	 * @param array $paths Array of paths.
	 *
	 * @return array
	 * @since 1.1.2
	 */
	public function add_modula_slider_templates_path( $paths ) {

		$paths[60] = trailingslashit( MODULA_SLIDER_PATH ) . '/templates';

		return $paths;
	}

	/**
	 * Enqueue slider styles
	 *
	 * @param array $styles Array of styles.
	 * @param array $settings Array of settings.
	 *
	 * @return array
	 * @since 1.1.2
	 */
	public function modula_slider_styles( $styles, $settings ) {

		if ( isset( $settings['album_type'] ) && 'slider' === $settings['album_type'] ) {
			$styles[] = 'modula-slider-css';
		}

		return $styles;
	}


	/**
	 * Returns the singleton instance of the class.
	 *
	 * @return object The Modula_Albums_Slider object.
	 * @since 1.1.2
	 */
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Modula_Albums_Slider ) ) {
			self::$instance = new Modula_Albums_Slider();
		}

		return self::$instance;
	}

	/**
	 * Add Modula Slider fields and type
	 *
	 * @param array $fields Array of fields.
	 *
	 * @return array
	 * @since 1.1.2
	 */
	public function modula_album_slider_gallery_type( $fields ) {
		// Let's manipulate the general field and album type field.
		if ( isset( $fields['general']['album_type'] ) ) {
			// Add slider fields.
			$slider_fields = array(
				'slider_adaptiveHeight'                      => array(
					'name'        => esc_html__( 'Adaptive Height', 'modula-slider' ),
					'type'        => 'toggle',
					'description' => esc_html__( 'Enables adaptive height for single slide horizontal carousels', 'modula-slider' ),
					'default'     => 0,
					'priority'    => 134,
					'data_type' => 'bool',
				),
				'slider_syncing'                             => array(
					'name'        => esc_html__( 'Thumbnail Navigator', 'modula-slider' ),
					'type'        => 'toggle',
					'description' => esc_html__( 'Enables Thumbnail Navigation', 'modula-slider' ),
					'default'     => 0,
					'priority'    => 135,
					'data_type' => 'bool',
				),
				'slider_syncing_nav_size'                    => array(
					'name'        => esc_html__( 'Image Size', 'modula-slider' ),
					'type'        => 'select',
					'description' => esc_html__( 'Select the image sizes for the thumbnail navigator from a selection of options. Select custom if you want to enter the sizes yourself.', 'modula-slider' ),
					'values'      => Modula_Helper::get_image_sizes( true ),
					'default'     => 'default',
					'is_child'    => true,
					'priority'    => 136,
					'data_type' => 'text',
				),
				'slider_syncing_nav_image_dimensions'        => array(
					'name'        => esc_html__( 'Dimensions', 'modula-slider' ),
					'type'        => 'image-size',
					'description' => esc_html__( 'Define thumbnail image dimensions. Width and Height.', 'modula-slider' ),
					'default'     => 0,
					'is_child'    => true,
					'priority'    => 137,
					'data_type' => 'number',
				),
				'slider_syncing_nav_image_crop'              => array(
					'name'        => esc_html__( 'Crop Images ?', 'modula-slider' ),
					'type'        => 'toggle',
					'description' => esc_html__( 'If this is enabled, images will be cropped down to exactly the sizes defined above.', 'modula-slider' ),
					'default'     => 0,
					'is_child'    => true,
					'priority'    => 138,
					'data_type' => 'bool',
				),
				'slider_syncing_nav_thumbnails_number'       => array(
					'name'        => esc_html__( 'Slides to Show', 'modula-slider' ),
					'type'        => 'text',
					'description' => esc_html__( 'Define the number of thumbnails shown in the navigator.', 'modula-slider' ),
					'default'     => 2,
					'is_child'    => true,
					'priority'    => 139,
					'data_type' => 'number',
				),
				'slider_syncing_nav_thumbnails_gutter'       => array(
					'name'        => esc_html__( 'Gutter', 'modula-slider' ),
					'type'        => 'ui-slider',
					'description' => esc_html__( 'Define the distance you want between each thumbnail in your navigator.', 'modula-slider' ),
					'min'         => 0,
					'max'         => 100,
					'step'        => 1,
					'default'     => 10,
					'after'       => 'px',
					'is_child'    => true,
					'priority'    => 140,
					'data_type' => 'number',
				),
				'slider_syncing_nav_thumbnails_border_color' => array(
					'name'        => esc_html__( 'Active Border Color', 'modula-slider' ),
					'type'        => 'color',
					'description' => esc_html__( 'Select the border color for the active thumbnail navigator slider', 'modula-slider' ),
					'default'     => '#000000',
					'is_child'    => true,
					'priority'    => 141,
					'data_type' => 'text',
				),
				'slider_autoplay'                            => array(
					'name'        => esc_html__( 'Autoplay', 'modula-slider' ),
					'type'        => 'toggle',
					'description' => esc_html__( 'Enables Autoplay for the slider', 'modula-slider' ),
					'default'     => 0,
					'priority'    => 142,
					'data_type' => 'bool',
				),
				'slider_autoplaySpeed'                       => array(
					'name'        => esc_html__( 'Autoplay speed', 'modula-slider' ),
					'type'        => 'text',
					'description' => esc_html__( 'Autoplay speed in milliseconds', 'modula-slider' ),
					'default'     => 3000,
					'is_child'    => true,
					'priority'    => 143,
					'data_type' => 'number',
				),
				'slider_pauseOnHover'                        => array(
					'name'        => esc_html__( 'Pause on hover', 'modula-slider' ),
					'type'        => 'toggle',
					'description' => esc_html__( 'Pause slider when mouse is hovering.', 'modula-slider' ),
					'default'     => 0,
					'is_child'    => true,
					'priority'    => 144,
					'data_type' => 'bool',
				),
				'slider_arrows'                              => array(
					'name'        => esc_html__( 'Arrows', 'modula-slider' ),
					'type'        => 'toggle',
					'description' => esc_html__( 'Toggle slide arrows navigation', 'modula-slider' ),
					'default'     => 1,
					'priority'    => 145,
					'data_type' => 'bool',
				),
				'slider_dots'                                => array(
					'name'        => esc_html__( 'Dots', 'modula-slider' ),
					'type'        => 'toggle',
					'description' => esc_html__( 'Show dot indicators.', 'modula-slider' ),
					'default'     => 0,
					'priority'    => 146,
					'data_type' => 'bool',
				),
				'slider_draggable'                           => array(
					'name'        => esc_html__( 'Mouse dragging', 'modula-slider' ),
					'type'        => 'toggle',
					'description' => esc_html__( 'Enable mouse dragging.', 'modula-slider' ),
					'default'     => 1,
					'priority'    => 147,
					'data_type' => 'bool',
				),
				'slider_centerMode'                          => array(
					'name'        => esc_html__( 'Center mode', 'modula-slider' ),
					'type'        => 'toggle',
					'description' => esc_html__( 'Enables centered view with partial prev/next slides. Only works if "Fade" is disabled.', 'modula-slider' ),
					'default'     => 20,
					'priority'    => 148,
					'data_type' => 'bool',
				),
				'slider_centerPadding'                       => array(
					'name'        => esc_html__( 'Center Padding', 'modula-slider' ),
					'type'        => 'text',
					'description' => esc_html__( 'Side padding when in center mode (px or %).', 'modula-slider' ),
					'default'     => '50px',
					'is_child'    => true,
					'priority'    => 149,
					'data_type' => 'text',
				),
				'slider_fade'                                => array(
					'name'        => esc_html__( 'Fade', 'modula-slider' ),
					'type'        => 'toggle',
					'description' => esc_html__( 'Enable fade. If number of slides on single view( "Slides to show" option ) is bigger than 1 fade will automatically be considered as disabled. ', 'modula-slider' ),
					'default'     => 0,
					'priority'    => 150,
					'data_type' => 'bool',
				),
				'slider_infinite'                            => array(
					'name'        => esc_html__( 'Infinite loop', 'modula-slider' ),
					'type'        => 'toggle',
					'description' => esc_html__( 'Toggle infinite loop.', 'modula-slider' ),
					'default'     => 1,
					'priority'    => 151,
					'data_type' => 'bool',
				),
				'slider_slidesToShow'                        => array(
					'name'        => esc_html__( 'Slides to show', 'modula-slider' ),
					'type'        => 'ui-slider',
					'description' => esc_html__( 'Number of slides to show on single view.', 'modula-slider' ),
					'min'         => 1,
					'max'         => 10,
					'step'        => 1,
					'default'     => 1,
					'priority'    => 152,
					'data_type' => 'number',
				),
				'slider_slidesToScroll'                      => array(
					'name'        => esc_html__( 'Slides to scroll', 'modula-slider' ),
					'type'        => 'ui-slider',
					'description' => esc_html__( 'Number of slides to scroll. "Center mode" must be disabled for this to work.', 'modula-slider' ),
					'min'         => 1,
					'max'         => 10,
					'step'        => 1,
					'default'     => 1,
					'priority'    => 153,
					'data_type' => 'number',
				),
				'slider_speed'                               => array(
					'name'        => esc_html__( 'Animation speed', 'modula-slider' ),
					'type'        => 'text',
					'description' => esc_html__( 'Animation speed in milliseconds', 'modula-slider' ),
					'default'     => 300,
					'priority'    => 154,
					'data_type' => 'number',
				),
				'slider_rtl'                                 => array(
					'name'        => esc_html__( 'Right-to-left', 'modula-slider' ),
					'type'        => 'toggle',
					'description' => esc_html__( 'Change the slider\'s direction to become right-to-left', 'modula-slider' ),
					'default'     => 0,
					'priority'    => 155,
					'data_type' => 'bool',
				),
				'slider_initialSlide'                        => array(
					'name'        => esc_html__( 'Initial slide', 'modula-slider' ),
					'type'        => 'text',
					'description' => esc_html__( 'Slide to start on.', 'modula-slider' ),
					'default'     => 0,
					'priority'    => 156,
					'data_type' => 'number',
				),
				'slider_image_info'                          => array(
					'name'        => esc_html__( 'Slider captions', 'modula-slider' ),
					'description' => esc_html__( 'Display image title and caption under the image.', 'modula-slider' ),
					'type'        => 'toggle',
					'default'     => 0,
					'priority'    => 157,
					'data_type' => 'bool',
				),
			);

			$slider_responsive_fields = array(
				'slider_tablet_slides'  => array(
					'name'        => esc_html__( 'Tablet slides to show ', 'modula-slider' ),
					'description' => esc_html__( 'Number of slides to show on single view in tablet mode.', 'modula-slider' ),
					'type'        => 'ui-slider',
					'min'         => 1,
					'max'         => 12,
					'step'        => 1,
					'default'     => 1,
					'priority'    => 30,
					'is_child'    => true,
					'data_type' => 'number',
				),
				'slider_tablet_scrolls' => array(
					'name'        => esc_html__( 'Tablet slides to scroll', 'modula-slider' ),
					'description' => esc_html__( 'Number of slides to scroll in tablet mode. "Center mode" must be disabled for this to work.', 'modula-slider' ),
					'type'        => 'ui-slider',
					'min'         => 1,
					'max'         => 12,
					'step'        => 1,
					'default'     => 1,
					'is_child'    => true,
					'priority'    => 31,
					'data_type' => 'number',
				),
				'slider_mobile_slides'  => array(
					'name'        => esc_html__( 'Mobile slides to show', 'modula-slider' ),
					'description' => esc_html__( 'Number of slides to show on single view in mobile mode', 'modula-slider' ),
					'type'        => 'ui-slider',
					'min'         => 1,
					'max'         => 12,
					'step'        => 1,
					'default'     => 1,
					'is_child'    => true,
					'priority'    => 32,
					'data_type' => 'number',
				),
				'slider_mobile_scrolls' => array(
					'name'        => esc_html__( 'Mobile slides to scroll', 'modula-slider' ),
					'description' => esc_html__( 'Number of slides to scroll in mobile mode. "Center mode" must be disabled for this to work.', 'modula-slider' ),
					'type'        => 'ui-slider',
					'min'         => 1,
					'max'         => 12,
					'step'        => 1,
					'default'     => 1,
					'is_child'    => true,
					'priority'    => 33,
					'data_type' => 'number',
				),
			);

			$fields['general']    = array_merge( $fields['general'], $slider_fields );
			$fields['responsive'] = array_merge( $fields['responsive'], $slider_responsive_fields );
			// Add the slider type to the album type field.
			$fields['general']['album_type']['values']['slider'] = esc_html__( 'Slider', 'modula-slider' );
			// Unset slider disabled option.
			if ( isset( $fields['general']['album_type']['disabled']['values']['slider'] ) ) {
				unset( $fields['general']['album_type']['disabled']['values']['slider'] );
			}
		}

		return $fields;
	}

	/**
	 * Remake the gallery classes and add slider_lightbox to config
	 *
	 * @param array $template_data The template data.
	 *
	 * @return array
	 * @since 1.1.2
	 */
	public function modula_album_gallery_classes( $template_data ) {

		$settings = $template_data['settings'];
		if ( isset( $settings['album_type'] ) && 'slider' === $settings['album_type'] ) {
			$template_data['album_container']['class'] = array( 'modula-slider' );

			$js_config           = json_decode( $template_data['album_container']['data-config'] );
			$js_config->lightbox = isset( $settings['enable_lightbox'] ) ? 'fancybox' : 'no-link';

			$js_config->slider_settings = array(
				'adaptiveHeight' => boolval( $settings['slider_adaptiveHeight'] ),
				'autoplay'       => boolval( $settings['slider_autoplay'] ),
				'autoplaySpeed'  => ( 0 <= absint( $settings['slider_autoplaySpeed'] ) ) ? absint( $settings['slider_autoplaySpeed'] ) : 3000,
				'arrows'         => boolval( $settings['slider_arrows'] ),
				'centerMode'     => boolval( $settings['slider_centerMode'] ),
				'centerPadding'  => $settings['slider_centerPadding'],
				'dots'           => true,
				'draggable'      => boolval( $settings['slider_draggable'] ),
				'fade'           => ( absint( $settings['slider_slidesToShow'] ) > 1 ) ? false : boolval( $settings['slider_fade'] ),
				'navImageShow'   => absint( $settings['slider_syncing_nav_thumbnails_number'] ),
				'infinite'       => boolval( $settings['slider_infinite'] ),
				'pauseOnHover'   => boolval( $settings['slider_pauseOnHover'] ),
				'slidesToShow'   => absint( $settings['slider_slidesToShow'] ),
				'slidesToScroll' => absint( $settings['slider_slidesToScroll'] ),
				'speed'          => absint( $settings['slider_speed'] ),
				'rtl'            => boolval( $settings['slider_rtl'] ),
				'initialSlide'   => ( $settings['slider_initialSlide'] > 0 ) ? (int) $settings['slider_initialSlide'] - 1 : (int) $settings['slider_initialSlide'],
				'prevArrow'      => '<a href="#" class="slick-prev"><svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="angle-left" class="svg-inline--fa fa-angle-left fa-w-8" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 512"><path fill="currentColor" d="M31.7 239l136-136c9.4-9.4 24.6-9.4 33.9 0l22.6 22.6c9.4 9.4 9.4 24.6 0 33.9L127.9 256l96.4 96.4c9.4 9.4 9.4 24.6 0 33.9L201.7 409c-9.4 9.4-24.6 9.4-33.9 0l-136-136c-9.5-9.4-9.5-24.6-.1-34z"></path></svg></a>',
				'nextArrow'      => '<a href="#" class="slick-next"><svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="angle-right" class="svg-inline--fa fa-angle-right fa-w-8" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 512"><path fill="currentColor" d="M224.3 273l-136 136c-9.4 9.4-24.6 9.4-33.9 0l-22.6-22.6c-9.4-9.4-9.4-24.6 0-33.9l96.4-96.4-96.4-96.4c-9.4-9.4-9.4-24.6 0-33.9L54.3 103c9.4-9.4 24.6-9.4 33.9 0l136 136c9.5 9.4 9.5 24.6.1 34z"></path></svg></a>',
			);

			if ( $settings['slider_rtl'] ) {
				$js_config->slider_settings['prevArrow'] = '<a href="#" class="slick-prev"><svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="angle-right" class="svg-inline--fa fa-angle-right fa-w-8" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 512"><path fill="currentColor" d="M224.3 273l-136 136c-9.4 9.4-24.6 9.4-33.9 0l-22.6-22.6c-9.4-9.4-9.4-24.6 0-33.9l96.4-96.4-96.4-96.4c-9.4-9.4-9.4-24.6 0-33.9L54.3 103c9.4-9.4 24.6-9.4 33.9 0l136 136c9.5 9.4 9.5 24.6.1 34z"></path></svg></a>';
				$js_config->slider_settings['nextArrow'] = '<a href="#" class="slick-next"><svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="angle-left" class="svg-inline--fa fa-angle-left fa-w-8" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 512"><path fill="currentColor" d="M31.7 239l136-136c9.4-9.4 24.6-9.4 33.9 0l22.6 22.6c9.4 9.4 9.4 24.6 0 33.9L127.9 256l96.4 96.4c9.4 9.4 9.4 24.6 0 33.9L201.7 409c-9.4 9.4-24.6 9.4-33.9 0l-136-136c-9.5-9.4-9.5-24.6-.1-34z"></path></svg></a>';
			}

			if ( $settings['lazy_load'] ) {
				if ( isset( $settings['slider_adaptiveHeight'] ) && 1 != $settings['slider_adaptiveHeight'] ) {
					$js_config->slider_settings['lazyLoad'] = 'ondemand';
				} else {
					$js_config->slider_settings['lazyLoad'] = 'progressive';
				}
			}

			if ( $js_config->slider_settings['centerMode'] ) {
				$js_config->slider_settings['centerPadding'] = esc_attr( $settings['slider_centerPadding'] );
			}

			if ( isset( $settings['enable_responsive'] ) && '1' == $settings['enable_responsive'] ) {
				$responsive = array(
					array(
						'breakpoint' => 1024,
						'settings'   => array(
							'slidesToShow'   => isset( $settings['slider_tablet_slides'] ) ? absint( $settings['slider_tablet_slides'] ) : 1,
							'slidesToScroll' => isset( $settings['slider_tablet_scrolls'] ) ? absint( $settings['slider_tablet_scrolls'] ) : 1,
						),
					),
					array(
						'breakpoint' => 600,
						'settings'   => array(
							'slidesToShow'   => isset( $settings['slider_mobile_slides'] ) ? absint( $settings['slider_mobile_slides'] ) : 1,
							'slidesToScroll' => isset( $settings['slider_mobile_scrolls'] ) ? absint( $settings['slider_mobile_scrolls'] ) : 1,
						),
					),
				);

				$js_config->slider_settings['responsive'] = $responsive;
			}

			$template_data['album_container']['data-config'] = json_encode( $js_config );

			// Remove hover effects from slider.
			$template_data['settings']['effect'] = 'slider';
		}

		return $template_data;
	}

	/**
	 * We need to overwrite the height but still display the correct css
	 *
	 * @param string $css The css.
	 * @param int $album_id The Album ID.
	 * @param array $settings The gallery settings.
	 *
	 * @return string
	 * @since 1.1.2
	 */
	public function modula_album_slider_output_css( $css, $album_id, $settings ) {

		if ( isset( $settings['album_type'] ) && 'slider' == $settings['album_type'] ) {

			$width = ( ! empty( $settings['width'] ) ) ? $settings['width'] : '100%';
			$css   .= "#{$album_id}.modula-slider { max-width: {$width} }";
			$css   .= "#{$album_id}.modula-slider .modula-item .video-sizer iframe { min-width:100% }";

			// set the height to auto
			$css .= "#{$album_id} .modula-items{height:auto}";
			// remove the hover background color
			$css .= "#{$album_id} .modula-item{background-color:transparent}";
			// remove loading effects
			$css .= "#{$album_id} .modula-item{transform:none;}";
			// remove hover opacity on image
			$css .= "#{$album_id} .modula-item:hover img{opacity:1;}";
			$css .= "#{$album_id} .modula-slider-nav .slick-list .slick-track .slick-slide { margin: " . absint( $settings['slider_syncing_nav_thumbnails_gutter'] / 2 ) . 'px}';

			$css .= "#{$album_id} .slider-clickable { position:relative;width:max-content;margin:0 auto;max-width:100%; }#{$album_id} .slider-clickable .modula-item-link, #{$album_id}  .slider-clickable .modula-simple-link { position:absolute; width:100%; height:100%; cursor:pointer; z-index:999;}";

			// Gutter only for < 1 slidesToShow.
			if ( 1 == $settings['slider_slidesToShow'] && ( ! isset( $settings['slider_centerMode'] ) || '1' != $settings['slider_centerMode'] ) ) {
				$css .= $css .= "#{$album_id} .modula-items .slick-list .slick-track .slick-slide { margin: 0}";
			} else {
				$css .= "#{$album_id} .modula-items .slick-list .slick-track .slick-slide { margin: 0 " . absint( $settings['gutter'] / 2 ) . 'px}';
			}

			// arrow fix.
			if ( isset( $settings['slider_arrows'] ) && '1' == $settings['slider_arrows'] ) {
				$css .= "#{$album_id} .modula-items .slick-list { width :calc(100% - 40px); margin: 0 auto; }";
				$css .= "#{$album_id} .slider-nav .slick-list { width :calc(100% - 40px); margin: 0 auto; }";
			}

			// Slider dimensions for video.
			$real_sizes = self::get_real_sizes( $settings );
			$min_height = esc_attr( '50vh' );
			if ( 0 == $real_sizes['width'] ) {
				$width = esc_attr( '100%' );
			} else {
				$min_height = esc_attr( 'auto' );
				$width      = absint( $real_sizes['width'] ) . 'px';
			}

			$css .= "#{$album_id} .modula-items .slick-slide .modula-item .video-sizer { width:" . $width . '; max-width:100%;}';
			$css .= "#{$album_id} .modula-items .slick-slide .modula-item .video-sizer iframe, #{$album_id}  .modula-items .slick-slide .modula-item .video-sizer video { height:" . $width . '!important; position: relative;max-height:100vh;min-height:' . $min_height . ' }';

			if ( ! isset( $settings['hide_title'] ) || '1' != $settings['hide_title'] ) {

				$css .= "#{$album_id}.modula-slider .modula-items .slider-image-info .jtg-title {margin-bottom:0px;}";

				if ( '' != $settings['titleColor'] ) {
					$css .= "#{$album_id}.modula-slider .modula-items .slider-image-info .jtg-title {";
					$css .= 'color:' . Modula_Helper::sanitize_rgba_colour( $settings['titleColor'] ) . ';';
					$css .= '}';
				}

				if ( '' != $settings['titleFontSize'] && '0' != $settings['titleFontSize'] ) {
					$css .= "#{$album_id}.modula-slider .modula-items .slider-image-info .jtg-title {";
					$css .= 'font-size:' . absint( $settings['titleFontSize'] ) . 'px;';
					$css .= '}';
				}

				if ( '' != $settings['mobileTitleFontSize'] && '0' != $settings['mobileTitleFontSize'] ) {
					$css .= "@media all and (max-width:600px){#{$album_id}.modula-slider .modula-items .slider-image-info .description {";
					$css .= 'font-size:' . absint( $settings['mobileCaptionFontSize'] ) . 'px;';
					$css .= '}}';
				}
			}

			if ( ! isset( $settings['hide_description'] ) || '1' != $settings['hide_description'] ) {

				$css .= "#{$album_id}.modula-slider .modula-items .slider-image-info .description {margin-bottom:0px;}";

				if ( '' != $settings['captionColor'] ) {
					$css .= "#{$album_id}.modula-slider .modula-items .slider-image-info .description {";
					$css .= 'color:' . Modula_Helper::sanitize_rgba_colour( $settings['captionColor'] ) . ';';
					$css .= '}';
				}

				if ( '' != $settings['captionFontSize'] && '0' != $settings['captionFontSize'] ) {
					$css .= "#{$album_id}.modula-slider .modula-items .slider-image-info .description {";
					$css .= 'font-size:' . absint( $settings['captionFontSize'] ) . 'px;';
					$css .= '}';
				}

				if ( '' != $settings['mobileCaptionFontSize'] && '0' != $settings['mobileCaptionFontSize'] ) {
					$css .= "@media all and (max-width:600px){#{$album_id}.modula-slider .modula-items .slider-image-info .description {";
					$css .= 'font-size:' . absint( $settings['mobileCaptionFontSize'] ) . 'px;';
					$css .= '}}';
				}
			}

			if ( isset( $settings['slider_syncing_nav_thumbnails_border_color'] ) ) {
				$css .= '#' . $album_id . '.modula-slider .modula-slider-nav .slick-slide.is-active img{border: 2px solid ' . Modula_Helper::sanitize_rgba_colour( $settings['slider_syncing_nav_thumbnails_border_color'] ) . '}';

			}
			$css .= '#' . $album_id . '.modula-slider .modula-item-content{ position: relative; display:table; margin: 0 auto;}';
			$css .= '#' . $album_id . '.modula-slider .modula-item-content a.gallery-link{ position: absolute;left: 0; right: 0;top: 0;bottom: 0; z-index: 9999; }';
		}

		return $css;
	}

	/**
	 * Set defaults
	 *
	 * @param $defaults
	 *
	 * @return mixed
	 * @since 1.1.2
	 */
	public function set_defaults( $defaults ) {
		$defaults['album_type']                                 = 'grid';
		$defaults['slider_adaptiveHeight']                      = 0;
		$defaults['slider_image_height']                        = '800';
		$defaults['slider_syncing']                             = 0;
		$defaults['slider_syncing_nav_size']                    = 'thumbnail';
		$defaults['slider_syncing_nav_thumbnails_number']       = '2';
		$defaults['slider_syncing_nav_thumbnails_gutter']       = 10;
		$defaults['slider_syncing_nav_thumbnails_border_color'] = '#000000';
		$defaults['slider_autoplay']                            = 0;
		$defaults['slider_autoplaySpeed']                       = 3000;
		$defaults['slider_arrows']                              = 1;
		$defaults['slider_centerMode']                          = 1;
		$defaults['slider_centerPadding']                       = '50px';
		$defaults['slider_dots']                                = 0;
		$defaults['slider_draggable']                           = 1;
		$defaults['slider_fade']                                = 0;
		$defaults['slider_infinite']                            = 1;
		$defaults['slider_pauseOnHover']                        = 1;
		$defaults['slider_slidesToShow']                        = 1;
		$defaults['slider_slidesToScroll']                      = 1;
		$defaults['slider_speed']                               = 300;
		$defaults['slider_rtl']                                 = 0;
		$defaults['slider_initialSlide']                        = 0;
		$defaults['slider_tablet_slides']                       = 1;
		$defaults['slider_tablet_scrolls']                      = 1;
		$defaults['slider_mobile_slides']                       = 1;
		$defaults['slider_mobile_scrolls']                      = 1;
		$defaults['slider_syncing_nav_image_dimensions']        = 0;
		$defaults['slider_syncing_nav_image_crop']              = 0;
		$defaults['slider_image_info']                          = 0;

		return $defaults;
	}

	/**
	 * Slider JS config
	 *
	 * @param array $js_config JS settings.
	 * @param array $settings Album settings.
	 *
	 * @return array
	 * @since 1.1.2
	 */
	public function js_slider_config( $js_config, $settings ) {

		if ( isset( $settings['album_type'] ) && 'slider' == $settings['album_type'] ) {
			$js_config['slider_settings'] = array(
				'adaptiveHeight' => boolval( $settings['slider_adaptiveHeight'] ),
				'autoplay'       => boolval( $settings['slider_autoplay'] ),
				'autoplaySpeed'  => absint( $settings['slider_autoplaySpeed'] ),
				'arrows'         => boolval( $settings['slider_arrows'] ),
				'centerMode'     => boolval( $settings['slider_centerMode'] ),
				'centerPadding'  => $settings['slider_centerPadding'],
				'dots'           => true,
				'draggable'      => boolval( $settings['slider_draggable'] ),
				'fade'           => ( absint( $settings['slider_slidesToShow'] ) > 1 ) ? false : boolval( $settings['slider_fade'] ),
				'navImageShow'   => absint( $settings['slider_syncing_nav_thumbnails_number'] ),
				'infinite'       => boolval( $settings['slider_infinite'] ),
				'pauseOnHover'   => boolval( $settings['slider_pauseOnHover'] ),
				'slidesToShow'   => absint( $settings['slider_slidesToShow'] ),
				'slidesToScroll' => absint( $settings['slider_slidesToScroll'] ),
				'speed'          => absint( $settings['slider_speed'] ),
				'rtl'            => boolval( $settings['slider_rtl'] ),
				'initialSlide'   => ( $settings['slider_initialSlide'] > 0 ) ? (int) $settings['slider_initialSlide'] - 1 : (int) $settings['slider_initialSlide'],
				'prevArrow'      => '<a href="#" class="slick-prev"><svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="angle-left" class="svg-inline--fa fa-angle-left fa-w-8" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 512"><path fill="currentColor" d="M31.7 239l136-136c9.4-9.4 24.6-9.4 33.9 0l22.6 22.6c9.4 9.4 9.4 24.6 0 33.9L127.9 256l96.4 96.4c9.4 9.4 9.4 24.6 0 33.9L201.7 409c-9.4 9.4-24.6 9.4-33.9 0l-136-136c-9.5-9.4-9.5-24.6-.1-34z"></path></svg></a>',
				'nextArrow'      => '<a href="#" class="slick-next"><svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="angle-right" class="svg-inline--fa fa-angle-right fa-w-8" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 512"><path fill="currentColor" d="M224.3 273l-136 136c-9.4 9.4-24.6 9.4-33.9 0l-22.6-22.6c-9.4-9.4-9.4-24.6 0-33.9l96.4-96.4-96.4-96.4c-9.4-9.4-9.4-24.6 0-33.9L54.3 103c9.4-9.4 24.6-9.4 33.9 0l136 136c9.5 9.4 9.5 24.6.1 34z"></path></svg></a>',
			);

			if ( $settings['slider_rtl'] ) {
				$js_config['slider_settings']['prevArrow'] = '<a href="#" class="slick-prev"><svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="angle-right" class="svg-inline--fa fa-angle-right fa-w-8" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 512"><path fill="currentColor" d="M224.3 273l-136 136c-9.4 9.4-24.6 9.4-33.9 0l-22.6-22.6c-9.4-9.4-9.4-24.6 0-33.9l96.4-96.4-96.4-96.4c-9.4-9.4-9.4-24.6 0-33.9L54.3 103c9.4-9.4 24.6-9.4 33.9 0l136 136c9.5 9.4 9.5 24.6.1 34z"></path></svg></a>';
				$js_config['slider_settings']['nextArrow'] = '<a href="#" class="slick-next"><svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="angle-left" class="svg-inline--fa fa-angle-left fa-w-8" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 512"><path fill="currentColor" d="M31.7 239l136-136c9.4-9.4 24.6-9.4 33.9 0l22.6 22.6c9.4 9.4 9.4 24.6 0 33.9L127.9 256l96.4 96.4c9.4 9.4 9.4 24.6 0 33.9L201.7 409c-9.4 9.4-24.6 9.4-33.9 0l-136-136c-9.5-9.4-9.5-24.6-.1-34z"></path></svg></a>';
			}

			if ( $settings['lazy_load'] ) {
				if ( isset( $settings['slider_adaptiveHeight'] ) && 1 != $settings['slider_adaptiveHeight'] ) {
					$js_config['slider_settings']['lazyLoad'] = 'ondemand';
				} else {
					$js_config['slider_settings']['lazyLoad'] = 'progressive';
				}
			}

			if ( $js_config['slider_settings']['centerMode'] ) {
				$js_config['slider_settings']['centerPadding'] = esc_attr( $settings['slider_centerPadding'] );
			}

			if ( isset( $settings['enable_responsive'] ) && '1' == $settings['enable_responsive'] ) {
				$responsive = array(
					array(
						'breakpoint' => 1024,
						'settings'   => array(
							'slidesToShow'   => isset( $settings['slider_tablet_slides'] ) ? absint( $settings['slider_tablet_slides'] ) : 1,
							'slidesToScroll' => isset( $settings['slider_tablet_scrolls'] ) ? absint( $settings['slider_tablet_scrolls'] ) : 1,
						),
					),
					array(
						'breakpoint' => 600,
						'settings'   => array(
							'slidesToShow'   => isset( $settings['slider_mobile_slides'] ) ? absint( $settings['slider_mobile_slides'] ) : 1,
							'slidesToScroll' => isset( $settings['slider_mobile_scrolls'] ) ? absint( $settings['slider_mobile_scrolls'] ) : 1,
						),
					),
				);

				$js_config['slider_settings']['responsive'] = $responsive;
			}
		}

		return $js_config;
	}

	public static function get_real_sizes( $settings ) {

		if ( ! is_array( $settings ) || empty( $settings ) || 'full' == $settings['image_size'] ) {
			return array(
				'width'  => 0,
				'height' => 0,
			);
		}

		if ( 'custom' != $settings['image_size'] ) {
			$size = Modula_Helper::get_image_sizes()[ $settings['image_size'] ];
		} else {
			$size = $settings['image_dimensions'];
		}

		return $size;
	}

	public function change_effect( $template_data ) {

		if ( 'slider' != $template_data['settings']['album_type'] ) {
			return $template_data;
		}

		$template_data['settings']['effect'] = 'slider';

		return $template_data;

	}

	/**
	 * Sanitize slider fields
	 *
	 * @param $sanitized_value
	 * @param $value
	 * @param $field_id
	 * @param $field
	 *
	 * @return mixed
	 *
	 * @since 1.1.2
	 */
	public function sanitize_slider_fields( $sanitized_value, $value, $field_id, $field ) {

		if ( 'slider_image_dimensions' == $field_id ) {
			$new_value = array();
			if ( is_array( $value ) && ! empty( $value ) ) {
				foreach ( $value as $key => $img_size ) {
					$new_value[ $key ] = absint( $img_size );
				}
			}

			return $new_value;
		}

		return $sanitized_value;

	}

	public function enqueue_modula_block_styles() {

		wp_register_style( 'modula-slider-css', MODULA_SLIDER_URL . 'assets/css/front.min.css' );
	}

	/**
	 * Render Thumbnail navigation
	 *
	 * @param $settings
	 * @param $item_data
	 * @param $images
	 *
	 * @return void $html
	 * @since 1.1.2
	 */
	public function modula_slider_syncing( $settings ) {

		// get the galleries in album
		$galleries = get_post_meta( absint( preg_replace( '/\D+/', '', $settings['gallery_id'] ) ), 'modula-album-galleries', true );

		// get each gallery's first image or cover image.
		foreach ( $galleries as $gallery ) {
			if ( ! empty( $gallery['cover'] ) ) {
				$images[]['id'] = absint( $gallery['cover'] );
				continue;
			}

			$images[] = get_post_meta( $gallery['id'], 'modula-images', true )[0];
		}

		if ( isset( $settings['album_type'] ) && isset( $settings['slider_slidesToShow'] ) && 'slider' == $settings['album_type'] && 1 == $settings['slider_slidesToShow'] ) {

			if ( 1 == $settings['slider_syncing'] ) {

				if ( ! empty( $images ) ) {
					global $_wp_additional_image_sizes;

					$slider_width = '';
					if ( 'custom' == $settings['slider_syncing_nav_size'] ) {
						$slider_width = $settings['slider_syncing_nav_image_dimensions']['width'] * $settings['slider_syncing_nav_thumbnails_number'] + $settings['gutter'] - 10;
					} else {
						$width = 100;
						if ( isset( $_wp_additional_image_sizes[ $settings['slider_syncing_nav_size'] ] ) ) {
							$width = $_wp_additional_image_sizes[ $settings['slider_syncing_nav_size'] ]['width'];
						}
						$slider_width = $width * $settings['slider_syncing_nav_thumbnails_number'] + $settings['gutter'] - 10;
					}

					$config = array(
						'slidesToShow'   => absint( $settings['slider_syncing_nav_thumbnails_number'] ),
						'slidesToScroll' => absint( $settings['slider_slidesToScroll'] ),
						'focusOnSelect'  => false,
						'focusOnChange'  => false,
						'dots'           => false,
						'infinite'       => false,
						'prevArrow'      => '<a href="#" class="slick-prev"><svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="angle-left" class="svg-inline--fa fa-angle-left fa-w-8" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 512"><path fill="currentColor" d="M31.7 239l136-136c9.4-9.4 24.6-9.4 33.9 0l22.6 22.6c9.4 9.4 9.4 24.6 0 33.9L127.9 256l96.4 96.4c9.4 9.4 9.4 24.6 0 33.9L201.7 409c-9.4 9.4-24.6 9.4-33.9 0l-136-136c-9.5-9.4-9.5-24.6-.1-34z"></path></svg></a>',
						'nextArrow'      => '<a href="#" class="slick-next"><svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="angle-right" class="svg-inline--fa fa-angle-right fa-w-8" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 512"><path fill="currentColor" d="M224.3 273l-136 136c-9.4 9.4-24.6 9.4-33.9 0l-22.6-22.6c-9.4-9.4-9.4-24.6 0-33.9l96.4-96.4-96.4-96.4c-9.4-9.4-9.4-24.6 0-33.9L54.3 103c9.4-9.4 24.6-9.4 33.9 0l136 136c9.5 9.4 9.5 24.6.1 34z"></path></svg></a>',
					);

					echo "<div class='modula-slider-nav' style='width:" . absint( $slider_width ) . "px; max-width: 100%;' data-config='" . esc_attr( json_encode( $config ) ) . "'>";
					foreach ( $images as $index => $image ) {
						$thumbnail_src = '';

						if ( 'custom' != $settings['slider_syncing_nav_size'] ) {
							$thumb = wp_get_attachment_image_src( $image['id'], $settings['slider_syncing_nav_size'] );

							if ( $thumb ) {
								$thumbnail_src = $thumb[0];
							}
						} else {
							$image_full = wp_get_attachment_image_src( $image['id'], 'full' );
							$width      = $settings['slider_syncing_nav_image_dimensions']['width'];
							$height     = $settings['slider_syncing_nav_image_dimensions']['height'];
							$crop       = boolval( $settings['slider_syncing_nav_image_crop'] );

							if ( ! $image_full ) {
								break;
							}

							$resizer = new Modula_Image();
							$resized = $resizer->resize_image( $image_full[0], $width, $height, $crop );

							// Let's check if resize gives us both URL and image info.
							if ( isset( $resized['resized_url'] ) ) {
								$image_url = $resized['resized_url'];
							} else {
								$image_url = $resized;
							}

							if ( is_wp_error( $image_url ) ) {
								$image_url = $image_full[0];
							}

							$thumbnail_src = $image_url;
						}

						echo '<div class="modula-slider-nav-item">';
						do_action( 'modula_slider_before_thumbnail', $image, $images, $settings );
						echo "<img class='modula-slider-nav-image' src='" . esc_url( $thumbnail_src ) . "'>";
						do_action( 'modula_slider_after_thumbnail', $image, $images, $settings );
						echo '</div>';

					}
					echo '</div>';
				}
			}
		}
	}
}
