<?php
// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 *
 */
class Modula_Slider {

	/**
	 * Holds the class object.
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public static $instance;

	/**
	 * The name of the plugin.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $plugin_name = 'Modula Slider';

	/**
	 * Unique plugin slug identifier.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $plugin_slug = 'modula-slider';

	/**
	 * Primary class constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		// Load the plugin textdomain.
		add_action( 'init', array( $this, 'set_locale' ) );

		// Set admin requirements on admin_init so that we won't trigger get_packages from WPChill_Upsells on frontend.
		add_action( 'admin_init', array( $this, 'set_admin_requirements' ), 25 );

		// Register Slider Scripts.
		add_action( 'wp_enqueue_scripts', array( $this, 'register_slider_scripts' ) );

		// Elementor Scripts and Styles actions.
		add_action( 'modula_elementor_before_enqueue_elementor-preview', array( $this, 'modula_slider_elementor_scripts' ) );
		add_action( 'modula_elementor_before_enqueue_styles', array( $this, 'modula_slider_elementor_styles' ) );

		// Filter Modula slider Scripts & Styles.
		add_filter( 'modula_necessary_scripts', array( $this, 'modula_slider_scripts' ), 15, 2 );
		add_filter( 'modula_necessary_styles', array( $this, 'modula_slider_styles' ), 15, 2 );

		// Enqueue Modula slider Scripts & Styles for Divi pagebuilder.
		add_action( 'modula_divi_builder_sripts_after_modula', array( $this, 'modula_divi_builder_slider_scripts' ) );

		// Add defaults.
		add_filter( 'modula_lite_default_settings', array( $this, 'set_defaults' ) );

		// Add image sizes.
		add_filter( 'modula_resize_image_slider', array( $this, 'modula_slider_image_sizes' ), 15, 4 );

		add_filter( 'modula_speedup_image_size', array( $this, 'modula_slider_get_size' ), 20, 2 );

		// Add new classe for each item and add lightbox_slider option to lightbox.
		add_filter( 'modula_shortcode_item_data', array( $this, 'modula_slider_item_data' ), 90, 3 );

		// Generate new images' links.
		add_filter( 'modula_shortcode_item_data', array( $this, 'change_image_size' ), 6, 3 );

		// Add slider class and remove the other + add the slider_lightbox setting to config ( v1.0.6 ).
		add_filter( 'modula_gallery_template_data', array( $this, 'modula_gallery_classes' ), 40 );

		// Right now slider doesn't support hover effects.
		add_filter( 'modula_gallery_template_data', array( $this, 'change_effect' ), 99 );

		// Model output css.
		add_filter( 'modula_shortcode_css', array( $this, 'modula_slider_output_css' ), 17, 3 );

		// Remove filters.
		add_filter( 'modula_gallery_template_data', array( $this, 'modula_slider_settings' ), 10, 1 );

		// Add direction attribute to the slider.
		add_filter( 'modula_gallery_template_data', array( $this, 'modula_items_attribute_config' ), 10 );

		// Add slider settings to js config.
		add_filter( 'modula_gallery_settings', array( $this, 'js_slider_config' ), 10, 2 );

		// Add action for slider syncing.
		add_action( 'modula_shortcode_after_items', array( $this, 'modula_slider_syncing' ), 85, 3 );

		// Add new path for templates.
		add_filter( 'modula_template_paths', array( $this, 'add_modula_slider_templates_path' ), 20 );

		add_filter( 'modula_settings_field_sanitization', array( $this, 'sanitize_slider_fields' ), 25, 4 );

		add_filter( 'modula_migrate_gallery_data', array( $this, 'modula_migrate_data' ), 99, 3 );

		// Enqueue block assets.
		add_action( 'modula_block_style', array( $this, 'enqueue_modula_block_styles' ) );
		add_action( 'modula_block_scripts', array( $this, 'enqueue_modula_block_scripts' ) );

		register_activation_hook( MODULA_SLIDER_FILE, array( $this, 'plugin_activation' ) );

		// Troubleshooting scripts.
		add_filter( 'modula_troubleshooting_fields', array( $this, 'add_troubleshooting_fields' ) );
		add_filter( 'modula_troubleshooting_defaults', array( $this, 'add_troubleshooting_defaults' ) );
		add_filter( 'modula_troubleshooting_frontend_handles', array( $this, 'add_main_slider_files' ), 70, 2 );

		// Add shuffle for slider.
		add_filter( 'modula_shuffle_grid_types', array( $this, 'add_slider_shuffle' ) );

		// Load the plugin.
		$this->init();

	}

	/**
	 * Set admin requirements and check for license.
	 *
	 * @return void
	 * @since 1.0.14
	 */
	public function set_admin_requirements() {
		if ( class_exists( 'WPChill_Upsells' ) ) {
			$args           = apply_filters(
				'modula_upsells_args',
				array(
					'shop_url' => 'https://wp-modula.com',
					'slug'     => 'modula',
				)
			);
			$wpchill_upsell = WPChill_Upsells::get_instance( $args );

			if ( $wpchill_upsell && ! $wpchill_upsell->is_upgradable_addon( 'modula-slider' ) ) {
				add_filter( 'modula_gallery_fields', array( $this, 'modula_slider_gallery_type' ), 15, 1 );

				// Enqueue admin required scripts
				add_action( 'modula_scripts_before_wp_modula', array( $this, 'modula_slider_admin_scripts' ), 1 );
				add_action( 'modula_defaults_scripts_after_wp_modula', array( $this, 'modula_slider_admin_scripts' ), 40 );
			}
		}
	}

	/**
	 * Prevent plugin activation if Modula Pro is not installed and activated
	 *
	 * @since 1.0.1
	 */
	public function plugin_activation() {
		if ( ! class_exists( 'Modula_PRO' ) ) {
			deactivate_plugins( plugin_basename( MODULA_SLIDER_FILE ) );
			wp_die( __( 'Please install and activate Modula Pro before using Modula Slider add-on.', 'modula-slider' ) );

		}

	}

	/**
	 * Loads the plugin textdomain for translation.
	 *
	 * @since 1.0.0
	 */
	public function set_locale() {
		load_plugin_textdomain( $this->plugin_slug, false, MODULA_SLIDER_TRANSLATE );
	}

	/**
	 * Loads the plugin into WordPress.
	 *
	 * @since 1.0.0
	 */
	public function init() {

		// Load admin only components.
		if ( is_admin() ) {
			add_action( 'modula_pro_updater', array( $this, 'addon_updater' ), 15, 2 );
		}

	}

	public function addon_updater( $license_key, $store_url ) {

		if ( class_exists( 'Modula_Pro_Base_Updater' ) ) {
			$modula_addon_updater = new Modula_Pro_Base_Updater(
				$store_url,
				MODULA_SLIDER_FILE,
				array(
					'version' => MODULA_SLIDER_VERSION,        // current version number
					'license' => $license_key,               // license key (used get_option above to retrieve from DB)
					'item_id' => 408783,                      // ID of the product
					'author'  => 'MachoThemes',            // author of this plugin
					'beta'    => false,
				)
			);
		}
	}

	/**
	 * Returns the singleton instance of the class.
	 *
	 * @since 1.0.0
	 *
	 * @return object The Modula_Slider object.
	 */
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Modula_Slider ) ) {
			self::$instance = new Modula_Slider();
		}

		return self::$instance;

	}

	/**
	 * Register slider scripts and styles
	 */
	public function register_slider_scripts() {
		// Register Slider script
		wp_register_script( 'modula-slider-script', MODULA_SLIDER_URL . 'assets/js/front.js', array( 'jquery' ) );

		wp_register_style( 'modula-slider-css', MODULA_SLIDER_URL . 'assets/css/front.min.css' );

	}

	/**
	 * Enqueue admin Modula Slider scripts
	 */
	public function modula_slider_admin_scripts() {
		wp_enqueue_script( 'modula-slider-conditions', MODULA_SLIDER_URL . 'assets/js/modula-slider-conditions.js', array( 'modula-pro' ), '', true );
		wp_enqueue_script( 'modula-slider-settings', MODULA_SLIDER_URL . 'assets/js/wp-modula-slider.js', array(), '', true );
	}

	/**
	 * Enqueue slider scripts
	 *
	 * @param $scripts
	 * @param $settings
	 *
	 * @return array
	 */
	public function modula_slider_scripts( $scripts, $settings ) {

		if ( 'slider' == $settings['type'] ) {
			$excludes = array( 'modula', 'modula-isotope', 'modula-isotope-packery');
			if( isset( $settings['slider_lightbox'] ) && 'fancybox' != $settings['slider_lightbox'] ){
				$excludes[] = 'modula-fancybox';
			}
			
			foreach( $scripts as $key => $script ){
				if( in_array( $script, $excludes, true ) ){
					unset( $scripts[$key] );
				}
			}

			$scripts[] = 'modula-slider-script';
			$scripts[] = 'modula-pro';
		}
		return $scripts;
	}

	/**
	 * Enqueue slider styles
	 *
	 * @param $styles
	 * @param $settings
	 * @return array
	 */
	public function modula_slider_styles( $styles, $settings ) {

		if ( 'slider' == $settings['type'] ) {
			$styles[] = 'modula-slider-css';
		}

		return $styles;
	}
	
	/**
	 * Enqueue slider styles & scripts for Divi builder.
	 *
	 * @return void
	 */
	public function modula_divi_builder_slider_scripts() {
		wp_enqueue_script( 'modula-pro' );
		wp_enqueue_script( 'modula-slider-script' );
		wp_enqueue_style( 'modula-slider-css' );
	}
	

	/**
	 * Enqueue slider elementor scripts
	 */

	public function modula_slider_elementor_scripts() {
		wp_enqueue_script( 'modula-slider-elementor', MODULA_SLIDER_URL . 'assets/js/modula-slider-elementor.js', true );
		wp_enqueue_script( 'slick-js', MODULA_SLIDER_URL . 'assets/js/slick.min.js', true );
	}

	/**
	 * Enqueue slider elementor styles
	 */

	public function modula_slider_elementor_styles() {
		wp_enqueue_style( 'modula-slider-css' );
	}

	/**
	 * Add Modula Slider fields and type
	 *
	 * @param $fields
	 * @return mixed
	 */
	public function modula_slider_gallery_type( $fields ) {

		// Add new Gallery type
		if ( isset( $fields['general']['type']['disabled'] ) && isset( $fields['general']['type']['disabled']['values'] ) && isset( $fields['general']['type']['disabled']['values']['slider'] ) ) {
			unset( $fields['general']['type']['disabled']['values']['slider'] );
		}
		$fields['general']['type']['values']['slider'] = esc_html__( 'Slider', 'modula-slider' );

		// Add slider fields
		$slider_fields = array(
			'slider_image_size'                          => array(
				'name'        => esc_html__( 'Image size', 'modula-slider' ),
				'type'        => 'dimensions-select',
				'description' => esc_html__( 'Select the image sizes for the slider from a selection of options. Select custom if you want to enter the sizes yourself.', 'modula-slider' ),
				'values'      => Modula_Helper::get_image_sizes( true ),
				'default'     => 'default',
				'priority'    => 27,
                'data_type' => 'text',
			),
			'slider_image_dimensions'                    => array(
				'name'        => esc_html__( 'Image Dimensions', 'modula-slider' ),
				'type'        => 'image-size',
				'description' => esc_html__( 'Define image dimensions. If Crop images isn\'t enabled, images will be proportional.', 'modula-slider' ),
				'default'     => 0,
				'is_child'    => true,
				'priority'    => 28,
                'data_type' => 'number',
			),

			'slider_image_crop'                          => array(
				'name'        => esc_html__( 'Crop Images ?', 'modula-slider' ),
				'type'        => 'toggle',
				'description' => esc_html__( 'If this is enabled, images will be cropped down to exactly the sizes defined above.', 'modula-slider' ),
				'default'     => 0,
				'is_child'    => true,
				'priority'    => 29,
                'data_type' => 'bool',
			),
			'slider_adaptiveHeight'                      => array(
				'name'        => esc_html__( 'Adaptive Height', 'modula-slider' ),
				'type'        => 'toggle',
				'description' => esc_html__( 'Enables adaptive height for single slide horizontal carousels', 'modula-slider' ),
				'default'     => 0,
				'priority'    => 134,
                'data_type' => 'bool',
			),
			'slider_syncing'                             => array(
				'name'        => esc_html__( 'Thumbnail Navigator', 'modula-slider' ),
				'type'        => 'toggle',
				'description' => esc_html__( 'Enables Thumbnail Navigation', 'modula-slider' ),
				'default'     => 0,
				'priority'    => 135,
                'data_type' => 'bool',
			),

			'slider_syncing_nav_size'                    => array(
				'name'        => esc_html__( 'Image Size', 'modula-slider' ),
				'type'        => 'select',
				'description' => esc_html__( 'Select the image sizes for the thumbnail navigator from a selection of options. Select custom if you want to enter the sizes yourself.', 'modula-slider' ),
				'values'      => Modula_Helper::get_image_sizes( true ),
				'default'     => 'default',
				'is_child'    => true,
				'priority'    => 136,
                'data_type' => 'text',
			),

			'slider_syncing_nav_image_dimensions'        => array(
				'name'        => esc_html__( 'Dimensions', 'modula-slider' ),
				'type'        => 'image-size',
				'description' => esc_html__( 'Define thumbnail image dimensions. Width and Height.', 'modula-slider' ),
				'default'     => 0,
				'is_child'    => true,
				'priority'    => 137,
                'data_type' => 'number',
			),

			'slider_syncing_nav_image_crop'              => array(
				'name'        => esc_html__( 'Crop Images ?', 'modula-slider' ),
				'type'        => 'toggle',
				'description' => esc_html__( 'If this is enabled, images will be cropped down to exactly the sizes defined above.', 'modula-slider' ),
				'default'     => 0,
				'is_child'    => true,
				'priority'    => 138,
                'data_type' => 'bool',
			),

			'slider_syncing_nav_thumbnails_number'       => array(
				'name'        => esc_html__( 'Slides to Show', 'modula-slider' ),
				'type'        => 'text',
				'description' => esc_html__( 'Define the number of thumbnails shown in the navigator.', 'modula-slider' ),
				'default'     => '2',
				'is_child'    => true,
				'priority'    => 139,
                'data_type' => 'number',
			),

			'slider_syncing_nav_thumbnails_gutter'       => array(
				'name'        => esc_html__( 'Gutter', 'modula-slider' ),
				'type'        => 'ui-slider',
				'description' => esc_html__( 'Define the distance you want between each thumbnail in your navigator.', 'modula-slider' ),
				'min'         => 0,
				'max'         => 100,
				'step'        => 1,
				'default'     => 10,
				'after'       => 'px',
				'is_child'    => true,
				'priority'    => 140,
                'data_type' => 'number',
			),

			'slider_syncing_nav_thumbnails_border_color' => array(
				'name'        => esc_html__( 'Active Border Color', 'modula-slider' ),
				'type'        => 'color',
				'description' => esc_html__( 'Select the border color for the active thumbnail navigator slider' ),
				'default'     => '#000000',
				'is_child'    => true,
				'priority'    => 141,
                'data_type' => 'text',
			),

			'slider_autoplay'                            => array(
				'name'        => esc_html__( 'Autoplay', 'modula-slider' ),
				'type'        => 'toggle',
				'description' => esc_html__( 'Enables Autoplay for the slider', 'modula-slider' ),
				'default'     => 0,
				'priority'    => 142,
                'data_type' => 'bool',
			),
			'slider_autoplaySpeed'                       => array(
				'name'        => esc_html__( 'Autoplay speed', 'modula-slider' ),
				'type'        => 'text',
				'description' => esc_html__( 'Autoplay speed in milliseconds', 'modula-slider' ),
				'default'     => 3000,
				'is_child'    => true,
				'priority'    => 143,
                'data_type' => 'number',
			),
			'slider_pauseOnHover'                        => array(
				'name'        => esc_html__( 'Pause on hover', 'modula-slider' ),
				'type'        => 'toggle',
				'description' => esc_html__( 'Pause slider when mouse is hovering.', 'modula-slider' ),
				'default'     => 0,
				'is_child'    => true,
				'priority'    => 144,
                'data_type' => 'bool',
			),
			'slider_arrows'                              => array(
				'name'        => esc_html__( 'Arrows', 'modula-slider' ),
				'type'        => 'toggle',
				'description' => esc_html__( 'Toggle slide arrows navigation', 'modula-slider' ),
				'default'     => 1,
				'priority'    => 145,
                'data_type' => 'bool',
			),
			'slider_dots'                                => array(
				'name'        => esc_html__( 'Dots', 'modula-slider' ),
				'type'        => 'toggle',
				'description' => esc_html__( 'Show dot indicators.', 'modula-slider' ),
				'default'     => 0,
				'priority'    => 146,
                'data_type' => 'bool',
			),
			'slider_draggable'                           => array(
				'name'        => esc_html__( 'Mouse dragging', 'modula-slider' ),
				'type'        => 'toggle',
				'description' => esc_html__( 'Enable mouse dragging.', 'modula-slider' ),
				'default'     => 1,
				'priority'    => 147,
                'data_type' => 'bool',
			),
			'slider_centerMode'                          => array(
				'name'        => esc_html__( 'Center mode', 'modula-slider' ),
				'type'        => 'toggle',
				'description' => esc_html__( 'Enables centered view with partial prev/next slides. Only works if "Fade" is disabled.', 'modula-slider' ),
				'default'     => 20,
				'priority'    => 148,
                'data_type' => 'bool',
			),
			'slider_centerPadding'                       => array(
				'name'        => esc_html__( 'Center Padding', 'modula-slider' ),
				'type'        => 'text',
				'description' => 'Side padding when in center mode (px or %). ',
				'default'     => '50px',
				'is_child'    => true,
				'priority'    => 149,
                'data_type' => 'text',
			),
			'slider_fade'                                => array(
				'name'        => esc_html__( 'Fade', 'modula-slider' ),
				'type'        => 'toggle',
				'description' => esc_html__( 'Enable fade. If number of slides on single view( "Slides to show" option ) is bigger than 1 fade will automatically be considered as disabled. ', 'modula-slider' ),
				'default'     => 0,
				'priority'    => 150,
                'data_type' => 'bool',
			),
			'slider_infinite'                            => array(
				'name'        => esc_html__( 'Infinite loop', 'modula-slider' ),
				'type'        => 'toggle',
				'description' => esc_html__( 'Toggle infinite loop.', 'modula-slider' ),
				'default'     => 1,
				'priority'    => 151,
                'data_type' => 'bool',
			),
			'slider_slidesToShow'                        => array(
				'name'        => esc_html__( 'Slides to show', 'modula-slider' ),
				'type'        => 'ui-slider',
				'description' => esc_html__( 'Number of slides to show on single view.', 'modula-slider' ),
				'min'         => 1,
				'max'         => 10,
				'step'        => 1,
				'default'     => 1,
				'priority'    => 152,
                'data_type' => 'number',
			),
			'slider_slidesToScroll'                      => array(
				'name'        => esc_html__( 'Slides to scroll', 'modula-slider' ),
				'type'        => 'ui-slider',
				'description' => esc_html__( 'Number of slides to scroll. "Center mode" must be disabled for this to work.', 'modula-slider' ),
				'min'         => 1,
				'max'         => 10,
				'step'        => 1,
				'default'     => 1,
				'priority'    => 153,
                'data_type' => 'number',
			),
			'slider_speed'                               => array(
				'name'        => esc_html__( 'Animation speed', 'modula-slider' ),
				'type'        => 'text',
				'description' => esc_html__( 'Animation speed in milliseconds', 'modula-slider' ),
				'default'     => 300,
				'priority'    => 154,
                'data_type' => 'number',
			),
			'slider_rtl'                                 => array(
				'name'        => esc_html__( 'Right-to-left', 'modula-slider' ),
				'type'        => 'toggle',
				'description' => esc_html__( 'Change the slider\'s direction to become right-to-left', 'modula-slider' ),
				'default'     => 0,
				'priority'    => 155,
                'data_type' => 'bool',
			),
			'slider_initialSlide'                        => array(
				'name'        => esc_html__( 'Initial slide', 'modula-slider' ),
				'type'        => 'text',
				'description' => esc_html__( 'Slide to start on.', 'modula-slider' ),
				'default'     => 0,
				'priority'    => 156,
                'data_type' => 'number',
			),
			'slider_image_info'                          => array(
				'name'        => esc_html__( 'Slider captions', 'modula-slider' ),
				'description' => esc_html__( 'Display image title and caption under the image.', 'modula-slider' ),
				'type'        => 'toggle',
				'default'     => 0,
				'priority'    => 157,
                'data_type' => 'bool',
			),
		);

		$slider_responsive_fields = array(
			'slider_tablet_slides'  => array(
				'name'        => esc_html__( 'Tablet slides to show ', 'modula-slider' ),
				'description' => esc_html__( 'Number of slides to show on single view in tablet mode.', 'modula-slider' ),
				'type'        => 'ui-slider',
				'min'         => 1,
				'max'         => 12,
				'step'        => 1,
				'default'     => 1,
				'priority'    => 30,
				'is_child'    => true,
                'data_type' => 'number',
			),
			'slider_tablet_scrolls' => array(
				'name'        => esc_html__( 'Tablet slides to scroll', 'modula-slider' ),
				'description' => esc_html__( 'Number of slides to scroll in tablet mode. "Center mode" must be disabled for this to work.', 'modula-slider' ),
				'type'        => 'ui-slider',
				'min'         => 1,
				'max'         => 12,
				'step'        => 1,
				'default'     => 1,
				'is_child'    => true,
				'priority'    => 31,
                'data_type' => 'number',
			),
			'slider_mobile_slides'  => array(
				'name'        => esc_html__( 'Mobile slides to show', 'modula-slider' ),
				'description' => esc_html__( 'Number of slides to show on single view in mobile mode', 'modula-slider' ),
				'type'        => 'ui-slider',
				'min'         => 1,
				'max'         => 12,
				'step'        => 1,
				'default'     => 1,
				'is_child'    => true,
				'priority'    => 32,
                'data_type' => 'number',
			),
			'slider_mobile_scrolls' => array(
				'name'        => esc_html__( 'Mobile slides to scroll', 'modula-slider' ),
				'description' => esc_html__( 'Number of slides to scroll in mobile mode. "Center mode" must be disabled for this to work.', 'modula-slider' ),
				'type'        => 'ui-slider',
				'min'         => 1,
				'max'         => 12,
				'step'        => 1,
				'default'     => 1,
				'is_child'    => true,
				'priority'    => 33,
                'data_type' => 'number',
			),
		);

		$slider_lightbox_fields['slider_lightbox'] = array(
			'name'        => esc_html__( 'Slider Lightbox &amp; Links', 'modula-slider' ),
			'type'        => 'select',
			'description' => esc_html__( 'Choose how the slider should behave on image clicking.', 'modula-slider' ),
			'default'     => 'no-link',
			'values'      => array(
				'no-link'         => esc_html__( 'No link', 'modula-slider' ),
				'direct'          => esc_html__( 'Direct link to image', 'modula-slider' ),
				'attachment-page' => esc_html__( 'Attachment page', 'modula-slider' ),
				'fancybox'        => esc_html__( 'Open Images in a Lightbox', 'modula-slider' ),
			),
			'priority'    => 0,
			'data_type' => 'text',
		);

		$fields['lightboxes'] = array_merge( $fields['lightboxes'], $slider_lightbox_fields );
		$fields['general']    = array_merge( $fields['general'], $slider_fields );
		$fields['responsive'] = array_merge( $fields['responsive'], $slider_responsive_fields );

		return $fields;
	}

	/**
	 * Add image sizes for slider type
	 *
	 * @param $return
	 * @param $id
	 * @param $sizes
	 * @param $crop
	 *
	 * @return mixed
	 */
	public function modula_slider_image_sizes( $return, $id, $sizes, $crop ) {

		if ( is_array( $sizes ) ) {

			if ( ! $crop ) {
				$imageObj = wp_get_attachment_image_src( $id, 'full' );
				$ratio    = (float) ( (int) $imageObj[1] / (int) $imageObj[2] );
				$width    = absint( $sizes['width'] );
				$height   = absint( $sizes['height'] );

				if ( 0 == $width || 0 == $height ) {
					if ( 0 == $width ) {
						$width = $height * $ratio;
					} else {
						$height = $width / $ratio;
					}
				} elseif ( $sizes['width'] / $sizes['height'] != $imageObj[1] / $imageObj[2] ) {

					// keep height
					if ( $sizes['width'] >= $sizes['height'] ) {
						$height = $sizes['width'] / $ratio;
					} else {
						$width = $sizes['height'] * $ratio;
					}
				}

				$return['width']  = (int) $width;
				$return['height'] = (int) $height;
			} else {
				$return['width']  = $sizes['width'];
				$return['height'] = $sizes['height'];
			}
		} else {
			$image_sizes = wp_get_attachment_image_src( $id, $sizes );

			if ( $image_sizes ) {
				$return['width']  = $image_sizes[1];
				$return['height'] = $image_sizes[2];
			}
		}

		return $return;

	}


	public function modula_slider_get_size( $size, $settings ) {

		if ( isset( $settings['type'] ) && 'slider' == $settings['type'] ) {
			$size = ( 'custom' != $settings['slider_image_size'] ) ? $settings['slider_image_size'] : $settings['slider_image_dimensions'];
		}

		return $size;
	}

	/**
	 * Set item data
	 *
	 * @param $item_data
	 * @param $image
	 * @param $settings
	 *
	 * @return mixed
	 */
	public function modula_slider_item_data( $item_data, $image, $settings ) {

		if ( 'slider' == $settings['type'] ) {
			// empty array
			$item_data['item_classes'] = array();
			// add only item class
			$item_data['item_classes'][] = 'modula-item';
			// allow showing social icons no matter the hover effect
			$item_data['hide_socials'] = false;
			// allow showing title no matter the hover effect
			$item_data['hide_title'] = ( '1' == $settings['hide_title'] ) ? true : false;
			// allow showing description no matter the hover effect
			$item_data['hide_description'] = ( '1' == $settings['hide_description'] ) ? true : false;

			// if has link use as clickable link
			if ( $image['link'] != '' ) {
				$item_data['link_attributes']['href'] = $image['link'];
				$item_data['is_custom_url']           = true;
				$item_data['item_classes'][]          = 'modula-simple-link';
			}

			if ( isset( $settings['slider_image_info'] ) && '1' == $settings['slider_image_info'] ) {
				$item_data['slider_image_info'] = '1';
			}

			// As of v1.0.6 we introduce a new setting called slider_lightbox
			// It is used for links in the slider
			if ( isset( $settings['slider_lightbox'] ) ) {
				$item_data['lightbox'] = $settings['slider_lightbox'];
			} else {
				$item_data['lightbox'] = 'no-link';
			}
			// Set the item data custom grid, used for srcset and sizes behaviour.
			if ( 'custom' === $settings['slider_image_size'] ) {
				$item_data['custom_grid'] = true;
			}
			// Set the crop option for the image, used for srcset and sizes behaviour.
			if ( isset( $settings['slider_image_crop'] ) && '1' === $settings['slider_image_crop'] ) {
				$item_data['crop'] = true;
			}
		}
		return $item_data;
	}


	/**
	 * Model image attributes
	 *
	 * @param $image
	 * @param $settings
	 * @return mixed
	 */
	public function modula_slider_image_data( $image, $settings ) {

		$slider_size = Modula_Helper::get_image_sizes( false, $settings['slider_image_size'] );

		// check if selected image size is defined
		if ( $slider_size ) {
			$image['width']  = $slider_size['width'];
			$image['height'] = $slider_size['height'];
		} else {
			if ( 'default' == $settings['slider_image_size'] ) {
				$image['width']  = $settings['slider_image_dimensions']['width'];
				$image['height'] = $settings['slider_image_dimensions']['height'];
			} else {
				$image['width']  = 0;
				$image['height'] = 0;
			}
		}

		return $image;
	}

	/**
	 * Modula image attributes
	 *
	 * @param $item_data
	 * @param $item
	 * @param $settings
	 *
	 * @return mixed
	 */
	public function change_image_size( $item_data, $item, $settings ) {

		if ( ! apply_filters( 'modula_resize_images', true, $settings ) ) {
			return $item_data;
		}

		// We need to change only for gallery type slider
		if ( 'slider' != $settings['type'] ) {
			return $item_data;
		}

		if ( ! class_exists( 'Modula_Image' ) ) {
			return $item_data;
		}

		$resizer = new Modula_Image();

		if ( 'custom' == $settings['slider_image_size'] ) {
			$grid_sizes = array(
				'width'  => $settings['slider_image_dimensions']['width'],
				'height' => $settings['slider_image_dimensions']['height'],
			);
		} else {
			$grid_sizes = $settings['slider_image_size'];
		}

		$crop = false;

		if ( 'custom' == $settings['slider_image_size'] ) {
			$crop = boolval( boolval( $settings['slider_image_crop'] ) );
		}

		$align = 'c';

		if ( 'top' == $item['valign'] ) {
			$valign = 't';
		} elseif ( 'bottom' == $item['valign'] ) {
			$valign = 'b';
		} else {
			$valign = 'c';
		}

		if ( 'left' == $item['halign'] ) {
			$halign = 'l';
		} elseif ( 'right' == $item['halign'] ) {
			$halign = 'r';
		} else {
			$halign = 'c';
		}

		if ( $valign !== $halign ) {
			$align = $valign . $halign;
		}

		$sizes = $resizer->get_image_size( $item['id'], 'slider', $grid_sizes, $crop );

		$resized    = $resizer->resize_image( $sizes['url'], $sizes['width'], $sizes['height'], $crop, $align );
		$image_info = false;

		// If we couldn't resize the image we will return the full image.
		if ( is_wp_error( $resized ) ) {
			$image_url = $sizes['url'];
		} else {
			// Let's check if resize gives us both URL and image info
			// Also, if resized_url is available, image_info should be available
			if ( isset( $resized['resized_url'] ) ) {
				$image_url  = $resized['resized_url'];
				$image_info = $resized['image_info'];
			} else {
				$image_url = $resized;
			}
		}

		$item_data['img_attributes']['width']  = $sizes['width'];
		$item_data['img_attributes']['height'] = $sizes['height'];
		$item_data['image_full']               = $sizes['url'];
		$item_data['image_url']                = ( isset( $sizes['thumb_url'] ) ) ? $sizes['thumb_url'] : $image_url;
		// If thumb_url exists it means we are in predefined sizes
		$item_data['img_attributes']['src']      = ( isset( $sizes['thumb_url'] ) ) ? $sizes['thumb_url'] : $image_url;
		$item_data['img_attributes']['data-src'] = ( isset( $sizes['thumb_url'] ) ) ? $sizes['thumb_url'] : $image_url;
		$item_data['image_info']                 = $image_info;

		if ( $settings['lazy_load'] ) {
			$item_data['img_attributes']['data-lazy'] = $image_url;
			$item_data['img_attributes']['src']       = '';
		}

		return $item_data;

	}


	/**
	 * Remake the gallery classes and add slider_lightbox to config
	 *
	 * @param $template_data
	 *
	 * @return mixed
	 */
	public function modula_gallery_classes( $template_data ) {
		if ( 'slider' == $template_data['settings']['type'] ) {
			$template_data['gallery_container']['class'] = array( 'modula-slider' );

			$js_config           = json_decode( $template_data['gallery_container']['data-config'] );
			$js_config->lightbox = isset( $template_data['settings']['slider_lightbox'] ) ? $template_data['settings']['slider_lightbox'] : 'no-link';

			$template_data['gallery_container']['data-config'] = json_encode( $js_config );
		}

		return $template_data;

	}

	public static function get_real_sizes( $settings ) {

		if ( ! is_array( $settings ) || empty( $settings ) || 'full' == $settings['slider_image_size'] ) {
			$size = array(
				'width'  => 0,
				'height' => 0,
			);
			return $size;
		}

		if ( 'custom' != $settings['slider_image_size'] ) {
			$size = Modula_Helper::get_image_sizes()[ $settings['slider_image_size'] ];
		} else {
			$size = $settings['slider_image_dimensions'];
		}

		return $size;
	}


	/**
	 * We need to overwrite the height but still display the correct css
	 *
	 * @param $css
	 * @param $gallery_id
	 * @param $settings
	 * @return mixed
	 */
	public function modula_slider_output_css( $css, $gallery_id, $settings ) {

		if ( 'slider' == $settings['type'] ) {

			$width = ( ! empty( $settings['width'] ) ) ? $settings['width'] : '100%';
			$css  .= "#{$gallery_id}.modula-slider { max-width: {$width}; margin: 0 auto; }";
			$css  .= "#{$gallery_id}.modula-slider .modula-item .video-sizer iframe { min-width:100% }";

			// set the height to auto
			$css .= "#{$gallery_id} .modula-items{height:auto}";
			// remove the hover background color
			$css .= "#{$gallery_id} .modula-item{background-color:transparent}";
			// remove loading effects
			$css .= "#{$gallery_id} .modula-item{transform:none;}";
			// remove hover opacity on image
			$css .= "#{$gallery_id} .modula-item:hover img{opacity:1;}";
			$css .= "#{$gallery_id} .modula-slider-nav .slick-list .slick-track .slick-slide { margin: " . absint( $settings['slider_syncing_nav_thumbnails_gutter'] / 2 ) . 'px}';
			// $css .= "#{$gallery_id} .modula-slider-nav .slick-list { margin: -" . absint( $settings['slider_syncing_nav_thumbnails_gutter'] / 2 ) .  "px}";

			$css .= "#{$gallery_id} .slider-clickable { position:relative;width:max-content;margin:0 auto;max-width:100%; }#{$gallery_id} .slider-clickable .modula-item-link, #{$gallery_id}  .slider-clickable .modula-simple-link { position:absolute; width:100%; height:100%; cursor:pointer; z-index:999;}";

			// Gutter only for < 1 slidesToShow
			if ( 1 == $settings['slider_slidesToShow'] && ( ! isset( $settings['slider_centerMode'] ) || '1' != $settings['slider_centerMode'] ) ) {
				$css .= $css .= "#{$gallery_id} .modula-items .slick-list .slick-track .slick-slide { margin: 0}";
			} else {
				$css .= "#{$gallery_id} .modula-items .slick-list .slick-track .slick-slide { margin: 0 " . absint( $settings['gutter'] / 2 ) . 'px}';
				// $css .= "#{$gallery_id} .modula-items .slick-list { margin: 0 -" . absint( $settings['gutter'] / 2 ) .  "px}";
			}

			// arrow fix.
			if ( isset( $settings['slider_arrows'] ) && '1' === $settings['slider_arrows'] ) {
				$css .= "#{$gallery_id} .modula-items .slick-list { width :calc(100% - 40px); margin: 0 auto; }";
				$css .= "#{$gallery_id} .slider-nav .slick-list { width :calc(100% - 40px); margin: 0 auto; }";
			}

			// Slider dimensions for video
			$real_sizes = self::get_real_sizes( $settings );
			$min_height = esc_attr( '50vh' );
			if ( 0 == $real_sizes['width'] ) {
				$width = esc_attr( '100%' );
			} else {
				$min_height = esc_attr( 'auto' );
				$width      = absint( $real_sizes['width'] ) . 'px';
			}

			$css .= "#{$gallery_id} .modula-items .slick-slide .modula-item .video-sizer { width:" . $width . '; max-width:100%;}';
			$css .= "#{$gallery_id} .modula-items .slick-slide .modula-item .video-sizer iframe, #{$gallery_id}  .modula-items .slick-slide .modula-item .video-sizer video { height:" . $width . '!important; position: relative;max-height:100vh;min-height:' . $min_height . ' }';

			if ( ! isset( $settings['hide_title'] ) || '1' != $settings['hide_title'] ) {

				$css .= "#{$gallery_id}.modula-slider .modula-items .slider-image-info .jtg-title {margin-bottom:0px;}";

				if ( '' != $settings['titleColor'] ) {
					$css .= "#{$gallery_id}.modula-slider .modula-items .slider-image-info .jtg-title {";
					$css .= 'color:' . Modula_Helper::sanitize_rgba_colour( $settings['titleColor'] ) . ';';
					$css .= '}';
				}

				if ( '' != $settings['titleFontSize'] && '0' != $settings['titleFontSize'] ) {
					$css .= "#{$gallery_id}.modula-slider .modula-items .slider-image-info .jtg-title {";
					$css .= 'font-size:' . absint( $settings['titleFontSize'] ) . 'px;';
					$css .= '}';
				}

				// Settings for font family caption and title
				if ( 'Default' != $settings['titleFontFamily'] ) {
					$css .= "#{$gallery_id} .jtg-title{ font-family:" . esc_attr( $settings['titleFontFamily'] ) . '; }';
				}

				// Settings for Title Font Weight
				if ( 'default' != $settings['titleFontWeight'] ) {
					$css .= "#{$gallery_id} .jtg-title {font-weight:" . esc_attr( $settings['titleFontWeight'] ) . '; }';
				}

				if ( '' != $settings['mobileTitleFontSize'] && '0' != $settings['mobileTitleFontSize'] ) {
					$css .= "@media all and (max-width:600px){#{$gallery_id}.modula-slider .modula-items .slider-image-info .description {";
					$css .= 'font-size:' . absint( $settings['mobileCaptionFontSize'] ) . 'px;';
					$css .= '}}';
				}
			}

			if ( ! isset( $settings['hide_description'] ) || '1' != $settings['hide_description'] ) {

				$css .= "#{$gallery_id}.modula-slider .modula-items .slider-image-info .description {margin-bottom:0px;}";

				if ( '' != $settings['captionColor'] ) {
					$css .= "#{$gallery_id}.modula-slider .modula-items .slider-image-info .description {";
					$css .= 'color:' . Modula_Helper::sanitize_rgba_colour( $settings['captionColor'] ) . ';';
					$css .= '}';
				}

				if ( '' != $settings['captionFontSize'] && '0' != $settings['captionFontSize'] ) {
					$css .= "#{$gallery_id}.modula-slider .modula-items .slider-image-info .description {";
					$css .= 'font-size:' . absint( $settings['captionFontSize'] ) . 'px;';
					$css .= '}';
				}

				// Settings for font family caption and title
				if ( 'Default' != $settings['captionsFontFamily'] ) {
					$css .= "#{$gallery_id}.modula-slider .modula-items .slider-image-info .description{ font-family:" . esc_attr( $settings['captionsFontFamily'] ) . '; }';
				}

				// Settings for Captions Font Weight
				if ( 'default' != $settings['captionFontWeight'] ) {
					$css .= "#{$gallery_id}.modula-slider .modula-items .slider-image-info .description{font-weight:" . esc_attr( $settings['captionFontWeight'] ) . '; }';
				}

				if ( '' != $settings['mobileCaptionFontSize'] && '0' != $settings['mobileCaptionFontSize'] ) {
					$css .= "@media all and (max-width:600px){#{$gallery_id}.modula-slider .modula-items .slider-image-info .description {";
					$css .= 'font-size:' . absint( $settings['mobileCaptionFontSize'] ) . 'px;';
					$css .= '}}';
				}
			}

			if ( isset( $settings['slider_syncing_nav_thumbnails_border_color'] ) ) {
				$css .= '#' . $gallery_id . '.modula-slider .modula-slider-nav .slick-slide.is-active img{border: 2px solid ' . Modula_Helper::sanitize_rgba_colour( $settings['slider_syncing_nav_thumbnails_border_color'] ) . '}';

			}
		}

		return $css;
	}


	/**
	 * Hide used filters
	 *
	 * @param $template_data
	 *
	 * @return mixed
	 */
	public function modula_slider_settings( $template_data ) {
		if ( 'slider' == $template_data['settings']['type'] ) {
			$template_data['settings']['filters'] = array();
		}

		return $template_data;
	}


	/**
	 * Add attributes to 'items' element wrapper
	 *
	 * @param $template_data
	 *
	 * @return string
	 */
	public function modula_items_attribute_config( $template_data ) {

		if ( 'slider' == $template_data['settings']['type'] ) {
			if ( '1' == $template_data['settings']['slider_rtl'] ) {
				$template_data['items_container']['dir'] = 'rtl';
			} else {
				$template_data['items_container']['dir'] = 'ltr';
			}
		}

		return $template_data;
	}


	/**
	 * Set defaults
	 *
	 * @param $defaults
	 * @return mixed
	 */
	public function set_defaults( $defaults ) {
		$defaults['slider_adaptiveHeight']                      = 0;
		$defaults['slider_image_size']                          = 'default';
		$defaults['slider_image_dimensions']                    = 0;
		$defaults['slider_image_height']                        = '800';
		$defaults['slider_syncing']                             = 0;
		$defaults['slider_syncing_nav_size']                    = 'thumbnail';
		$defaults['slider_syncing_nav_thumbnails_number']       = '2';
		$defaults['slider_syncing_nav_thumbnails_gutter']       = 10;
		$defaults['slider_syncing_nav_thumbnails_border_color'] = '#000000';
		$defaults['slider_autoplay']                            = 0;
		$defaults['slider_autoplaySpeed']                       = 3000;
		$defaults['slider_arrows']                              = 1;
		$defaults['slider_centerMode']                          = 1;
		$defaults['slider_centerPadding']                       = '50px';
		$defaults['slider_dots']                                = 0;
		$defaults['slider_draggable']                           = 1;
		$defaults['slider_fade']                                = 0;
		$defaults['slider_infinite']                            = 1;
		$defaults['slider_pauseOnHover']                        = 1;
		$defaults['slider_slidesToShow']                        = 1;
		$defaults['slider_slidesToScroll']                      = 1;
		$defaults['slider_initialSlide']                        = 0;
		$defaults['slider_speed']                               = 300;
		$defaults['slider_rtl']                                 = 0;
		$defaults['slider_initialSlide']                        = 0;
		$defaults['slider_image_crop']                          = 0;
		$defaults['slider_tablet_slides']                       = 1;
		$defaults['slider_tablet_scrolls']                      = 1;
		$defaults['slider_mobile_slides']                       = 1;
		$defaults['slider_mobile_scrolls']                      = 1;
		$defaults['slider_lightbox']                            = 'no-link';
		$defaults['slider_syncing_nav_image_dimensions']        = 0;
		$defaults['slider_syncing_nav_image_crop']              = 0;
		$defaults['slider_image_info']                          = 0;

		return $defaults;
	}

	public function js_slider_config( $js_config, $settings ) {

		if ( 'slider' == $js_config['type'] ) {
			$js_config['slider_settings'] = array(
				'adaptiveHeight' => boolval( $settings['slider_adaptiveHeight'] ),
				'autoplay'       => boolval( $settings['slider_autoplay'] ),
				'autoplaySpeed'  => ( 0 <= absint( $settings['slider_autoplaySpeed'] ) ) ? absint( $settings['slider_autoplaySpeed'] ) : 3000,
				'arrows'         => boolval( $settings['slider_arrows'] ),
				'centerMode'     => boolval( $settings['slider_centerMode'] ),
				'centerPadding'  => $settings['slider_centerPadding'],
				'dots'           => boolval( $settings['slider_dots'] ),
				'draggable'      => boolval( $settings['slider_draggable'] ),
				'fade'           => ( absint( $settings['slider_slidesToShow'] ) > 1 ) ? false : boolval( $settings['slider_fade'] ),
				'navImageShow'   => absint( $settings['slider_syncing_nav_thumbnails_number'] ),
				'infinite'       => boolval( $settings['slider_infinite'] ),
				'pauseOnHover'   => boolval( $settings['slider_pauseOnHover'] ),
				'slidesToShow'   => absint( $settings['slider_slidesToShow'] ),
				'slidesToScroll' => absint( $settings['slider_slidesToScroll'] ),
				'speed'          => absint( $settings['slider_speed'] ),
				'rtl'            => boolval( $settings['slider_rtl'] ),
				'initialSlide'   => ( $settings['slider_initialSlide'] > 0 ) ? (int) $settings['slider_initialSlide'] - 1 : (int) $settings['slider_initialSlide'],
				'prevArrow'      => '<a href="#" class="slick-prev"><svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="angle-left" class="svg-inline--fa fa-angle-left fa-w-8" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 512"><path fill="currentColor" d="M31.7 239l136-136c9.4-9.4 24.6-9.4 33.9 0l22.6 22.6c9.4 9.4 9.4 24.6 0 33.9L127.9 256l96.4 96.4c9.4 9.4 9.4 24.6 0 33.9L201.7 409c-9.4 9.4-24.6 9.4-33.9 0l-136-136c-9.5-9.4-9.5-24.6-.1-34z"></path></svg></a>',
				'nextArrow'      => '<a href="#" class="slick-next"><svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="angle-right" class="svg-inline--fa fa-angle-right fa-w-8" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 512"><path fill="currentColor" d="M224.3 273l-136 136c-9.4 9.4-24.6 9.4-33.9 0l-22.6-22.6c-9.4-9.4-9.4-24.6 0-33.9l96.4-96.4-96.4-96.4c-9.4-9.4-9.4-24.6 0-33.9L54.3 103c9.4-9.4 24.6-9.4 33.9 0l136 136c9.5 9.4 9.5 24.6.1 34z"></path></svg></a>',
				'showCount'      => apply_filters( 'modula_show_slider_numbers', 0, $settings ),
			);

			if ( $settings['slider_rtl'] ) {
				$js_config['slider_settings']['prevArrow'] = '<a href="#" class="slick-prev"><svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="angle-right" class="svg-inline--fa fa-angle-right fa-w-8" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 512"><path fill="currentColor" d="M224.3 273l-136 136c-9.4 9.4-24.6 9.4-33.9 0l-22.6-22.6c-9.4-9.4-9.4-24.6 0-33.9l96.4-96.4-96.4-96.4c-9.4-9.4-9.4-24.6 0-33.9L54.3 103c9.4-9.4 24.6-9.4 33.9 0l136 136c9.5 9.4 9.5 24.6.1 34z"></path></svg></a>';
				$js_config['slider_settings']['nextArrow'] = '<a href="#" class="slick-next"><svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="angle-left" class="svg-inline--fa fa-angle-left fa-w-8" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 512"><path fill="currentColor" d="M31.7 239l136-136c9.4-9.4 24.6-9.4 33.9 0l22.6 22.6c9.4 9.4 9.4 24.6 0 33.9L127.9 256l96.4 96.4c9.4 9.4 9.4 24.6 0 33.9L201.7 409c-9.4 9.4-24.6 9.4-33.9 0l-136-136c-9.5-9.4-9.5-24.6-.1-34z"></path></svg></a>';
			}

			if ( $settings['lazy_load'] ) {
				if ( isset( $settings['slider_adaptiveHeight'] ) && 1 != $settings['slider_adaptiveHeight'] ) {
					$js_config['slider_settings']['lazyLoad'] = 'ondemand';
				} else {
					$js_config['slider_settings']['lazyLoad'] = 'progressive';
				}
			}

			if ( $js_config['slider_settings']['centerMode'] ) {
				$js_config['slider_settings']['centerPadding'] = esc_attr( $settings['slider_centerPadding'] );
			}

			if ( isset( $settings['enable_responsive'] ) && '1' == $settings['enable_responsive'] ) {
				$responsive = array(
					array(
						'breakpoint' => 1024,
						'settings'   => array(
							'slidesToShow'   => ( isset( $settings['slider_tablet_slides'] ) && 0 <= $settings['slider_tablet_scrolls'] ) ? absint( $settings['slider_tablet_slides'] ) : 1,
							'slidesToScroll' => ( isset( $settings['slider_tablet_scrolls'] ) && 0 <= $settings['slider_tablet_scrolls'] ) ? absint( $settings['slider_tablet_scrolls'] ) : 1,
						),
					),
					array(
						'breakpoint' => 600,
						'settings'   => array(
							'slidesToShow'   => ( isset( $settings['slider_mobile_slides'] ) && 0 <= $settings['slider_tablet_scrolls'] ) ? absint( $settings['slider_mobile_slides'] ) : 1,
							'slidesToScroll' => ( isset( $settings['slider_mobile_scrolls'] ) && 0 <= $settings['slider_tablet_scrolls'] ) ? absint( $settings['slider_mobile_scrolls'] ) : 1,
						),
					),
				);

				$js_config['slider_settings']['responsive'] = $responsive;
			}
		}

		return $js_config;

	}


	public function change_effect( $template_data ) {

		if ( 'slider' != $template_data['settings']['type'] ) {
			return $template_data;
		}

		$template_data['settings']['effect'] = 'slider';
		return $template_data;

	}

	/**
	 * Add extra path for modula templates
	 *
	 * @param $paths
	 *
	 * @return mixed
	 */
	public function add_modula_slider_templates_path( $paths ) {

		$paths[60] = trailingslashit( MODULA_SLIDER_PATH ) . '/templates';
		return $paths;
	}

	/**
	 * Render Thumbnail navigation
	 *
	 * @param $settings
	 * @param $item_data
	 * @param $images
	 *
	 * @return void $html
	 */
	public function modula_slider_syncing( $settings, $item_data, $images ) {

		if ( 'slider' == $settings['type'] && 1 == $settings['slider_slidesToShow'] ) {

			if ( 1 == $settings['slider_syncing'] ) {

				if ( ! empty( $images ) ) {
					global $_wp_additional_image_sizes;

					$slider_width = '';
					if ( 'custom' == $settings['slider_syncing_nav_size'] ) {
						$slider_width = $settings['slider_syncing_nav_image_dimensions']['width'] * $settings['slider_syncing_nav_thumbnails_number'] + $settings['gutter'] - 10;
					} else {
						$width = 100;
						if ( isset( $_wp_additional_image_sizes[ $settings['slider_syncing_nav_size'] ] ) ) {
							$width = $_wp_additional_image_sizes[ $settings['slider_syncing_nav_size'] ]['width'];
						}
						$slider_width = $width * $settings['slider_syncing_nav_thumbnails_number'] + $settings['gutter'] - 10;
					}

					$config = array(
						'slidesToShow'   => absint( $settings['slider_syncing_nav_thumbnails_number'] ),
						'slidesToScroll' => absint( $settings['slider_slidesToScroll'] ),
						'focusOnSelect'  => false,
						'focusOnChange'  => false,
						'dots'           => false,
						'infinite'       => false,
						'prevArrow'      => '<a href="#" class="slick-prev"><svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="angle-left" class="svg-inline--fa fa-angle-left fa-w-8" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 512"><path fill="currentColor" d="M31.7 239l136-136c9.4-9.4 24.6-9.4 33.9 0l22.6 22.6c9.4 9.4 9.4 24.6 0 33.9L127.9 256l96.4 96.4c9.4 9.4 9.4 24.6 0 33.9L201.7 409c-9.4 9.4-24.6 9.4-33.9 0l-136-136c-9.5-9.4-9.5-24.6-.1-34z"></path></svg></a>',
						'nextArrow'      => '<a href="#" class="slick-next"><svg aria-hidden="true" focusable="false" data-prefix="fas" data-icon="angle-right" class="svg-inline--fa fa-angle-right fa-w-8" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 512"><path fill="currentColor" d="M224.3 273l-136 136c-9.4 9.4-24.6 9.4-33.9 0l-22.6-22.6c-9.4-9.4-9.4-24.6 0-33.9l96.4-96.4-96.4-96.4c-9.4-9.4-9.4-24.6 0-33.9L54.3 103c9.4-9.4 24.6-9.4 33.9 0l136 136c9.5 9.4 9.5 24.6.1 34z"></path></svg></a>',
					);

					echo "<div class='modula-slider-nav' style='width:" . absint( $slider_width ) . "px; max-width: 100%;' data-config='" . esc_attr( json_encode( $config ) ) . "'>";
					foreach ( $images as $index => $image ) {
						$thumbnail_src = '';

						if ( 'custom' != $settings['slider_syncing_nav_size'] ) {
							$thumb = wp_get_attachment_image_src( $image['id'], $settings['slider_syncing_nav_size'] );

							if ( $thumb ) {
								$thumbnail_src = $thumb[0];
							}
						} else {
							$image_full = wp_get_attachment_image_src( $image['id'], 'full' );
							$width      = $settings['slider_syncing_nav_image_dimensions']['width'];
							$height     = $settings['slider_syncing_nav_image_dimensions']['height'];
							$crop       = boolval( $settings['slider_syncing_nav_image_crop'] );

							if ( ! $image_full ) {
								break;
							}

							$resizer    = new Modula_Image();
							$resized    = $resizer->resize_image( $image_full[0], $width, $height, $crop );
							$image_info = false;

							// Let's check if resize gives us both URL and image info
							// Also, if resized_url is available, image_info should be available
							if ( isset( $resized['resized_url'] ) ) {
								$image_url  = $resized['resized_url'];
								$image_info = $resized['image_info'];
							} else {
								$image_url = $resized;
							}

							if ( is_wp_error( $image_url ) ) {
								$image_url = $image_full[0];
							}

							$thumbnail_src = $image_url;
						}

						echo '<div class="modula-slider-nav-item">';
						do_action( 'modula_slider_before_thumbnail', $image, $images, $settings );
						echo "<img class='modula-slider-nav-image' src='" . $thumbnail_src . "'>";
						do_action( 'modula_slider_after_thumbnail', $image, $images, $settings );
						echo '</div>';

					}
					echo '</div>';

				}
			}
		}
	}


	/**
	 * Sanitize slider fields
	 *
	 * @param $sanitized_value
	 * @param $value
	 * @param $field_id
	 * @param $field
	 *
	 * @return mixed
	 *
	 * @since 1.0.3
	 */
	public function sanitize_slider_fields( $sanitized_value, $value, $field_id, $field ) {

		if ( 'slider_image_dimensions' == $field_id ) {
			$new_value = array();
			if ( is_array( $value ) && ! empty( $value ) ) {
				foreach ( $value as $key => $img_size ) {
					$new_value[ $key ] = absint( $img_size );
				}
			}

			return $new_value;
		}

		return $sanitized_value;

	}

	/**
	 * Modula migration data
	 *
	 * @param $modula_settings
	 * @param $guest_settings
	 * @param $guest
	 *
	 * @return mixed
	 * @since 1.0.5
	 */
	public function modula_migrate_data( $modula_settings, $guest_settings, $guest ) {

		switch ( $guest ) {
			case 'foogallery':
				if ( 'image-viewer' == $guest_settings['grid_type'] ) {
					$modula_settings['type']                    = 'slider';
					$modula_settings['slider_image_size']       = 'custom';
					$modula_settings['slider_image_dimensions'] = array(
						'width'  => $guest_settings['image-viewer_thumbnail_size']['width'],
						'height' => $guest_settings['image-viewer_thumbnail_size']['height'],
					);
				}

				break;
		}

		return $modula_settings;
	}

	public function enqueue_modula_block_styles() {
		wp_enqueue_style( 'modula-slider-css', MODULA_SLIDER_URL . 'assets/css/front.min.css' );
	}

	public function enqueue_modula_block_scripts() {
		wp_enqueue_script( 'slick-js', MODULA_SLIDER_URL . 'assets/js/slick.min.js', true );
	}

	/**
	 * Adds modula slider troubleshooting field
	 *
	 * @param $fields
	 * 
	 * @return array
	 * @since 1.1.2
	*/
	public function add_troubleshooting_fields( $fields ) {

		$fields['slider_scripts'] = array(
			'label'       => esc_html__( 'Slider Scripts & Styles', 'modula-slider' ),
			'description' => esc_html__( 'Check this if you\'re using slider type galleries', 'modula-slider' ),
			'type'        => 'toggle',
			'class'       => array( 'troubleshoot-subfield' ),
			'priority'    => 50,
		);

		return $fields;

	}

	/**
	 * Default values for modula slider troubleshooting field
	 *
	 * @param array $defaults Array of default values.
	 *
	 * @return array
	 * @since 1.1.2
	 */
	public function add_troubleshooting_defaults( $defaults ) {
		$defaults['slider_scripts'] = false;

		return $defaults;
	}

	/**
	 * Files to be enqueued globally
	 *
	 * @param array $handles Array of handles to be enqueued.
	 * @param array $options Array of options.
	 *
	 * @return array
	 * @since 1.1.2
	 */
	public function add_main_slider_files( $handles, $options ) {

		if ( $options['slider_scripts'] ) {
			$handles['scripts'][] = 'modula-slider-script';
			$handles['scripts'][] = 'modula-pro';
			$handles['styles'][]  = 'modula-slider-css';
		}

		return $handles;
	}

	/**
	 * Add slider to allowed shuffle gallery types
	 *
	 * @param $allowed_types
	 * 
	 * @return array
	 * @since 1.1.2
	*/
	public function add_slider_shuffle( $allowed_types ) {
		$allowed_types[] = 'slider';
		
		return $allowed_types;
	}
	
}
