<?php

/**
 *
 */
class Modula_Watermark {

	/**
	 * Holds the class object.
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public static $instance;

	/**
	 * The name of the plugin.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $plugin_name = 'Modula Watermark';

	/**
	 * Unique plugin slug identifier.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $plugin_slug = 'modula-watermark';

	/**
	 * Primary class constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		// Load the plugin textdomain.
		add_action( 'init', array( $this, 'set_locale' ) );

		// Set admin requirements on admin_init so that we won't trigger get_packages from WPChill_Upsells on frontend
		add_action( 'admin_init', array( $this, 'set_admin_requirements' ), 25 );

		// Filter Modula Watermark Scripts & Styles
		add_filter( 'admin_enqueue_scripts', array( $this, 'modula_watermark_scripts' ), 25 );

		// Add ajax_button field type used for ajax
		add_filter( 'modula_render_ajax_button_field_type', array( $this, 'ajax_button_field_type' ), 10, 5 );

		// Add image_upload field type
		add_filter( 'modula_render_image_upload_field_type', array( $this, 'image_upload_field_type' ), 10, 5 );

		// Add defaults
		add_filter( 'modula_lite_default_settings', array( $this, 'add_watermark_defaults' ), 10, 5 );

		add_action( 'admin_init', array( $this, 'register_watermark_option' ) );

		add_filter( 'modula_uninstall_db_options', array( $this, 'watermark_uninstall' ), 16, 1 );

		add_action( 'modula_defaults_scripts_after_wp_modula', array( $this, 'watermark_defaults_script' ) );

		add_filter( 'modula_migrate_gallery_data', array( $this, 'watermark_migrator_gallery_data' ), 25, 3 );

		// Add debug information
		add_filter( 'modula_debug_information', array( $this, 'debug_information' ), 45 );

		register_activation_hook( MODULA_WATERMARK_FILE, array( $this, 'plugin_activation' ) );

		// Load the plugin.
		$this->init();

	}

	/**
	 * Set admin requirements and check for license.
	 *
	 * @return void
	 * @since 1.0.7
	 */
	public function set_admin_requirements() {

		if ( class_exists( 'WPChill_Upsells' ) ) {

			$args           = apply_filters(
				'modula_upsells_args',
				array(
					'shop_url' => 'https://wp-modula.com',
					'slug'     => 'modula',
				)
			);

			$wpchill_upsell = WPChill_Upsells::get_instance( $args );

			// Show settings and tabs only if Watermark is not an upgradable addon.
			if ( $wpchill_upsell && ! $wpchill_upsell->is_upgradable_addon( 'modula-watermark' ) ) {

				// Filter Modula Watermark Tab
				add_filter( 'modula_gallery_tabs', array( $this, 'modula_watermark_tabs' ), 99 );

				// Filter Modula Watermark Fields
				add_filter( 'modula_gallery_fields', array( $this, 'modula_watermark_fields' ) );

				// Add Setings Tab
				add_filter( 'modula_admin_page_tabs', array( $this, 'add_watermark_tab' ) );
				add_action( 'modula_admin_tab_watermark', array( $this, 'render_watermark_tab' ) );

				// Ajax for applying watermark
				add_action( 'wp_ajax_modula_remove_watermark', array( $this, 'modula_remove_watermark' ), 10, 2 );
				add_action( 'wp_ajax_modula_apply_single_watermark', array( $this, 'modula_apply_watermark' ), 10, 2 );

				require MODULA_WATERMARK_PATH . 'includes/class-modula-watermarking.php';
			}
		}
	}


	/**
	 * Prevent plugin activation if Modula Pro is not installed and activated
	 *
	 * @since 1.0.1
	 */
	public function plugin_activation() {
		if ( ! class_exists( 'Modula_PRO' ) ) {
			deactivate_plugins( plugin_basename( MODULA_WATERMARK_FILE ) );
			wp_die( esc_html__( 'Please install and activate Modula Pro before using Modula Watermark add-on.', 'modula-watermark' ) );

		}

	}


	/**
	 * Loads the plugin textdomain for translation.
	 *
	 * @since 1.0.0
	 */
	public function set_locale() {
		load_plugin_textdomain( $this->plugin_slug, false, MODULA_WATERMARK_TRANSLATE );
	}

	/**
	 * Loads the plugin into WordPress.
	 *
	 * @since 1.0.0
	 */
	public function init() {

		// Load admin only components.
		if ( is_admin() ) {
			add_action( 'modula_pro_updater', array( $this, 'addon_updater' ), 15, 2 );
		}

	}

	public function addon_updater( $license_key, $store_url ) {

		if ( class_exists( 'Modula_Pro_Base_Updater' ) ) {
			$modula_addon_updater = new Modula_Pro_Base_Updater(
				$store_url,
				MODULA_WATERMARK_FILE,
				array(
					'version' => MODULA_WATERMARK_VERSION,        // current version number
					'license' => $license_key,               // license key (used get_option above to retrieve from DB)
					'item_id' => 313091,                      // ID of the product
					'author'  => 'MachoThemes',            // author of this plugin
					'beta'    => false,
				)
			);
		}
	}


	/**
	 * Returns the singleton instance of the class.
	 *
	 * @return object The Modula_Watermark object.
	 * @since 1.0.0
	 */
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Modula_Watermark ) ) {
			self::$instance = new Modula_Watermark();
		}

		return self::$instance;

	}


	/**
	 * @param $scripts
	 * @param $settings
	 *
	 * @return array
	 *
	 * Enqueue watermark script
	 */
	public function modula_watermark_scripts( $hook ) {

		if ( 'modula-gallery_page_modula' == $hook ) {

			if ( isset( $_GET['modula-tab'] ) && 'watermark' == $_GET['modula-tab'] ) {
				wp_enqueue_media();

				if ( defined( 'MODULA_URL' ) ) {
					wp_enqueue_style( 'jquery-ui', MODULA_URL . 'assets/css/admin/jquery-ui.min.css', null, MODULA_LITE_VERSION );
				}

				wp_enqueue_script( 'modula-watermark-settings', MODULA_WATERMARK_URL . 'assets/js/modula-watermark-settings.js', array( 'jquery', 'jquery-ui-core', 'jquery-ui-slider' ), MODULA_WATERMARK_VERSION, true );
			}
		}

		// Register Watermark script
		wp_register_script( 'modula-watermark-script', MODULA_WATERMARK_URL . 'assets/js/modula-watermark.js', array( 'jquery', 'modula-gallery' ), MODULA_WATERMARK_VERSION, true );
		// Register Watermark style
		wp_register_style( 'modula-watermark-style', MODULA_WATERMARK_URL . 'assets/css/modula-watermark.css' );

		if ( 'post-new.php' == $hook || 'post.php' == $hook ) {

			wp_enqueue_style( 'modula-watermark-style' );
			wp_enqueue_script( 'modula-watermark-script' );

			$modula_wm = array(
				'upd_images'        => esc_html__( 'Watermarking image %s1 of %s2...', 'modula-watermark' ),
				'upd_done'          => esc_html__( 'Finished watermarking images.', 'modula-watermark' ),
				'err_no_images'     => esc_html__( 'Add images and save the gallery before trying to watermark.', 'modula-watermark' ),
				'restore_images'    => esc_html__( 'Restoring image %s1 of %s2...', 'modula-watermark' ),
				'restore_done'      => esc_html__( 'Finished restoring images.', 'modula-watermark' ),

			);
			$modula_wm = apply_filters( 'modula_watermarking_response_messages', $modula_wm );
			wp_localize_script( 'modula-watermark-script', 'modulaWmMessages', $modula_wm );
			wp_add_inline_script( 'modula-watermark-script', 'const watermarkNonce = "' . wp_create_nonce( 'modula_watermark_images' ) . '"', 'before' );

		}

	}

	/**
	 * Enqueue Modula Defaults neede scripts
	 *
	 * @since 1.0.2
	 */
	public function watermark_defaults_script() {
		wp_enqueue_script( 'modula-watermark-script', MODULA_WATERMARK_URL . 'assets/js/modula-watermark.js', array(), MODULA_WATERMARK_VERSION, true );
	}

	/**
	 * Add Modula Watermark Tab
	 *
	 * @param $tabs
	 *
	 * @return mixed
	 */
	public function modula_watermark_tabs( $tabs ) {

		if ( ! isset( $tabs['watermark'] ) ) {
			$tabs['watermark'] = array(
				'label'    => esc_html__( 'Watermark', 'modula-watermark' ),
				'icon'     => 'dashicons dashicons-id-alt',
				'priority' => 100,
			);
		}

		unset( $tabs['watermark']['badge'] );

		return $tabs;
	}

	/**
	 * Add fields for Modula Watermark tab
	 *
	 * @param $fields
	 *
	 * @return mixed
	 */
	public function modula_watermark_fields( $fields ) {

		// Add filters settings
		$fields['watermark'] = array(
			'enable_watermark'                   => array(
				'name'        => esc_html__( 'Enable Watermark', 'modula-watermark' ),
				'type'        => 'toggle',
				'default'     => 0,
				'description' => __( 'Enable watermark for modula gallery images, adding a watermark on the images.', 'modula-watermark' ),
				'priority'    => 10,
				'children'    => array( 'custom_settings_watermark', 'watermark_apply_to_existing_images', 'watermark_remove_watermark' ),
			),
			'custom_settings_watermark'          => array(
				'name'        => esc_html__( 'Use custom settings', 'modula-watermark' ),
				'type'        => 'toggle',
				'default'     => 0,
				'description' => __( 'Add custom settings and don\'t use default settings.', 'modula-watermark' ),
				'priority'    => 20,
				'children'    => array( 'watermark_image', 'watermark_position', 'watermark_margin', 'watermark_image_dimension_width', 'watermark_image_dimension_height', 'watermark_enable_backup' ),
				'parent'      => 'enable_watermark',
			),
			'watermark_image'                    => array(
				'name'        => esc_html__( 'Watermark Image', 'modula-watermark' ),
				'type'        => 'image_upload',
				'default'     => 0,
				'description' => __( 'Enable watermark for modula gallery images, adding a watermark on the images.', 'modula-watermark' ),
				'class'       => 'button insert-media-url',
				'button_text' => esc_html__( 'Upload Watermark', 'modula-watermark' ),
				'is_child'    => true,
				'priority'    => 30,
			),
			'watermark_position'                 => array(
				'name'        => esc_html__( 'Watermark Position', 'modula-watermark' ),
				'type'        => 'select',
				'values'      => array(
					'top_left'     => esc_html__( 'Top left', 'modula-watermark' ),
					'top_right'    => esc_html__( 'Top right', 'modula-watermark' ),
					'bottom_right' => esc_html__( 'Bottom right', 'modula-watermark' ),
					'bottom_left'  => esc_html__( 'Bottom left', 'modula-watermark' ),
					'center'       => esc_html__( 'Center', 'modula-watermark' ),
				),
				'default'     => 'bottom_right',
				'description' => __( 'Choose a position where the watermark will be applied.', 'modula-watermark' ),
				'is_child'    => true,
				'priority'    => 40,
			),
			'watermark_margin'                   => array(
				'name'        => esc_html__( 'Watermark Margin', 'modula-watermark' ),
				'type'        => 'ui-slider',
				'description' => __( 'Set the space between the watermark and the margins of the image.', 'modula-watermark' ),
				'is_child'    => true,
				'priority'    => 50,
				'min'         => 0,
				'max'         => 50,
				'step'        => 1,
				'default'     => 10,
			),
			'watermark_image_dimension_width'    => array(
				'name'        => esc_html__( 'Watermark Image Width', 'modula-watermark' ),
				'type'        => 'text',
				'description' => __( 'Specify the watermark image dimensions. If one of the sizes (width or height) is "0" or "" and the other is different than "0" or "" the watermark image will be scaled proportionally based on it\'s size. If both values are not empty or not "0" the watermark image will be morphed to the specified sizes.', 'modula-watermark' ),
				'is_child'    => true,
				'priority'    => 50,
				'default'     => 0,
			),
			'watermark_image_dimension_height'   => array(
				'name'        => esc_html__( 'Watermark Image Height', 'modula-watermark' ),
				'type'        => 'text',
				'description' => __( 'Specify the watermark image dimensions. If one of the sizes (width or height) is "0" or "" and the other is different than "0" or "" the watermark image will be scaled proportionally based on it\'s size. If both values are not empty or not "0" the watermark image will be morphed to the specified sizes.', 'modula-watermark' ),
				'is_child'    => true,
				'priority'    => 50,
				'default'     => 0,
			),
			'watermark_enable_backup'            => array(
				'name'        => esc_html__( 'Backup images ', 'modula-watermark' ),
				'type'        => 'toggle',
				'description' => __( 'Backup original images so that if you decide to delete the watermark you can restore the original image to your gallery/media library.', 'modula-watermark' ),
				'is_child'    => true,
				'priority'    => 51,
				'default'     => 1,
			),
			'watermark_apply_to_existing_images' => array(
				'name'          => esc_html__( 'Watermark inserted images', 'modula-watermark' ),
				'type'          => 'ajax_button',
				'value'         => 'apply_watermark',
				'button_text'   => 'Watermark Images',
				'description'   => __( 'Images already watermarked won\'t be watermarked again.', 'modula-watermark' ),
				'priority'      => 52,
				'error_message' => esc_html__( 'You need to save your gallery before applying watermark', 'modula_watermark' ),
			),
			'watermark_remove_watermark'         => array(
				'name'          => esc_html__( 'Remove Watermark', 'modula-watermark' ),
				'type'          => 'ajax_button',
				'value'         => 'remove_watermark',
				'button_text'   => 'Remove Watermark',
				'description'   => __( 'Remove watermark from watermarked images.', 'modula-watermark' ),
				'priority'      => 53,
				'error_message' => esc_html__( 'You need to save your gallery before removing watermark', 'modula_watermark' ),
			),

		);

		return $fields;

	}


	/***
	 * Add ajax_button field type, used for ajax
	 *
	 * @param $html
	 * @param $field
	 *
	 * @return string
	 */
	public function ajax_button_field_type( $html, $field ) {

		global $post;

		$modula_settings = get_post_meta( $post->ID, 'modula-settings', true );
		$modula_global   = get_option( 'modula_watermark' );
		$extra_class     = '';

		if ( ( ! isset( $modula_settings['watermark_enable_backup'] ) || '0' == $modula_settings['watermark_enable_backup'] ) && ( ! isset( $modula_global['watermark_enable_backup'] ) || '0' == $modula_global['watermark_enable_backup'] ) ) {

			$extra_class = ' disabled';
		}

		$html  = '<div class="modula-watermark-container ' . esc_attr( $field['value'] ) . '">';
		$html .= '<a href="#" class="button button-primary modula_ajax_button ' . esc_attr( $field['value'] ) . esc_attr( $extra_class ) . '">' . esc_html( $field['button_text'] ) . '</a>';
		$html .= '<div class="modula-watermark-error" data-msg="' . esc_attr( $field['error_message'] ) . '"></div>';
		$html .= '</div>';

		return $html;

	}


	/***
	 * Add image_upload field type
	 *
	 * @param $html
	 * @param $field
	 * @param $value
	 *
	 * @return string
	 */
	public function image_upload_field_type( $html, $field, $value ) {

		$style = array(
			'upload'  => '',
			'replace' => 'display:none;',
			'delete'  => 'display:none;',
		);

		if ( 0 != absint( $value ) ) {
			$style['upload']  = 'display:none;';
			$style['replace'] = '';
			$style['delete']  = '';
		}

		$html  = '<input type="hidden" name="modula-settings[' . esc_attr( $field['id'] ) . ']" id="' . esc_attr( $field['id'] ) . '" value="' . absint( $value ) . '">';
		$html .= '<div class="modula_watermark_image_preview">';
		if ( $value ) {
			$image = wp_get_attachment_image_src( $value, 'full' );
			if ( $image ) {
				$html .= '<img src="' . esc_url( $image[0] ) . '" id="modula_watermark_preview">';
			}
		}
		$html .= '</div>';
		$html .= '<input type="button" style="' . esc_attr( $style['upload'] ) . '" class="button button-primary" id="upload_watermark_file" value="' . esc_attr__( 'Upload', 'modula-watermark' ) . '">';
		$html .= '<input type="button" style="' . esc_attr( $style['replace'] ) . '" class="button button-primary" id="replace_watermark_file" value="' . esc_attr__( 'Replace', 'modula-watermark' ) . '">';
		$html .= '<input type="button" style="' . esc_attr( $style['delete'] ) . '" class="button" id="delete_watermark_file" value="' . esc_attr__( 'Delete', 'modula-watermark' ) . '">';

		return $html;

	}

	/**
	 * @param $tabs
	 *
	 * @return mixed
	 *
	 * Add Watermark Settings tab
	 */
	public function add_watermark_tab( $tabs ) {

		$tabs['watermark'] = array(
			'label'    => esc_html__( 'Watermark Settings', 'modula-watermark' ),
			'priority' => 30,
		);

		return $tabs;
	}

	/**
	 * Render Watermark Settings Tab
	 */
	public function render_watermark_tab() {
		if ( is_plugin_active( 'modula-best-grid-gallery/Modula.php' ) ) {
			include 'tabs/watermark.php';
		}
	}

	/**
	 * Register Watermark options
	 */
	public function register_watermark_option() {
		// creates our settings in the options table
		register_setting( 'modula_watermark', 'modula_watermark', array( $this, 'sanitize_option' ) );
	}

	public function sanitize_option( $current ) {
		return $current;
	}

	/**
	 * @param $defaults
	 *
	 * @return mixed
	 *
	 * Add Watermark Defaults
	 */
	public function add_watermark_defaults( $defaults ) {

		$defaults['enable_watermark']                 = 0;
		$defaults['custom_settings_watermark']        = 0;
		$defaults['watermark_image']                  = 0;
		$defaults['watermark_position']               = 'bottom_right';
		$defaults['watermark_margin']                 = 10;
		$defaults['watermark_image_dimension_width']  = 0;
		$defaults['watermark_image_dimension_height'] = 0;
		$defaults['watermark_enable_backup']          = 1;
		return $defaults;

	}

	/**
	 * Add watermark to single images using AJAX.
	 *
	 * @return void
	 * @throws ImagickException
	 *
	 * @since 1.0.2
	 */
	public function modula_apply_watermark() {

		check_ajax_referer( 'modula_watermark_images', 'nonce' );

		if ( ! isset( $_POST['gallery'] ) || 0 == absint( $_POST['gallery'] ) ) {
			wp_send_json_error( esc_html__( 'Something was wrong. Please try again', 'modula-watermark' ) );
			wp_die();
		}

		if ( ! isset( $_POST['modula_image_id'] ) ) {
			wp_send_json_error( esc_html__( 'Something was wrong. Please try again', 'modula-watermark' ) );
			wp_die();
		}

		$post_id  = absint( $_POST['gallery'] );
		$settings = get_post_meta( $post_id, 'modula-settings', true );
		$image_id = absint( $_POST['modula_image_id'] );

		if ( '0' == $settings['enable_watermark'] ) {
			wp_send_json_error( esc_html__( 'Please enable watermark and update gallery', 'modula-watermark' ) );
			wp_die();
		}

		if ( '0' == $settings['custom_settings_watermark'] ) {
			$modula_settings = get_option( 'modula_watermark' );

			if ( ! $modula_settings ) {
				wp_send_json_error( esc_html__( 'There seems to be an error with the Watermark settings. Please check.', 'modula-watermark' ) );
				wp_die();
			}

			$wattermark_settings = array(
				'watermark_image'                  => $modula_settings['watermark_image'],
				'watermark_position'               => $modula_settings['watermark_position'],
				'watermark_margin'                 => $modula_settings['watermark_margin'],
				'watermark_image_dimension_width'  => $modula_settings['watermark_image_dimension_width'],
				'watermark_image_dimension_height' => $modula_settings['watermark_image_dimension_height'],
				'watermark_enable_backup'          => $modula_settings['watermark_enable_backup'],
			);
		} else {

			if ( ! $settings ) {
				wp_send_json_error( esc_html__( 'There seems to be an error with the Watermark settings. Please check.', 'modula-watermark' ) );
				wp_die();
			}

			$wattermark_settings = array(
				'watermark_image'                  => $settings['watermark_image'],
				'watermark_position'               => $settings['watermark_position'],
				'watermark_margin'                 => $settings['watermark_margin'],
				'watermark_image_dimension_width'  => $settings['watermark_image_dimension_width'],
				'watermark_image_dimension_height' => $settings['watermark_image_dimension_height'],
				'watermark_enable_backup'          => $settings['watermark_enable_backup'],
			);
		}

		// check if we have a valid image id.
		if ( '0' == $wattermark_settings['watermark_image'] ) {
			wp_send_json_error( esc_html__( 'Please select a Watermark Image and update gallery', 'modula-watermark' ) );
			wp_die();
		}

		$watermarking        = Modula_Watermarking::get_instance();
		$watermark_path      = get_attached_file( $wattermark_settings['watermark_image'] );
		$already_watermarked = get_post_meta( $image_id, 'modula_watermark_applied', true );

		if ( $already_watermarked ) {
			wp_send_json_error( esc_html__( 'Image already watermarked. Skipping image.', 'modula-watermark' ) );
			wp_die();
		}

		$attachment_path = get_attached_file( $image_id );
		$success = $watermarking->add_watermark( $attachment_path, $watermark_path, $post_id, $wattermark_settings, array( 'id' => $image_id ) );

		if ( ! $success ) {
			wp_send_json_error( esc_html__( 'Please enable PHP GD or Imagick extensions', 'modula-watermark' ) );
			wp_die();
		}

		wp_update_attachment_metadata( $image_id, wp_generate_attachment_metadata( $image_id, $attachment_path ) );
		// Mark attachment as watermarked, so we don't do this again.
		update_post_meta( $image_id, 'modula_watermark_applied', true );

		$file_path = get_attached_file( $image_id );
		$info      = pathinfo( $file_path );
		$dir       = $info['dirname'];
		$ext       = $info['extension'];
		$name      = wp_basename( $file_path, ".$ext" );

		$resizer = new Modula_Image();

		if ( 'custom' == $settings['grid_image_size'] ) {
			$grid_sizes = array(
				'width'  => $settings['grid_image_dimensions']['width'],
				'height' => $settings['grid_image_dimensions']['height'],
			);
		} else {
			$grid_sizes = $settings['grid_image_size'];
		}

		$crop = false;

		if ( 'custom' == $settings['grid_image_size'] ) {
			$crop = boolval( boolval( $settings['grid_image_crop'] ) );
		}

		$sizes = $resizer->get_image_size( $image_id, $settings['type'], $grid_sizes, $crop );

		// Suffix applied to filename.
		$suffix = $sizes['width'] . 'x' . $sizes['height'];

		if ( isset( $settings['grid_image_crop'] ) && '1' == $settings['grid_image_crop'] ) {
			$suffix .= '_c';
		}

		// Get the destination file name.
		$dest_file_name = "${dir}/${name}-${suffix}.${ext}";

		if ( 'custom' == $settings['grid_image_size'] ) {
			$resized = $resizer->resize_image( $sizes['url'], $sizes['width'], $sizes['height'], $settings['grid_image_crop'], 'c', 100, false, array(), true );
		}

		wp_send_json( array( 'status' => 'success' ) );
		wp_die();

	}

	/**
	 * Remove watermark from images by restoring the original ones
	 *
	 * @since 1.0.2
	 */
	public function modula_remove_watermark() {

		check_ajax_referer( 'modula_watermark_images', 'nonce' );

		if ( ! isset( $_POST['gallery'] ) || 0 == absint( $_POST['gallery'] ) ) {
			wp_send_json_error( esc_html__( 'Something was wrong. Please try again', 'modula-watermark' ) );
			wp_die();
		}

		if ( ! isset( $_POST['modula_image_id'] ) ) {
			wp_send_json_error( esc_html__( 'Something was wrong. Please try again', 'modula-watermark' ) );
			wp_die();
		}

		$post_id         = absint( $_POST['gallery'] );
		$settings        = get_post_meta( $post_id, 'modula-settings', true );
		$modula_settings = get_option( 'modula_watermark' );
		$image_id        = absint( $_POST['modula_image_id'] );

		if ( '0' == $settings['watermark_enable_backup'] && '0' == $modula_settings['watermark_enable_backup'] ) {
			echo esc_html__( 'Backup is not enabled for this gallery, thus images can not be restored', 'modula-watermark' );
			wp_die();
		}

		require_once ABSPATH . 'wp-admin/includes/admin.php';
		$upload_dir = wp_get_upload_dir();
		$resizer    = new Modula_Image();

		$original_file_path = get_post_meta( $image_id, 'modula-backup', true );
		$attached_file      = get_attached_file( $image_id );
		$old_metadata       = wp_get_attachment_metadata( $image_id );
		$new_metadata       = wp_generate_attachment_metadata( $image_id, $attached_file );

		// We do rename so that we can delete the backup image to free up space.
		rename( $upload_dir['basedir'] . $original_file_path, $attached_file );
		update_post_meta( $image_id, 'modula_watermark_applied', false );

		// Redo the intermediate image sizes so that the watermark will be cleared.
		foreach ( $old_metadata['sizes'] as $old_size => $old_size_data ) {

			if ( empty( $new_metadata['sizes'][ $old_size ] ) ) {
				$new_metadata['sizes'][ $old_size ] = $old_metadata['sizes'][ $old_size ];
				continue;
			}

			$new_size_data = $new_metadata['sizes'][ $old_size ];

			if (
				$new_size_data['width'] !== $old_size_data['width']
				&& $new_size_data['height'] !== $old_size_data['height']
				&& file_exists( $upload_dir['basedir'] . $old_size_data['file'] )
			) {
				$new_metadata['sizes'][ $old_size . '_old_' . $old_size_data['width'] . 'x' . $old_size_data['height'] ] = $old_size_data;
			}
		}

		if ( 'custom' == $settings['grid_image_size'] ) {
			$sizes   = $resizer->get_image_size( $image_id, $settings['type'], $settings['grid_image_dimensions'], $settings['grid_image_crop'] );
			$resized = $resizer->resize_image( $sizes['url'], $sizes['width'], $sizes['height'], $settings['grid_image_crop'], 'c', 100, false, array(), true );
		}

		wp_update_attachment_metadata( $image_id, $new_metadata );

		echo esc_html__( 'Watermark has been removed', 'modula-watermark' );
		wp_die();

	}

	/**
	 * Uninstall filter
	 *
	 * @param $db_options
	 *
	 * @return mixed
	 *
	 * @since 1.0.1
	 */
	public function watermark_uninstall( $db_options ) {

		$db_options[] = 'modula_watermark';

		return $db_options;
	}

	/**
	 * Add Modula Watermark data to migrator
	 *
	 * @param $modula_settings
	 * @param $guest_settings
	 * @param $source
	 *
	 * @return mixed
	 *
	 * @since 1.0.2
	 */
	public function watermark_migrator_gallery_data( $modula_settings, $guest_settings, $source ) {

		if ( $source ) {
			switch ( $source ) {
				case 'envira':
					if ( isset( $guest_settings['config']['watermarking'] ) && 1 == $guest_settings['config']['watermarking'] ) {
						$modula_settings['enable_watermark']          = 1;
						$modula_settings['custom_settings_watermark'] = 1;
						$modula_settings['watermark_image']           = isset( $guest_settings['config']['watermarking_image_id'] ) ? $guest_settings['config']['watermarking_image_id'] : false;
						$modula_settings['watermark_position']        = str_replace( '-', '_', $guest_settings['config']['watermarking_position'] );
						$modula_settings['watermark_margin']          = $guest_settings['config']['watermarking_margin'];
					}
					break;
			}
		}

		return $modula_settings;
	}

	/**
	 * Add debug information
	 *
	 * @param $info
	 *
	 * @return mixed
	 * @since 1.0.5
	 */
	public function debug_information( $info ) {

		$watermark_opts = get_option( 'modula_watermark' );

		$info['modula_watermark_version'] = array(
			'label' => __( 'Modula Watermark Version', 'modula-watermark' ),
			'value' => esc_html( MODULA_WATERMARK_VERSION ),
			'debug' => esC_html( 'Modula Watermark version ' . MODULA_WATERMARK_VERSION ),
		);

		$info['watermark_image'] = array(
			'label' => __( 'Watermark image', 'modula-watermark' ),
			'value' => ( $watermark_opts && '0' != $watermark_opts['watermark_image'] ) ? __( 'Image present', 'modula-watermark' ) : __( 'Image not present', 'modula-watermark' ),
			'debug' => ( $watermark_opts && '0' != $watermark_opts['watermark_image'] ) ? 'Image present' : 'Image not present',
		);

		$info['watermark_position'] = array(
			'label' => __( 'Watermark position', 'modula-watermark' ),
			'value' => ( $watermark_opts && '' != $watermark_opts['watermark_position'] ) ? esc_html( $watermark_opts['watermark_position'] ) : 'No position',
			'debug' => ( $watermark_opts && '' != $watermark_opts['watermark_position'] ) ? esc_html( $watermark_opts['watermark_position'] ) : 'No position',
		);

		$info['watermark_margin'] = array(
			'label' => __( 'Watermark margin', 'modula-watermark' ),
			'value' => ( $watermark_opts && '' != $watermark_opts['watermark_margin'] ) ? esc_html( $watermark_opts['watermark_margin'] ) : 'No margin',
			'debug' => ( $watermark_opts && '' != $watermark_opts['watermark_margin'] ) ? esc_html( $watermark_opts['watermark_margin'] ) : 'No margin',
		);

		$info['watermark_image_dimension_height'] = array(
			'label' => __( 'Watermark image height', 'modula-watermark' ),
			'value' => ( $watermark_opts && isset( $watermark_opts['watermark_image_dimension_height'] ) ) ? esc_html( $watermark_opts['watermark_image_dimension_height'] . 'px' ) : __( 'No height', 'modula-watermark' ),
			'debug' => ( $watermark_opts && ! empty( $watermark_opts['watermark_image_dimension_height'] ) ) ? esc_html( $watermark_opts['watermark_image_dimension_height'] . 'px' ) : 'No height',
		);

		$info['watermark_image_dimension_width'] = array(
			'label' => __( 'Watermark image width', 'modula-watermark' ),
			'value' => ( $watermark_opts && isset( $watermark_opts['watermark_image_dimension_width'] ) ) ? esc_html( $watermark_opts['watermark_image_dimension_width'] . 'px' ) : __( 'No width', 'modula-watermark' ),
			'debug' => ( $watermark_opts && ! empty( $watermark_opts['watermark_image_dimension_width'] ) ) ? esc_html( $watermark_opts['watermark_image_dimension_width'] . 'px' ) : 'No width',
		);

		$info['watermark_backup'] = array(
			'label' => __( 'Watermark backup', 'modula-watermark' ),
			'value' => ( $watermark_opts && isset( $watermark_opts['watermark_enable_backup'] ) && '1' == $watermark_opts['watermark_enable_backup'] ) ? __( 'Enabled', 'modula-watermark' ) : __( 'Disabled' ),
			'debug' => ( $watermark_opts && isset( $watermark_opts['watermark_enable_backup'] ) && '1' == $watermark_opts['watermark_enable_backup'] ) ? 'Enabled' : 'Disabled',
		);

		return $info;
	}
}

