<?php

/**
 * AJAX class.
 *
 * @since 1.0.0
 *
 * @package Modula_Watermark
 */
class Modula_Watermarking {

	/**
	 * Holds the class object.
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public static $instance;

	/**
	 * Path to the file.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $file = __FILE__;

	/**
	 * Holds the base class object.
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public $base;

	/**
	 * Overlay a watermark, if specified, to the uploaded image
	 *
	 * @param string $attachment_path Path of the image
	 * @param string $watermark_path  Path of the watermark image
	 * @param int    $gallery_id      Gallery ID
	 * @param array  $settings        Ajax sent settings
	 *
	 * @param        $img_obj
	 *
	 * @throws ImagickException
	 * @since 1.0.0
	 */
	public function add_watermark( $attachment_path, $watermark_path, $gallery_id, $settings, $img_obj ) {

		// Watermark image using GD or Imagick
		if ( $this->has_gd_extension() ) {
			$file = $this->watermark_gd( $attachment_path, $watermark_path, $gallery_id, $settings, $img_obj );
		} elseif ( $this->has_imagick_extension() ) {
			$this->watermark_imagick( $attachment_path, $watermark_path, $gallery_id, $settings, $img_obj );
		} else {
			return false;
		}
	
	}


	/**
	 * Flag to determine if the GD library has been compiled.
	 *
	 * @return bool True if has proper extension, false otherwise.
	 * @since 1.0.0
	 *
	 */
	public function has_gd_extension() {

		return extension_loaded( 'gd' ) && function_exists( 'gd_info' );

	}

	/**
	 * Flag to determine if the Imagick library has been compiled.
	 *
	 * @return bool True if has proper extension, false otherwise.
	 * @since 1.0.0
	 *
	 */
	public function has_imagick_extension() {

		return extension_loaded( 'imagick' );

	}

	/**
	 * Watermark image using GD
	 *
	 * @param $attachment_path
	 * @param $watermark_path
	 * @param $gallery_id
	 * @param $settings
	 * @param $img_obj
	 *
	 * @return false|resource
	 * @since 1.0
	 */
	public function watermark_gd( $attachment_path, $watermark_path, $gallery_id, $settings, $img_obj ) {

		// Get images
		$image     = @imagecreatefromstring( file_get_contents( $attachment_path ) );
		$watermark = @imagecreatefromstring( file_get_contents( $watermark_path ) );

		// Get widths and heights for the image and watermark
		$image_width      = imagesx( $image );
		$image_height     = imagesy( $image );
		$watermark_width  = imagesx( $watermark );
		$watermark_height = imagesy( $watermark );

		// Get metadata (MIME type) for the image and watermark
		$image_meta     = getimagesize( $attachment_path );
		$watermark_meta = getimagesize( $watermark_path );

		// If the watermark exceeds the width or height of the image, scale the watermark down
        /*$scale_factor = 0.5;*/
        // made scale_factor smaller so that watermark image won't take too much space
        $scale_factor = 0.25;
        $resize_factor = $watermark_width / $watermark_height;

		if ( '1' == $settings['watermark_enable_backup'] ) {
			$current_user = wp_get_current_user();
			$upload_dir   = wp_get_upload_dir();

			if ( isset( $current_user->user_login ) && !empty( $upload_dir['basedir'] ) ) {

				$modula_gallery_folder = '/modula/gallery-' . $gallery_id . '/';
				$modula_gallery_dir    = $upload_dir['basedir'] . $modula_gallery_folder;

				if ( !file_exists( $modula_gallery_dir ) ) {
					wp_mkdir_p( $modula_gallery_dir );
				}

				// Copy the image in a personalized folder so we know where to get it from
				copy( $attachment_path, $modula_gallery_dir . basename( $attachment_path, $upload_dir['basedir'] ) );
				// Update image meta so we know where the original image is
				update_post_meta( $img_obj['id'], 'modula-backup', $modula_gallery_folder . basename( $attachment_path, $upload_dir['basedir'] )  );

			}

		}

        // check to see if user has entered only width size for watermark, and if so scale the image
		if( ('0' != $settings['watermark_image_dimension_width'] && '' != $settings['watermark_image_dimension_width'] ) && ( '0' == $settings['watermark_image_dimension_height'] || '' == $settings['watermark_image_dimension_height'] ) ){

            // Calculate new watermark size
            $new_watermark_width  = (int)$settings['watermark_image_dimension_width'];
            $new_watermark_height = (float)$settings['watermark_image_dimension_width'] / $resize_factor;

            // Create resized watermark image
            $watermark = imagecreatetruecolor( $new_watermark_width, (int)$new_watermark_height );
            imagecolortransparent( $watermark, imagecolorallocatealpha( $watermark, 0, 0, 0, 127 ) );
            imagealphablending( $watermark, false );
            imagesavealpha( $watermark, true );
            imagecopyresampled( $watermark, @imagecreatefromstring( file_get_contents( $watermark_path ) ), 0, 0, 0, 0, $new_watermark_width, $new_watermark_height, $watermark_width, $watermark_height );

            // From here on out, the "new" values are the actual width/height values to consider
            $watermark_width  = (int)$new_watermark_width;
            $watermark_height = (int)$new_watermark_height;
        }

        // check to see if user has entered only height size for watermark, and if so scale the image
        if( ('0' != $settings['watermark_image_dimension_height'] && '' != $settings['watermark_image_dimension_height'] ) && ( '0' == $settings['watermark_image_dimension_width'] || '' == $settings['watermark_image_dimension_width'] ) ){

            // Calculate new watermark size
            $new_watermark_width  = (float)$settings['watermark_image_dimension_height'] * $resize_factor;
            $new_watermark_height = (int)$settings['watermark_image_dimension_height'] ;

            // Create resized watermark image
            $watermark = imagecreatetruecolor( (int)$new_watermark_width, $new_watermark_height );
            imagecolortransparent( $watermark, imagecolorallocatealpha( $watermark, 0, 0, 0, 127 ) );
            imagealphablending( $watermark, false );
            imagesavealpha( $watermark, true );
            imagecopyresampled( $watermark, @imagecreatefromstring( file_get_contents( $watermark_path ) ), 0, 0, 0, 0, $new_watermark_width, $new_watermark_height, $watermark_width, $watermark_height );

            // From here on out, the "new" values are the actual width/height values to consider
            $watermark_width  = (int)$new_watermark_width;
            $watermark_height = (int)$new_watermark_height;
        }

        // check to see if user has entered both width and height size for watermark, and if so scale the image
        if( ('0' != $settings['watermark_image_dimension_height'] && '' != $settings['watermark_image_dimension_height'] ) && ( '0' != $settings['watermark_image_dimension_width'] && '' != $settings['watermark_image_dimension_width'] ) ){

            // Calculate new watermark size
            $new_watermark_width  = (int)$settings['watermark_image_dimension_width'];
            $new_watermark_height = (int)$settings['watermark_image_dimension_height'];

            // Create resized watermark image
            $watermark = imagecreatetruecolor( $new_watermark_width, $new_watermark_height );
            imagecolortransparent( $watermark, imagecolorallocatealpha( $watermark, 0, 0, 0, 127 ) );
            imagealphablending( $watermark, false );
            imagesavealpha( $watermark, true );
            imagecopyresampled( $watermark, @imagecreatefromstring( file_get_contents( $watermark_path ) ), 0, 0, 0, 0, $new_watermark_width, $new_watermark_height, $watermark_width, $watermark_height );

            // From here on out, the "new" values are the actual width/height values to consider
            $watermark_width  = (int)$new_watermark_width;
            $watermark_height = (int)$new_watermark_height;
        }

        // resize with factor after checking if height or width were set by the user
        if ( $watermark_width > $image_width || $watermark_height > $image_height ) {
            // Calculate new watermark size
            $new_watermark_width  = $watermark_width * $scale_factor;
            $new_watermark_height = $watermark_height * $scale_factor;

            // Create resized watermark image
            $watermark = imagecreatetruecolor( $new_watermark_width, $new_watermark_height );
            imagecolortransparent( $watermark, imagecolorallocatealpha( $watermark, 0, 0, 0, 127 ) );
            imagealphablending( $watermark, false );
            imagesavealpha( $watermark, true );
            imagecopyresampled( $watermark, @imagecreatefromstring( file_get_contents( $watermark_path ) ), 0, 0, 0, 0, $new_watermark_width, $new_watermark_height, $watermark_width, $watermark_height );

            // From here on out, the "new" values are the actual width/height values to consider
            $watermark_width  = $new_watermark_width;
            $watermark_height = $new_watermark_height;
        }



		// Enable imagealphablending for correct PNG rendering
		imagealphablending( $image, true );
		imagealphablending( $watermark, true );

		// Calculate position of watermark based on settings
		$watermark_position = $settings['watermark_position'];
		$watermark_margin   = $settings['watermark_margin'];
		$position           = array(
			'x' => 0,
			'y' => 0,
		);
		switch ( $watermark_position ) {
			case 'top_left':
				$position = array(
					'x' => ( 0 + $watermark_margin ),
					'y' => ( 0 + $watermark_margin ),
				);
				break;
			case 'top_right':
				$position = array(
					'x' => ( ( $image_width - $watermark_width ) - $watermark_margin ),
					'y' => ( 0 + $watermark_margin ),
				);
				break;
			case 'center':
				$position = array(
					'x' => ( ( $image_width - $watermark_width ) / 2 ),
					'y' => ( ( $image_height - $watermark_height ) / 2 ),
				);
				break;
			case 'bottom_left':
				$position = array(
					'x' => ( 0 + $watermark_margin ),
					'y' => ( ( $image_height - $watermark_height ) - $watermark_margin ),
				);
				break;
			case 'bottom_right':
				$position = array(
					'x' => ( ( $image_width - $watermark_width ) - $watermark_margin ),
					'y' => ( ( $image_height - $watermark_height ) - $watermark_margin ),
				);
				break;
			default:
				// Allow devs to run their own calculations here
				$position = apply_filters( 'modula_watermark_position', $position, $attachment_id, $gallery_id, $settings );
				break;
		}

		// Copy the entire $watermark image onto a matching sized portion of the $image
		imagecopy( $image, $watermark, $position['x'], $position['y'], 0, 0, $watermark_width, $watermark_height );

		// Get the MIME type of the original image, so we know which image function to call when saving
		switch ( $image_meta['mime'] ) {
			/**
			 * JPEG
			 */
			case 'image/jpeg':
			case 'image/jpg':
				// Save image as JPEG
				imagejpeg( $image, $attachment_path );
				break;

			/**
			 * PNG
			 */
			case 'image/png':
				// Save image as PNG
				imagepng( $image, $attachment_path );
				break;

			/**
			 * GIF
			 */
			case 'image/gif':
				// Save image as GIF
				imagegif( $image, $attachment_path );
				break;
		}

		// Free up resources
		imagedestroy( $image );
		// For the moment don't destroy the watermark
		//imagedestroy( $watermark );

		return $image;
	}

	/**
	 * Watermark image using Imagick
	 *
	 * @param $attachment_path
	 * @param $watermark_path
	 * @param $gallery_id
	 * @param $settings
	 * @param $img_obj
	 *
	 * @throws ImagickException
	 * @since 1.0
	 */
	public function watermark_imagick( $attachment_path, $watermark_path, $gallery_id, $settings, $img_obj ) {

		// Get watermark path
		$watermark_path = get_attached_file( $settings['watermark_image'] );

		// Get images
		$image     = new Imagick( $attachment_path );
		$watermark = new Imagick( $watermark_path );

		// Get widths and heights for the image and watermark
		$image_size   = $image->getImageGeometry();
		$image_width  = $image_size['width'];
		$image_height = $image_size['height'];

		$watermark_size   = $image->getImageGeometry();
		$watermark_width  = $watermark_size['width'];
		$watermark_height = $watermark_size['height'];

		// Get metadata (MIME type) for the image and watermark
		$image_meta     = $image->getFormat();
		$watermark_meta = $watermark->getFormat();

		// If the watermark exceeds the width or height of the image, scale the watermark down
		/*$scale_factor = 0.5;*/
        // made scale_factor smaller so that watermark image won't take too much space
        $scale_factor = 0.25;
        $resize_factor = $watermark_width / $watermark_height;

		if ( '1' == $settings['watermark_enable_backup'] ) {
			$current_user = wp_get_current_user();
			$upload_dir   = wp_get_upload_dir();

			if ( isset( $current_user->user_login ) && !empty( $upload_dir['basedir'] ) ) {

				$modula_gallery_folder = '/modula/gallery-' . $gallery_id . '/';
				$modula_gallery_dir    = $upload_dir['basedir'] . $modula_gallery_folder;

				if ( !file_exists( $modula_gallery_dir ) ) {
					wp_mkdir_p( $modula_gallery_dir );
				}

				// Copy the image in a personalized folder so we know where to get it from
				copy( $attachment_path, $modula_gallery_dir . basename( $attachment_path, $upload_dir['basedir'] ) );
				// Update image meta so we know where the original image is
				update_post_meta( $img_obj['id'], 'modula-backup', $modula_gallery_folder . basename( $attachment_path, $upload_dir['basedir'] ) );

			}

		}

		// check to see if user has entered only width size for watermark, and if so scale the image
        if( ('0' != $settings['watermark_image_dimension_width'] && '' != $settings['watermark_image_dimension_width'] ) && ( '0' == $settings['watermark_image_dimension_height'] || '' == $settings['watermark_image_dimension_height'] ) ){

            // Calculate new watermark size
            $new_watermark_width  = (int)$settings['watermark_image_dimension_width'];
            $new_watermark_height = (float)$settings['watermark_image_dimension_width'] / $resize_factor;

            // Create resized watermark image
            $watermark->scaleImage( $new_watermark_width, (int)$new_watermark_height );

        }

        // check to see if user has entered only height size for watermark, and if so scale the image
        if( ('0' != $settings['watermark_image_dimension_height'] && '' != $settings['watermark_image_dimension_height'] ) && ( '0' == $settings['watermark_image_dimension_width'] || '' == $settings['watermark_image_dimension_width'] ) ){

            // Calculate new watermark size
            $new_watermark_width  = (float)$settings['watermark_image_dimension_height'] * $resize_factor;
            $new_watermark_height = (int)$settings['watermark_image_dimension_height'] ;

            // Create resized watermark image
            $watermark->scaleImage( (int)$new_watermark_width, $new_watermark_height );


        }

        // check to see if user has entered both width and height size for watermark, and if so scale the image
        if( ('0' != $settings['watermark_image_dimension_height'] && '' != $settings['watermark_image_dimension_height'] ) && ( '0' != $settings['watermark_image_dimension_width'] && '' != $settings['watermark_image_dimension_width'] ) ){

            // Calculate new watermark size
            $new_watermark_width  = (int)$settings['watermark_image_dimension_width'];
            $new_watermark_height = (int)$settings['watermark_image_dimension_height'];

            // Create resized watermark image
            $watermark->scaleImage( $new_watermark_width, $new_watermark_height );

        }

        // resize with factor after checking if height or width were set by the user
        if ( $watermark_width > $image_width || $watermark_height > $image_height ) {
            // Calculate new watermark size
            $new_watermark_width  = $watermark_width * $scale_factor;
            $new_watermark_height = $watermark_height * $scale_factor;

            // Create resized watermark image
            $watermark->scaleImage( $new_watermark_width, $new_watermark_height );
        }


        // Calculate position of watermark based on settings
		$watermark_position = $settings['watermark_position'];
		$watermark_margin   = $settings['watermark_margin'];
		$position           = array(
			'x' => 0,
			'y' => 0,
		);
		switch ( $watermark_position ) {
			case 'top-left':
				$position = array(
					'x' => ( 0 + $watermark_margin ),
					'y' => ( 0 + $watermark_margin ),
				);
				break;
			case 'top-right':
				$position = array(
					'x' => ( ( $image_width - $watermark_width ) - $watermark_margin ),
					'y' => ( 0 + $watermark_margin ),
				);
				break;
			case 'center':
				$position = array(
					'x' => ( ( $image_width - $watermark_width ) / 2 ),
					'y' => ( ( $image_height - $watermark_height ) / 2 ),
				);
				break;
			case 'bottom-left':
				$position = array(
					'x' => ( 0 + $watermark_margin ),
					'y' => ( ( $image_height - $watermark_height ) - $watermark_margin ),
				);
				break;
			case 'bottom-right':
				$position = array(
					'x' => ( ( $image_width - $watermark_width ) - $watermark_margin ),
					'y' => ( ( $image_height - $watermark_height ) - $watermark_margin ),
				);
				break;
			default:
				// Allow devs to run their own calculations here
				$position = apply_filters( 'modula_watermark_position', $position, $attachment_id, $gallery_id, $settings );
				break;
		}

		// Copy the entire $watermark image onto a matching sized portion of the $image
		$image->compositeImage( $watermark, Imagick::COMPOSITE_MATHEMATICS, $position['x'], $position['y'] );

		// Save
		$image->writeImage( $attachment_path );

		// Free up resources
		unset( $image );
		unset( $watermark );

	}

	/**
	 * Returns the singleton instance of the class.
	 *
	 * @return object The Modula_Watermark_Ajax object.
	 * @since 1.0.0
	 *
	 */
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Modula_Watermarking ) ) {
			self::$instance = new Modula_Watermarking();
		}

		return self::$instance;

	}

}
