<?php

/**
 * Main class for Moula Zoom
 */
class Modula_Zoom {
	/**
	 * Holds the class object
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public static $instance;

	/**
	 * The name of the plugin
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $plugin_name = 'Modula Zoom';

	/**
	 * Unique plugin identifier
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $plugin_slug = 'modula-zoom';

	/**
	 * Primary class construct
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		$this->require_helper();
		add_action( 'init', array( $this, 'register_scripts' ) );
		// Set admin requirements on admin_init so that we won't trigger get_packages from WPChill_Upsells on frontend
		add_action( 'admin_init', array( $this, 'set_admin_requirements' ), 25 );
		add_action( 'modula_extra_scripts', array( $this, 'modula_zoom_scripts' ) );
		add_action( 'modula_scripts_after_wp_modula', array( $this, 'modula_zoom_admin_scripts' ) );
		add_action( 'modula_defaults_scripts_after_wp_modula', array( $this, 'modula_zoom_admin_scripts' ) );
		add_filter( 'modula_lite_default_settings', array( $this, 'modula_zoom_defaults' ) );
		add_action( 'modula_link_extra_scripts', array( $this, 'modula_zoom_link_scripts' ) );
		add_filter( 'modula_gallery_settings', array( $this, 'modula_zoom_gallery_settings' ), 10, 2 );
		add_filter( 'modula_link_gallery_settings', array( $this, 'modula_zoom_gallery_link_settings' ), 10, 2 );
		add_filter( 'modula_shortcode_css', array( $this, 'modula_zoom_css' ), 25, 3 );
		add_filter( 'modula_link_shortcode_css', array( $this, 'modula_zoom_css' ), 25, 3 );
		add_filter( 'modula_fancybox_options', array( $this, 'modula_zoom_fancybox_settings' ), 35, 2 );

		add_action( 'modula_album_extra_scripts', array( $this, 'modula_zoom_albums_scripts' ) );
		add_filter( 'modula_album_default_settings', array( $this, 'modula_zoom_albums_defaults' ) );
		add_filter( 'modula_album_shortcode_css', array( $this, 'modula_albums_zoom_css' ), 25, 3 );
		add_filter( 'modula_album_template_data', array( $this, 'modula_zoom_albums_config' ) );
		// add Zoom data to migrator
		add_filter( 'modula_migrate_gallery_data', array( $this, 'modula_zoom_migrator_data' ), 25, 3 );

		add_action( 'init', array( $this, 'set_locale' ) );

		register_activation_hook( MODULA_ZOOM_FILE, array( $this, 'plugin_activation' ) );
		if ( is_admin() ) {
			add_action( 'modula_pro_updater', array( $this, 'addon_updater' ), 15, 2 );
		}
	}

	/**
	 * Set admin requirements and check for license.
	 *
	 * @return void
	 * @since 1.0.4
	 */
	public function set_admin_requirements() {

		if ( class_exists( 'WPChill_Upsells' ) ) {
			$args           = apply_filters(
				'modula_upsells_args',
				array(
					'shop_url' => 'https://wp-modula.com',
					'slug'     => 'modula',
				)
			);
			$wpchill_upsell = WPChill_Upsells::get_instance( $args );
			
			if ( class_exists( 'WPChill_Upsells' ) ) {
				if ( $wpchill_upsell && ! $wpchill_upsell->is_upgradable_addon( 'modula-zoom' ) ) {
					$this->require_admin();
				}
			}
		}
	}

	/**
	 * Prevent plugin activation if Modula Pro is not installed
	 *
	 * @since 1.0.0
	 */
	public function plugin_activation() {
		if ( ! class_exists( 'Modula_PRO' ) ) {
			deactivate_plugins( plugin_basename( MODULA_ZOOM_FILE ) );
			wp_die( esc_html__( 'Please install and activate Modula Pro before using Modula Zoom add-on.', 'modula-zoom' ) );
		}
	}

	public function addon_updater( $license_key, $store_url ) {

		if ( class_exists( 'Modula_Pro_Base_Updater' ) ) {
			$modula_addon_updater = new Modula_Pro_Base_Updater(
				$store_url,
				MODULA_ZOOM_FILE,
				array(
					'version' => MODULA_ZOOM_VERSION,        // current version number
					'license' => $license_key,               // license key (used get_option above to retrieve from DB)
					'item_id' => 447523,                      // ID of the product
					'author'  => 'MachoThemes',            // author of this plugin
					'beta'    => false,
				)
			);
		}
	}

	/**
	 * Returns the singleton instance of the class.
	 *
	 * @since 1.0.0
	 *
	 * @return object The Modula_Zoom object.
	 */
	public static function get_instance() {
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Modula_Zoom ) ) {
			self::$instance = new Modula_Zoom();
		}

		return self::$instance;
	}

	/**
	 * Loads all admin related files into scope.
	 *
	 * @since 1.0.0
	 */
	public function require_admin() {

		require_once MODULA_ZOOM_PATH . 'includes/admin/class-modula-zoom-settings.php';
		require_once MODULA_ZOOM_PATH . 'includes/admin/class-modula-zoom-albums-settings.php';

	}

	/**
	 * Loads all the helper functions files
	 *
	 * @since 1.0.0
	 */
	public function require_helper() {
		require_once MODULA_ZOOM_PATH . 'includes/modula-zoom-helper.php';
	}

	/**
	 * Add Zoom defaults settings
	 *
	 * @param array $defaults
	 *
	 * @return mixed $defaults;
	 * @since 1.0.0
	 */
	public function modula_zoom_defaults( $defaults ) {
		$defaults['enable_zoom']          = 0;
		$defaults['zoom_on_hover']        = 1;
		$defaults['zoom_type']            = 'basic';
		$defaults['zoom_effect']          = 'fade_in';
		$defaults['zoom_window_position'] = 'upper_left';
		$defaults['zoom_window_size']     = 'medium';
		$defaults['zoom_lens_size']       = 'medium';
		$defaults['zoom_lens_shape']      = 'round';
		$defaults['zoom_tint_color']      = '';
		$defaults['zoom_tint_opacity']    = 0;

		return $defaults;
	}

	/**
	 * Add Zoom Albums defaults settings
	 *
	 * @param array $defaults
	 *
	 * @return mixed $defaults;
	 * @since 1.0.0
	 */
	public function modula_zoom_albums_defaults( $defaults ) {
		$defaults['enable_zoom_albums']          = 0;
		$defaults['zoom_on_hover_albums']        = 1;
		$defaults['zoom_type_albums']            = 'basic';
		$defaults['zoom_effect_albums']          = 'fade_in';
		$defaults['zoom_window_position_albums'] = 'upper_left';
		$defaults['zoom_window_size_albums']     = 'medium';
		$defaults['zoom_lens_size_albums']       = 'medium';
		$defaults['zoom_lens_shape_albums']      = 'round';
		$defaults['zoom_tint_color_albums']      = '';
		$defaults['zoom_tint_opacity_albums']    = 0;

		return $defaults;
	}

	/**
	 * Register all scripts and styles
	 */
	public function register_scripts() {
		wp_register_script( 'modula-elevate-zoom', MODULA_ZOOM_URL . 'assets/js/jquery.elevatezoom.js', array( 'jquery' ), MODULA_ZOOM_VERSION, true );
		wp_register_script( 'modula-zoom-js', MODULA_ZOOM_URL . 'assets/js/fancybox-modula-zoom.js', array( 'jquery' ), MODULA_ZOOM_VERSION, true );
		wp_register_script( 'modula-zoom-link-js', MODULA_ZOOM_URL . 'assets/js/fancybox-modula-link-zoom.js', array( 'jquery' ), MODULA_ZOOM_VERSION, true );
		wp_register_script( 'modula-zoom-albums-js', MODULA_ZOOM_URL . 'assets/js/fancybox-modula-albums-zoom.js', array( 'jquery' ), MODULA_ZOOM_VERSION, true );

	}

	/**
	 * Enqueue necessary zoom scripts
	 *
	 * @param array $settings
	 *
	 * @return void
	 *
	 * @since 1.0.0
	 */
	public function modula_zoom_scripts( $settings ) {

		if ( isset( $settings['enable_zoom'] ) && '1' == $settings['enable_zoom'] ) {
			wp_enqueue_script( 'modula-elevate-zoom' );
			wp_enqueue_script( 'modula-zoom-js' );
		}
	}

	/**
	 * Enqueue necessary link scripts
	 *
	 * @since 1.0.0
	 *
	 * @param array $settings
	 */
	public function modula_zoom_link_scripts( $settings ) {

		if ( isset( $settings['enable_zoom'] ) && '1' == $settings['enable_zoom'] ) {
			wp_enqueue_script( 'modula-elevate-zoom' );
			wp_enqueue_script( 'modula-zoom-link-js' );
		}
	}

	/**
	 * Enqueue necessary albums scripts
	 *
	 * @since 1.0.0
	 *
	 * @param array $settings
	 */
	public function modula_zoom_albums_scripts( $settings ) {
		if ( isset( $settings['enable_zoom_albums'] ) && '1' == $settings['enable_zoom_albums'] ) {
			wp_enqueue_script( 'modula-elevate-zoom' );
			wp_enqueue_script( 'modula-zoom-albums-js' );
		}
	}


	/**
	 * Enqueue admin scripts
	 */
	public function modula_zoom_admin_scripts() {

		wp_enqueue_script( 'modula-zoom-conditions', MODULA_ZOOM_URL . 'assets/js/modula-zoom-conditions.js', array( 'jquery' ), MODULA_ZOOM_VERSION, true );
	}

	/**
	 * JS Config for Zoom
	 *
	 * @param array $js_config
	 * @param array $settings
	 *
	 * @return mixed $js_config
	 *
	 * @since 1.0.0
	 */
	public function modula_zoom_gallery_settings( $js_config, $settings ) {

		if ( isset( $settings['enable_zoom'] ) && '1' == $settings['enable_zoom'] ) {
			$js_config['zoom'] = array(
				'zoomType'           => $settings['zoom_type'],
				'zoomOnHover'        => boolval( $settings['zoom_on_hover'] ),
				'zoomWindowPosition' => modula_zoom_get_window_position( $settings['zoom_window_position'] ),
				'zoomWindowWidth'    => modula_zoom_get_window_size( $settings['zoom_window_size'] ),
				'zoomWindowHeight'   => modula_zoom_get_window_size( $settings['zoom_window_size'] ),
				'lensSize'           => modula_zoom_get_window_size( $settings['zoom_lens_size'] ),
				'zoomLensShape'      => $settings['zoom_lens_shape'],
				'lensShape'          => $settings['zoom_lens_shape'],
				'tint'               => 'lens' != $settings['zoom_type'] ? true : false,
				'tintColour'         => !empty( $settings['zoom_tint_color'] ) ? $settings['zoom_tint_color'] : '#666' ,
				'tintOpacity'        => ( !empty( $settings['zoom_tint_opacity'] ) ? absint( $settings['zoom_tint_opacity'] ) : 40 ) / 100,
				'easing'             => 'easing' == $settings['zoom_effect'] ? true : false,
				'zoomWindowFadeIn'   => 'fade_in' == $settings['zoom_effect'] || 'fade_in_out' == $settings['zoom_effect'] ? 650 : false,
				'lensFadeIn'         => 'fade_in' == $settings['zoom_effect'] || 'fade_in_out' == $settings['zoom_effect'] ? 650 : false,
				'zoomTintFadeIn'     => 'fade_in' == $settings['zoom_effect'] || 'fade_in_out' == $settings['zoom_effect'] ? 650 : false,
				'zoomWindowFadeOut'  => 'fade_out' == $settings['zoom_effect'] || 'fade_in_out' == $settings['zoom_effect'] ? 650 : false,
				'lensFadeOut'        => 'fade_out' == $settings['zoom_effect'] || 'fade_in_out' == $settings['zoom_effect'] ? 650 : false,
				'zoomTintFadeOut'    => 'fade_out' == $settings['zoom_effect'] || 'fade_in_out' == $settings['zoom_effect'] ? 650 : false,
				'zIndex'			 => 999999999,
			);
		}

		return $js_config;
	}

	/**
	 * Fancybox options to add the zoom buttom
	 *
	 * @param array $fancybox_options
	 * @param array $settings
	 *
	 * @return mixed $fancybox_options
	 * @since 1.0.0
	 */
	public function modula_zoom_fancybox_settings( $fancybox_options, $settings ) {

		if ( isset( $settings['enable_zoom'] ) && '1' == $settings['enable_zoom'] ) {
			if ( '1' == $settings['zoom_on_hover'] ) {

				if ( false !== ( $key = array_search( 'zoom', $fancybox_options['buttons'] ) ) ) {
					unset( $fancybox_options['buttons'][ $key ] );
				}
			} elseif ( '0' == $settings['lightbox_zoom'] ) {
				array_unshift( $fancybox_options['buttons'], 'zoom' );
			}
		}

		return $fancybox_options;
	}

	/**
	 * JS Config & Fancybox Options for [modula-link]
	 *
	 * @param array $link_js_config
	 * @param array $settings
	 *
	 * @return mixed $link_js_config
	 * @since 1.0.0
	 */
	public function modula_zoom_gallery_link_settings( $link_js_config, $settings ) {

		if ( isset( $settings['enable_zoom'] ) && '1' == $settings['enable_zoom'] ) {
			$link_js_config['lightbox'] = array(
				'zoom' => array(
					'zoomType'           => $settings['zoom_type'],
					'zoomOnHover'        => boolval( $settings['zoom_on_hover'] ),
					'zoomWindowPosition' => modula_zoom_get_window_position( $settings['zoom_window_position'] ),
					'zoomWindowWidth'    => modula_zoom_get_window_size( $settings['zoom_window_size'] ),
					'zoomWindowHeight'   => modula_zoom_get_window_size( $settings['zoom_window_size'] ),
					'lensSize'           => modula_zoom_get_window_size( $settings['zoom_lens_size'] ),
					'zoomLensShape'      => $settings['zoom_lens_shape'],
					'lensShape'          => $settings['zoom_lens_shape'],
					'tint'               => 'lens' != $settings['zoom_type'] ? true : false,
					'tintColour'         => !empty( $settings['zoom_tint_color'] ) ? $settings['zoom_tint_color'] : '#666' ,
					'tintOpacity'        => ( !empty( $settings['zoom_tint_opacity'] ) ? absint( $settings['zoom_tint_opacity'] ) : 40 ) / 100,
					'easing'             => 'easing' == $settings['zoom_effect'] ? true : false,
					'zoomWindowFadeIn'   => 'fade_in' == $settings['zoom_effect'] || 'fade_in_out' == $settings['zoom_effect'] ? 650 : false,
					'lensFadeIn'         => 'fade_in' == $settings['zoom_effect'] || 'fade_in_out' == $settings['zoom_effect'] ? 650 : false,
					'zoomTintFadeIn'     => 'fade_in' == $settings['zoom_effect'] || 'fade_in_out' == $settings['zoom_effect'] ? 650 : false,
					'zoomWindowFadeOut'  => 'fade_out' == $settings['zoom_effect'] || 'fade_in_out' == $settings['zoom_effect'] ? 650 : false,
					'lensFadeOut'        => 'fade_out' == $settings['zoom_effect'] || 'fade_in_out' == $settings['zoom_effect'] ? 650 : false,
					'zoomTintFadeOut'    => 'fade_out' == $settings['zoom_effect'] || 'fade_in_out' == $settings['zoom_effect'] ? 650 : false,
					'zIndex'			 => 999999999,
				),
			);
		}
		return $link_js_config;
	}

	/**
	 * Fix for Zoom not on hover
	 *
	 * @param array $template_data
	 *
	 * @return mixed $template_data
	 * @since 1.0.0
	 */
	public function modula_zoom_albums_config( $template_data ) {

		if ( isset( $template_data['settings']['enable_zoom_albums'] ) && '1' == $template_data['settings']['enable_zoom_albums'] ) {

			$template_data['js_config']['zoom'] = array(
				'zoomType'           => $template_data['settings']['zoom_type_albums'],
				'zoomOnHover'        => boolval( $template_data['settings']['zoom_on_hover_albums'] ),
				'zoomWindowPosition' => modula_zoom_get_window_position( $template_data['settings']['zoom_window_position_albums'] ),
				'zoomWindowWidth'    => modula_zoom_get_window_size( $template_data['settings']['zoom_window_size_albums'] ),
				'zoomWindowHeight'   => modula_zoom_get_window_size( $template_data['settings']['zoom_window_size_albums'] ),
				'lensSize'           => modula_zoom_get_window_size( $template_data['settings']['zoom_lens_size_albums'] ),
				'zoomLensShape'      => $template_data['settings']['zoom_lens_shape_albums'],
				'lensShape'          => $template_data['settings']['zoom_lens_shape_albums'],
				'tint'               => 'lens' != $template_data['settings']['zoom_type_albums'] ? true : false,
				'tintColour'         => $template_data['settings']['zoom_tint_color_albums'],
				'tintOpacity'        => $template_data['settings']['zoom_tint_opacity_albums'] / 100,
				'easing'             => 'easing' == $template_data['settings']['zoom_effect_albums'] ? true : false,
				'zoomWindowFadeIn'   => 'fade_in' == $template_data['settings']['zoom_effect_albums'] || 'fade_in_out' == $template_data['settings']['zoom_effect_albums'] ? 650 : false,
				'lensFadeIn'         => 'fade_in' == $template_data['settings']['zoom_effect_albums'] || 'fade_in_out' == $template_data['settings']['zoom_effect_albums'] ? 650 : false,
				'zoomTintFadeIn'     => 'fade_in' == $template_data['settings']['zoom_effect_albums'] || 'fade_in_out' == $template_data['settings']['zoom_effect_albums'] ? 650 : false,
				'zoomWindowFadeOut'  => 'fade_out' == $template_data['settings']['zoom_effect_albums'] || 'fade_in_out' == $template_data['settings']['zoom_effect_albums'] ? 650 : false,
				'lensFadeOut'        => 'fade_out' == $template_data['settings']['zoom_effect_albums'] || 'fade_in_out' == $template_data['settings']['zoom_effect_albums'] ? 650 : false,
				'zoomTintFadeOut'    => 'fade_out' == $template_data['settings']['zoom_effect_albums'] || 'fade_in_out' == $template_data['settings']['zoom_effect_albums'] ? 650 : false,
				'zIndex'			 => 999999999,
			);
		}

		if ( '1' == $template_data['settings']['enable_zoom'] ) {

			$template_data['album_container']['data-config'] = json_encode( $template_data['js_config'] );
			return $template_data;

		} elseif ( ! isset( $template_data['settings']['zoom_on_hover_albums'] ) || '1' != $template_data['settings']['zoom_on_hover_albums'] ) {

			if ( ! in_array( 'zoom', $template_data['js_config']['lightbox_settings']['buttons'] ) ) {
				array_unshift( $template_data['js_config']['lightbox_settings']['buttons'], 'zoom' );
			}
		} else {

			if ( false !== ( $key = array_search( 'zoom', $template_data['js_config']['lightbox_settings']['buttons'] ) ) ) {
				unset( $template_data['js_config']['lightbox_settings']['buttons'][ $key ] );
			}
		}

		$template_data['album_container']['data-config'] = json_encode( $template_data['js_config'] );

		return $template_data;
	}


	/**
	 * Add inline CSS for Modula Gallery
	 *
	 * @param string $css
	 * @param string $gallery_id
	 * @param array  $settings
	 *
	 * @return string
	 * @since 1.0.0
	 */
	public function modula_zoom_css( $css, $gallery_id, $settings ) {

		if ( isset( $settings['enable_zoom'] ) && '1' == $settings['enable_zoom'] ) {
			$css .= '.zoomContainer {z-index: 300000000!important;}';
		}

		return $css;
	}

	/**
	 * Add inline CSS for Modula Albums
	 *
	 * @param string $css
	 * @param string $gallery_id
	 * @param array  $settings
	 *
	 * @return string
	 * @since 1.0.0
	 */
	public function modula_albums_zoom_css( $css, $gallery_id, $settings ) {

		if ( isset( $settings['enable_zoom_albums'] ) && '1' == $settings['enable_zoom_albums'] ) {
			$css .= '.zoomContainer {z-index: 300000000!important;}';
		}

		return $css;
	}

	/**
	 * Add Modula Zoom data to migrator
	 *
	 * @param $modula_settings
	 * @param $guest_settings
	 * @param $source
	 *
	 * @return mixed
	 *
	 * @since 1.0.1
	 */
	public function modula_zoom_migrator_data( $modula_settings, $guest_settings, $source ) {

		if ( $source ) {
			switch ( $source ) {
				case 'envira':
					if ( isset( $guest_settings['config']['zoom'] ) && 1 == $guest_settings['config']['zoom'] ) {
						$modula_settings['enable_zoom'] = 1;
					}

					if ( isset( $guest_settings['config']['zoom_hover'] ) && 1 == $guest_settings['config']['zoom_hover'] ) {
						$modula_settings['zoom_on_hover'] = 1;
					} else {
						$modula_settings['zoom_on_hover'] = 0;
					}

					if ( isset( $guest_settings['config']['zoom_effect'] ) ) {
						if ( 'no-effect' == $guest_settings['config']['zoom_effect'] ) {
							$modula_settings['zoom_effect'] = 'none';
						} else {
							$modula_settings['zoom_effect'] = str_replace( '-', '_', $guest_settings['config']['zoom_effect'] );
						}
					}

					if ( isset( $guest_settings['config']['zoom_type'] ) ) {
						if ( 'basic' == $guest_settings['config']['zoom_type'] ) {
							$modula_settings['zoom_type'] = 'window';
						} else {
							$modula_settings['zoom_type'] = $guest_settings['config']['zoom_type'];
						}
					}

					if ( isset( $guest_settings['config']['zoom_lens_shape'] ) ) {
						$modula_settings['zoom_lens_shape'] = $guest_settings['config']['zoom_lens_shape'];
					}

					if ( isset( $guest_settings['config']['zoom_position'] ) ) {
						$modula_settings['zoom_window_position'] = str_replace( '-', '_', $guest_settings['config']['zoom_position'] );
					}

					if ( isset( $guest_settings['config']['zoom_window_size'] ) ) {
						if ( 'x-large' == $guest_settings['config']['zoom_window_size'] ) {
							$modula_settings['zoom_window_size'] = 'xlarge';
						} else {
							$modula_settings['zoom_window_size'] = $guest_settings['config']['zoom_window_size'];
						}
					}

					if ( isset( $guest_settings['config']['zoom_tint_color'] ) ) {
						$modula_settings['zoom_tint_color'] = '#' . $guest_settings['config']['zoom_tint_color'];
					}

					if ( isset( $guest_settings['config']['zoom_tint_color_opacity'] ) ) {
						$modula_settings['zoom_tint_opacity'] = $guest_settings['config']['zoom_tint_color_opacity'];
					}

					break;
			}
		}

		return $modula_settings;
	}

	/**
	 * Loads the plugin textdomain for translation.
	 *
	 * @since 1.0.3
	 */
	public function set_locale() {
		load_plugin_textdomain( $this->plugin_slug, false, MODULA_ZOOM_TRANSLATE );
	}
}
