<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * @since      2.7.5
 */
class Modula_Pro_Bnb_Type {

	private $count = 0;
	public function __construct() {

		add_filter( 'modula_gallery_template_data', array( $this, 'bnb_class_helper' ), 20, 1 );
		add_filter( 'modula_shortcode_item_data', array( $this, 'modula_bnb_generate_image_links' ), 10, 4 );
		add_filter( 'modula_shortcode_before_item', array( $this, 'bnb_image_wrap_start' ), 10, 2 );
		add_filter( 'modula_shortcode_after_item', array( $this, 'bnb_image_wrap_end' ), 1, 2 );

		// Remove filters.
		add_filter( 'modula_gallery_template_data', array( $this, 'modula_slider_settings' ), 10, 1 );

		// Radio icon builder to update Icons
		add_filter( 'modula_render_pro-icon-radio_field_type', array( $this, 'pro_icon_radio_field_type' ), 10, 3 );
		// Add pro airnbn  icon
		add_filter( 'modula_radio_icon_url', array( $this, 'modula_bnb_icon_url' ), 10, 2 );
		add_filter( 'modula_shortcode_css', array( $this, 'generate_new_css' ), 10, 3 );

		add_action( 'modula_elementor_before_enqueue_styles', array( $this, 'add_elementor_styles' ) );
	}

	/**
	 * Remove Filters for bnb type.
	 *
	 * @param array $template_data The template data array.
	 * @return array Modified template data.
	 * @since 2.7.5
	 */
	public function modula_slider_settings( $template_data ) {
		if ( 'bnb' === $template_data['settings']['type'] ) {
			$template_data['settings']['filters'] = array();
		}

		return $template_data;
	}

	/**
	 * Add helper classes for filters
	 *
	 * @param $template_data
	 *
	 * @return mixed
	 *
	 * @since 2.7.5
	 */
	public function bnb_class_helper( $template_data ) {

		if ( isset( $template_data['settings']['type'] ) && 'bnb' === $template_data['settings']['type'] ) {
			$template_data['gallery_container']['class'][] = 'modula-gallery-bnb';
		}

		return $template_data;
	}

	/**
	 * Generate image links for BnB gallery type.
	 *
	 * @param array $item_data The item data array.
	 * @param array $item The item array.
	 * @param array $settings The settings array.
	 * @param array $images The images array.
	 * @return array Modified item data.
	 * @since 2.7.5
	 */
	public function modula_bnb_generate_image_links( $item_data, $item, $settings, $images ) {

		if ( ! isset( $settings['type'] ) || 'bnb' !== $settings['type'] ) {
			return $item_data;
		}

		$item_data['this_index'] = $this->count++;
		$item_data['max_index']  = count( $images ) - 1;

		$image_full = wp_get_attachment_image_src( $item['id'], 'full' );

		if ( ! $image_full ) {
			return $item_data;
		}

		$item_data['img_attributes']['width']  = $image_full[1];
		$item_data['img_attributes']['height'] = $image_full[2];
		$item_data['image_full']               = $image_full[0];
		$item_data['image_url']                = $image_full[0];

		$item_data['img_attributes']['src']      = $image_full[0];
		$item_data['img_attributes']['data-src'] = $image_full[0];

		return $item_data;
	}

	/**
	 * Start wrapping images for BnB gallery type.
	 *
	 * @param array $settings The settings array.
	 * @param array $item_data The item data array.
	 * @since 2.7.5
	 */
	public function bnb_image_wrap_start( $settings, $item_data ) {

		if ( ! isset( $settings['type'] ) || 'bnb' !== $settings['type'] || ! isset( $item_data['this_index'] ) ) {
			return;
		}

		// open main item wrapper
		if ( 0 === $item_data['this_index'] ) {
			echo '<div class="modula_bnb_main_wrapper">';
		}
		// open items wrapper
		if ( 1 === $item_data['this_index'] ) {
			echo '<div class="modula_bnb_items_wrapper">';
		}
	}

	/**
	 * End wrapping images for BnB gallery type.
	 *
	 * @param array $settings The settings array.
	 * @param array $item_data The item data array.
	 * @since 2.7.5
	 */
	public function bnb_image_wrap_end( $settings, $item_data ) {
		if ( ! isset( $settings['type'] ) || 'bnb' !== $settings['type'] || ! isset( $item_data['this_index'] ) ) {
			return;
		}

		// close main & items wrappers
		if ( 0 === $item_data['this_index'] || $item_data['this_index'] === $item_data['max_index'] ) {
			echo '</div>';
		}
	}

	/**
	 * Render icon radio field type for Pro version. Used to overwrite the lite setting display.
	 *
	 * @param string $html The HTML content.
	 * @param array  $field The field array.
	 * @param string $value The field value.
	 * @return string Modified HTML content.
	 * @since 2.7.5
	 */
	public function pro_icon_radio_field_type( $html, $field, $value ) {
		$wpchill_upsell = false;
		if ( class_exists( 'WPChill_Upsells' ) ) {
			// Initialize WPChill upsell class
			$args = apply_filters(
				'modula_upsells_args',
				array(
					'shop_url' => 'https://wp-modula.com',
					'slug'     => 'modula',
				)
			);

			$wpchill_upsell = WPChill_Upsells::get_instance( $args );
		}
		$html .= '<div class="modula-icons-radio-wrapper">';
		foreach ( $field['values'] as $key => $name ) {
			$html .= '<div class="modula-icons-radio-item">';
			$icon  = apply_filters( 'modula_radio_icon_url', esc_url( MODULA_URL . 'assets/images/settings/' . $key . '.png' ), $key, $name );
			$html .= '<input id="modula-icon-' . esc_attr( $key ) . '" type="radio" name="modula-settings[type]"  data-setting="' . esc_attr( $field['id'] ) . '" value="' . esc_attr( $key ) . '" ' . checked( $key, $value, false ) . '>';
			$html .= '<label class="modula-radio-icon" for="modula-icon-' . esc_attr( $key ) . '"><img src="' . esc_url( $icon ) . '" alt="' . esc_attr( $name ) . '" title="' . esc_attr( $name ) . '" class="modula-icon-radio" /><span class="modula-icon-radio-name">' . esc_html( $name ) . '</span></label>';
			$html .= '</div>';
		}

		foreach ( $field['disabled']['values'] as $key => $name ) {
			$html .= '<div class="modula-icons-radio-item">';
			$addon = 'bnb' === $key ? 'modula' : 'modula-' . $key;
			if ( $wpchill_upsell && ! $wpchill_upsell->is_upgradable_addon( $addon ) ) {
				$class = 'modula-radio-icon-install';
			} else {
				$class = 'modula-radio-icon-disabled';
			}

			$icon  = apply_filters( 'modula_radio_icon_url_disabled', esc_url( MODULA_URL . 'assets/images/settings/' . $key . '-disabled.png' ), $key, $name );
			$html .= '<label class="modula-radio-icon ' . esc_attr( $class ) . '" ><img src="' . esc_url( $icon ) . '" alt="' . esc_attr( $name ) . '" title="' . esc_attr( $name ) . '" class="modula-icon-radio" /><span class="modula-icon-radio-name">' . esc_html( $name ) . '</span></label>';
			$html .= '</div>';
		}

		$html .= '</div>';

		return $html;
	}

	/**
	 * Set the URL for the BnB icon.
	 *
	 * @param string $url The icon URL.
	 * @param string $type The type of icon.
	 * @return string Modified icon URL.
	 * @since 2.7.5
	 */
	public function modula_bnb_icon_url( $url, $type ) {

		if ( 'bnb' === $type ) {
			return esc_url( MODULA_PRO_URL . 'assets/images/admin/' . $type . '.png' );
		}

		return $url;
	}

	/**
	 * Generate Grid Css
	 *
	 * @param $css
	 * @param $gallery_id
	 * @param $settings
	 *
	 * @return string
	 *
	 * @since  2.7.5
	 */
	public function generate_new_css( $css, $gallery_id, $settings ) {

		if ( 'bnb' === $settings['type'] ) {
			$css .= "html body #{$gallery_id}.modula.modula-gallery-bnb .modula_bnb_main_wrapper{flex-basis: calc( 50% - " . ( absint( $settings['gutter'] ) / 2 ) . 'px );}';
			$css .= "html body #{$gallery_id}.modula.modula-gallery-bnb .modula_bnb_items_wrapper {flex-basis: calc( 50% - " . ( absint( $settings['gutter'] ) / 2 ) . 'px );gap: ' . absint( $settings['gutter'] ) . 'px;}';

			$css .= "@media (min-width: 768px) and (max-width:992px) {html body #{$gallery_id}.modula.modula-gallery-bnb .modula_bnb_main_wrapper{flex-basis: calc( 50% - " . ( absint( $settings['tablet_gutter'] ) / 2 ) . 'px );}}';
			$css .= "@media (min-width: 768px) and (max-width:992px) {html body #{$gallery_id}.modula.modula-gallery-bnb .modula_bnb_items_wrapper {flex-basis: calc( 50% - " . ( absint( $settings['tablet_gutter'] ) / 2 ) . 'px );gap: ' . absint( $settings['tablet_gutter'] ) . 'px;}}';

			$css .= "@media (max-width: 768px) {html body #{$gallery_id}.modula.modula-gallery-bnb .modula_bnb_main_wrapper {flex-basis: 100%;gap: 0;}}";
			$css .= "@media (max-width: 768px) {html body #{$gallery_id}.modula.modula-gallery-bnb .modula_bnb_items_wrapper {display:none;}}";
		}

		return $css;
	}

	/**
	 * Enqueue slider elementor styles
	 */
	public function add_elementor_styles() {
		wp_enqueue_style( 'modula-pro-bnb', MODULA_PRO_URL . 'assets/js/misc/modula-pro.css', array(), MODULA_PRO_VERSION );
	}
}
