<?php

/**
 * Main class for EXIF
 */

class Modula_Exif {
	/**
	 * Holds the class object
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public static $instance;

	/**
	 * The name of the plugin
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $plugin_name = 'Modula Exif';

	/**
	 * Unique plugin identifier
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $plugin_slug = 'modula-exif';

	/**
	 * Primary class construct
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		// Set admin requirements on admin_init so that we won't trigger get_packages from WPChill_Upsells on frontend
		add_action( 'admin_init', array( $this, 'set_admin_requirements' ), 25 );

		if ( is_admin() ) {
			add_action( 'modula_pro_updater', array( $this, 'addon_updater' ), 15, 2 );
		}
		
		$this->require_helpers();
		add_action( 'modula_scripts_after_wp_modula', array( $this, 'modula_exif_admin_scripts' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'modula_exif_enqueue_scripts' ) );
		add_filter( 'modula_lite_default_settings', array( $this, 'modula_exif_defaults' ) );
		add_filter( 'modula_album_default_settings', array( $this, 'modula_exif_albums_defaults' ) );
		add_filter( 'modula_link_item', array( $this, 'modula_exif_gallery_link' ), 10, 4 );
		add_filter( 'modula_shortcode_item_data', array( $this, 'modula_exif_item_data' ), 20, 3 );
		add_filter( 'modula_hidden_items_data', array( $this, 'modula_hidden_items_exif' ), 20, 3 );
		add_filter( 'modula_album_lightbox_item', array( $this, 'modula_exif_albums_lightbox' ), 10, 5 );
		add_action( 'modula_defaults_scripts_after_wp_modula', array( $this, 'modula_exif_admin_scripts' ) );
		// filter for migrator
		add_filter( 'modula_migrate_gallery_data', array( $this, 'modula_exif_migrator_data' ), 25, 3 );
		add_action( 'init', array( $this, 'set_locale' ) );
	}

	/**
	 * Set admin requirements and check for license.
	 *
	 * @return void
	 * @since 1.0.3
	 */
	public function set_admin_requirements() {

		if ( class_exists( 'WPChill_Upsells' ) ) {

			$args           = apply_filters(
				'modula_upsells_args',
				array(
					'shop_url' => 'https://wp-modula.com',
					'slug'     => 'modula',
				)
			);
			$wpchill_upsell = WPChill_Upsells::get_instance( $args );

			if ( $wpchill_upsell && ! $wpchill_upsell->is_upgradable_addon( 'modula-exif' ) ) {
				$this->require_admin();
			}
		}
	}

	/**
	 * Prevent plugin activation if Modula Pro is not installed
	 *
	 * @since 1.0.0
	 */
	public function plugin_activation() {
		if ( ! class_exists( 'Modula_PRO' ) ) {
			deactivate_plugins( plugin_basename( MODULA_EXIF_FILE ) );
			wp_die( __( 'Please install and activate Modula Pro before using Modula Exif add-on.', 'modula-exif' ) );
		}
	}

	public function addon_updater( $license_key, $store_url ) {

		if ( class_exists( 'Modula_Pro_Base_Updater' ) ) {
			$modula_addon_updater = new Modula_Pro_Base_Updater(
				$store_url,
				MODULA_EXIF_FILE,
				array(
					'version' => MODULA_EXIF_VERSION,        // current version number
					'license' => $license_key,               // license key (used get_option above to retrieve from DB)
					'item_id' => 447521,                      // ID of the product
					'author'  => 'MachoThemes',            // author of this plugin
					'beta'    => false,
				)
			);
		}
	}

	/**
	 * Loads all admin related files into scope.
	 *
	 * @since 1.0.0
	 */
	public function require_admin() {

		require_once MODULA_EXIF_PATH . 'includes/admin/class-modula-exif-settings.php';

	}

	/**
	 * Loads all the helper functions
	 *
	 * @since 1.0.0
	 */
	public function require_helpers() {
		require MODULA_EXIF_PATH . 'includes/modula-exif-helper.php';
	}


	/**
	 * Returns the singleton instance of the class.
	 *
	 * @since 1.0.0
	 *
	 * @return object The Modula_Exif object.
	 */
	public static function get_instance() {
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Modula_Exif ) ) {
			self::$instance = new Modula_Exif();
		}

		return self::$instance;
	}

	/**
	 * Enqueue front styles & scripts
	 *
	 * @since 1.0.0
	 */
	public function modula_exif_enqueue_scripts() {
		wp_enqueue_style( 'modula-exif-css', MODULA_EXIF_URL . 'assets/css/modula-exif-front.css', MODULA_EXIF_VERSION );
	}

	/**
	 * Enqueue admin scripts
	 *
	 * @since 1.0.0
	 */
	public function modula_exif_admin_scripts() {

		wp_enqueue_script( 'modula-exif-conditions', MODULA_EXIF_URL . 'assets/js/modula-exif-conditions.js', array( 'jquery' ), MODULA_EXIF_VERSION, true );
		wp_enqueue_script( 'modula-exif-js', MODULA_EXIF_URL . 'assets/js/wp-modula-exif.js', array( 'jquery' ), MODULA_EXIF_VERSION, true );
		wp_enqueue_style( 'modula-exif-admin-css', MODULA_EXIF_URL . 'assets/css/modula-exif-admin.css', MODULA_EXIF_VERSION );
	}

	/**
	 * Add Exif defaults settings
	 *
	 * @param $defaults
	 *
	 * @return mixed $defaults;
	 *
	 * @since 1.0.0
	 */
	public function modula_exif_defaults( $defaults ) {
		$defaults['enable_exif']        = 0;
		$defaults['exif_camera']        = 1;
		$defaults['exif_lens']          = 1;
		$defaults['exif_shutter_speed'] = 1;
		$defaults['exif_aperture']      = 1;
		$defaults['exif_focal_length']  = 1;
		$defaults['exif_iso']           = 1;
		$defaults['exif_date']          = 1;

		return $defaults;
	}

	/**
	 * Add Exif defaults settings
	 *
	 * @param $defaults
	 *
	 * @return mixed $defaults;
	 *
	 * @since 1.0.0
	 */
	public function modula_exif_albums_defaults( $defaults ) {

		$defaults['displayExif'] = 'default';

		return $defaults;
	}

	/**
	 * Integrate EXIF with [modula-link]
	 *
	 * @param $image_url
	 * @param $image
	 * @param $gall_id
	 * @param $settings
	 *
	 * @return mixed $image_url
	 *
	 * @since 1.0.0
	 */
	public function modula_exif_gallery_link( $image_url, $image, $gall_id, $settings ) {
		if ( isset( $settings['enable_exif'] ) && '1' == $settings['enable_exif'] ) {

			if ( '1' == $settings['exif_camera'] && '' != $image['exif_camera'] && 'N/A' != $image['exif_camera'] ) {
				$image_url['opts']['caption'] .= modula_format_exif( $image['exif_camera'], 'camera' );
			}

			if ( '1' == $settings['exif_lens'] && '' != $image['exif_lens'] && 'N/A' != $image['exif_lens'] ) {
				$image_url['opts']['caption'] .= modula_format_exif( $image['exif_lens'], 'lens' );
			}

			if ( '1' == $settings['exif_focal_length'] && '' != $image['exif_focal_length'] && 'N/A' != $image['exif_focal_length'] ) {
				$image_url['opts']['caption'] .= modula_format_exif( $image['exif_focal_length'], 'eye' );
			}

			if ( '1' == $settings['exif_shutter_speed'] && 0 != $image['exif_shutter_speed'] && 'N/A' != $image['exif_shutter_speed'] ) {
				$image_url['opts']['caption'] .= modula_format_exif( $image['exif_shutter_speed'], 'shutter_speed' );
			}

			if ( '1' == $settings['exif_aperture'] && '' != $image['exif_aperture'] && 'N/A' != $image['exif_aperture'] ) {
				$image_url['opts']['caption'] .= modula_format_exif( $image['exif_aperture'], 'aperture' );
			}

			if ( '1' == $settings['exif_iso'] && '' != $image['exif_iso'] && 'N/A' != $image['exif_iso'] ) {
				$image_url['opts']['caption'] .= modula_format_exif( $image['exif_iso'], 'iso' );
			}

			if ( '1' == $settings['exif_date'] && '0' != $image['exif_date'] && 'N/A' != $image['exif_date'] ) {
				$image_url['opts']['caption'] .= modula_format_exif( $image['exif_date'], 'calendar' );
			}
		}

		return $image_url;

	}

	/**
	 * Function to add exif information in item data
	 *
	 * @param $item_data
	 * @param $image
	 * @param $settings
	 *
	 * @return mixed $item_data
	 *
	 * @since 1.0.0
	 */
	public function modula_exif_item_data( $item_data, $image, $settings ) {

		if ( isset( $settings['enable_exif'] ) && '1' == $settings['enable_exif'] && isset( $item_data['link_attributes']['data-caption'] ) ) {

			if ( isset( $image['exif_camera'] ) && '1' == $settings['exif_camera'] && '' != $image['exif_camera'] && 'N/A' != $image['exif_camera'] ) {
				$item_data['link_attributes']['data-caption'] .= modula_format_exif( $image['exif_camera'], 'camera' );
			}

			if ( isset( $image['exif_lens'] ) && '1' == $settings['exif_lens'] && '' != $image['exif_lens'] && 'N/A' != $image['exif_lens'] ) {
				$item_data['link_attributes']['data-caption'] .= modula_format_exif( $image['exif_lens'], 'lens' );
			}

			if ( isset( $image['exif_lens'] ) && '1' == $settings['exif_focal_length'] && '' != $image['exif_focal_length'] && 'N/A' != $image['exif_focal_length'] ) {
				$item_data['link_attributes']['data-caption'] .= modula_format_exif( $image['exif_focal_length'], 'eye' );
			}

			if ( isset( $image['exif_shutter_speed'] ) && '1' == $settings['exif_shutter_speed'] && 0 != $image['exif_shutter_speed'] && 'N/A' != $image['exif_shutter_speed'] && 'sec' != $image['exif_shutter_speed'] ) {
				$item_data['link_attributes']['data-caption'] .= modula_format_exif( $image['exif_shutter_speed'], 'shutter_speed' );
			}

			if ( isset( $image['exif_aperture'] ) && '1' == $settings['exif_aperture'] && '' != $image['exif_aperture'] && 'N/A' != $image['exif_aperture'] ) {
				$item_data['link_attributes']['data-caption'] .= modula_format_exif( $image['exif_aperture'], 'aperture' );
			}

			if ( isset( $image['exif_iso'] ) && '1' == $settings['exif_iso'] && '' != $image['exif_iso'] && 'N/A' != $image['exif_iso'] ) {
				$item_data['link_attributes']['data-caption'] .= modula_format_exif( $image['exif_iso'], 'iso' );
			}

			if ( isset( $image['exif_date'] ) && '1' == $settings['exif_date'] && '0' != $image['exif_date'] && 'N/A' != $image['exif_date'] && ! empty( $image['exif_date'] ) ) {
				$item_data['link_attributes']['data-caption'] .= modula_format_exif( $image['exif_date'], 'calendar' );
			}
		}
		return $item_data;
	}

	/**
	 * Add EXIF image data for hidden items
	 *
	 * @param $attr
	 * @param $settings
	 * @param $image
	 *
	 * @return mixed
	 * @since 1.0.0
	 */
	public function modula_hidden_items_exif( $attr, $settings, $image ) {

		if ( isset( $settings['enable_exif'] ) && '1' == $settings['enable_exif'] && isset( $attr['data-caption'] ) ) {

			if ( '1' == $settings['exif_camera'] && '' != $image['exif_camera'] && 'N/A' != $image['exif_camera'] ) {
				$attr['data-caption'] .= modula_format_exif( $image['exif_camera'], 'camera' );
			}

			if ( '1' == $settings['exif_lens'] && '' != $image['exif_lens'] && 'N/A' != $image['exif_lens'] ) {
				$attr['data-caption'] .= modula_format_exif( $image['exif_lens'], 'lens' );
			}

			if ( '1' == $settings['exif_focal_length'] && '' != $image['exif_focal_length'] && 'N/A' != $image['exif_focal_length'] ) {
				$attr['data-caption'] .= modula_format_exif( $image['exif_focal_length'], 'eye' );
			}

			if ( '1' == $settings['exif_shutter_speed'] && 0 != $image['exif_shutter_speed'] && 'N/A' != $image['exif_shutter_speed'] && 'sec' != $image['exif_shutter_speed'] ) {
				$attr['data-caption'] .= modula_format_exif( $image['exif_shutter_speed'], 'shutter_speed' );
			}

			if ( '1' == $settings['exif_aperture'] && '' != $image['exif_aperture'] && 'N/A' != $image['exif_aperture'] ) {
				$attr['data-caption'] .= modula_format_exif( $image['exif_aperture'], 'aperture' );
			}

			if ( '1' == $settings['exif_iso'] && '' != $image['exif_iso'] && 'N/A' != $image['exif_iso'] ) {
				$attr['data-caption'] .= modula_format_exif( $image['exif_iso'], 'iso' );
			}

			if ( '1' == $settings['exif_date'] && '0' != $image['exif_date'] && 'N/A' != $image['exif_date'] && ! empty( $image['exif_date'] ) ) {
				$attr['data-caption'] .= modula_format_exif( $image['exif_date'], 'calendar' );
			}
		}

		return $attr;
	}

	/**
	 * Exif information in lightbox
	 *
	 * @param $image_config
	 * @param $image
	 * @param $settings
	 * @param $gallery
	 *
	 * @param $album_settings
	 *
	 * @return mixed $image_config
	 *
	 * @since 1.0.0
	 */
	public function modula_exif_albums_lightbox( $image_config, $image, $settings, $gallery, $album_settings ) {

		if ( 'default' == $album_settings['displayExif'] ) {

			if ( isset( $settings['enable_exif'] ) && '1' == $settings['enable_exif'] ) {
				if ( '1' == $settings['exif_camera'] && '' != $image['exif_camera'] && 'N/A' != $image['exif_camera'] ) {
					$image_config['opts']['caption'] .= modula_format_exif( $image['exif_camera'], 'camera' );
				}

				if ( '1' == $settings['exif_lens'] && '' != $image['exif_lens'] && 'N/A' != $image['exif_lens'] ) {
					$image_config['opts']['caption'] .= modula_format_exif( $image['exif_lens'], 'lens' );
				}

				if ( '1' == $settings['exif_focal_length'] && '' != $image['exif_focal_length'] && 'N/A' != $image['exif_focal_length'] ) {
					$image_config['opts']['caption'] .= modula_format_exif( $image['exif_focal_length'], 'eye' );
				}

				if ( '1' == $settings['exif_shutter_speed'] && 0 != $image['exif_shutter_speed'] && 'N/A' != $image['exif_shutter_speed'] ) {
					$image_config['opts']['caption'] .= modula_format_exif( $image['exif_shutter_speed'], 'shutter_speed' );
				}

				if ( '1' == $settings['exif_aperture'] && '' != $image['exif_aperture'] && 'N/A' != $image['exif_aperture'] ) {
					$image_config['opts']['caption'] .= modula_format_exif( $image['exif_aperture'], 'aperture' );
				}

				if ( '1' == $settings['exif_iso'] && '' != $image['exif_iso'] && 'N/A' != $image['exif_iso'] ) {
					$image_config['opts']['caption'] .= modula_format_exif( $image['exif_iso'], 'iso' );
				}

				if ( '1' == $settings['exif_date'] && '0' != $image['exif_date'] && 'N/A' != $image['exif_date'] ) {
					$image_config['opts']['caption'] .= modula_format_exif( $image['exif_date'], 'calendar' );
				}
			}
		} elseif ( 'on' == $album_settings['displayExif'] ) {
			if ( isset( $settings['exif_camera'] ) && '' != $image['exif_camera'] && 'N/A' != $image['exif_camera'] ) {
				$image_config['opts']['caption'] .= modula_format_exif( $image['exif_camera'], 'camera' );
			}

			if ( isset( $image['exif_lens'] ) && '' != $image['exif_lens'] && 'N/A' != $image['exif_lens'] ) {
				$image_config['opts']['caption'] .= modula_format_exif( $image['exif_lens'], 'lens' );
			}

			if ( isset( $image['exif_focal_length'] ) && '' != $image['exif_focal_length'] && 'N/A' != $image['exif_focal_length'] ) {
				$image_config['opts']['caption'] .= modula_format_exif( $image['exif_focal_length'], 'eye' );
			}

			if ( isset( $image['exif_shutter_speed'] ) && 0 != $image['exif_shutter_speed'] && 'N/A' != $image['exif_shutter_speed'] ) {
				$image_config['opts']['caption'] .= modula_format_exif( $image['exif_shutter_speed'], 'shutter_speed' );
			}

			if ( isset( $image['exif_aperture'] ) && '' != $image['exif_aperture'] && 'N/A' != $image['exif_aperture'] ) {
				$image_config['opts']['caption'] .= modula_format_exif( $image['exif_aperture'], 'aperture' );
			}

			if ( isset( $image['exif_iso'] ) && '' != $image['exif_iso'] && 'N/A' != $image['exif_iso'] ) {
				$image_config['opts']['caption'] .= modula_format_exif( $image['exif_iso'], 'iso' );
			}

			if ( isset( $image['exif_date'] ) && '0' != $image['exif_date'] && 'N/A' != $image['exif_date'] ) {
				$image_config['opts']['caption'] .= modula_format_exif( $image['exif_date'], 'calendar' );
			}
		}

		return $image_config;
	}

	/**
	 * Get icons function
	 *
	 * @param $icon
	 *
	 * @return mixed|string|void $icons;
	 *
	 * @since 1.0.0
	 */
	public static function modula_exif_get_icons( $icon ) {

		switch ( $icon ) {

			case 'camera':
				return '<svg width="24" height="24" viewBox="0 0 24 24"><path fill="#000000" d="M4,4H7L9,2H15L17,4H20A2,2 0 0,1 22,6V18A2,2 0 0,1 20,20H4A2,2 0 0,1 2,18V6A2,2 0 0,1 4,4M12,7A5,5 0 0,0 7,12A5,5 0 0,0 12,17A5,5 0 0,0 17,12A5,5 0 0,0 12,7M12,9A3,3 0 0,1 15,12A3,3 0 0,1 12,15A3,3 0 0,1 9,12A3,3 0 0,1 12,9Z"/></svg>';
				break;

			case 'iso':
				return '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24"><path d="M19 3H5c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h14c1.1 0 2-.9 2-2V5c0-1.1-.9-2-2-2zM5.5 7.5h2v-2H9v2h2V9H9v2H7.5V9h-2V7.5zM19 19H5L19 5v14zm-2-2v-1.5h-5V17h5z"/></svg>';
				break;

			case 'shutter_speed':
				return '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24"><path d="M15 1H9v2h6V1zm4.03 6.39l1.42-1.42c-.43-.51-.9-.99-1.41-1.41l-1.42 1.42C16.07 4.74 14.12 4 12 4c-4.97 0-9 4.03-9 9s4.02 9 9 9 9-4.03 9-9c0-2.12-.74-4.07-1.97-5.61zM12 20c-3.87 0-7-3.13-7-7s3.13-7 7-7 7 3.13 7 7-3.13 7-7 7zm-.32-5H6.35c.57 1.62 1.82 2.92 3.41 3.56l-.11-.06 2.03-3.5zm5.97-4c-.57-1.6-1.78-2.89-3.34-3.54L12.26 11h5.39zm-7.04 7.83c.45.11.91.17 1.39.17 1.34 0 2.57-.45 3.57-1.19l-2.11-3.9-2.85 4.92zM7.55 8.99C6.59 10.05 6 11.46 6 13c0 .34.04.67.09 1h4.72L7.55 8.99zm8.79 8.14C17.37 16.06 18 14.6 18 13c0-.34-.04-.67-.09-1h-4.34l2.77 5.13zm-3.01-9.98C12.9 7.06 12.46 7 12 7c-1.4 0-2.69.49-3.71 1.29l2.32 3.56 2.72-4.7z"/></svg>';
				break;

			case 'lens':
				return '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2z"/></svg>';
				break;

			case 'aperture':
				return '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24"><path d="M9.4 10.5l4.77-8.26C13.47 2.09 12.75 2 12 2c-2.4 0-4.6.85-6.32 2.25l3.66 6.35.06-.1zM21.54 9c-.92-2.92-3.15-5.26-6-6.34L11.88 9h9.66zm.26 1h-7.49l.29.5 4.76 8.25C21 16.97 22 14.61 22 12c0-.69-.07-1.35-.2-2zM8.54 12l-3.9-6.75C3.01 7.03 2 9.39 2 12c0 .69.07 1.35.2 2h7.49l-1.15-2zm-6.08 3c.92 2.92 3.15 5.26 6 6.34L12.12 15H2.46zm11.27 0l-3.9 6.76c.7.15 1.42.24 2.17.24 2.4 0 4.6-.85 6.32-2.25l-3.66-6.35-.93 1.6z"/></svg>';
				break;

			case 'calendar':
				return '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24"><path d="M20 3h-1V1h-2v2H7V1H5v2H4c-1.1 0-2 .9-2 2v16c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V5c0-1.1-.9-2-2-2zm0 18H4V10h16v11zm0-13H4V5h16v3z"/></svg>';
				break;

			case 'eye':
				return '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24"><path d="M12 6.5c3.79 0 7.17 2.13 8.82 5.5-1.65 3.37-5.02 5.5-8.82 5.5S4.83 15.37 3.18 12C4.83 8.63 8.21 6.5 12 6.5m0-2C7 4.5 2.73 7.61 1 12c1.73 4.39 6 7.5 11 7.5s9.27-3.11 11-7.5c-1.73-4.39-6-7.5-11-7.5zm0 5c1.38 0 2.5 1.12 2.5 2.5s-1.12 2.5-2.5 2.5-2.5-1.12-2.5-2.5 1.12-2.5 2.5-2.5m0-2c-2.48 0-4.5 2.02-4.5 4.5s2.02 4.5 4.5 4.5 4.5-2.02 4.5-4.5-2.02-4.5-4.5-4.5z"/></svg>';
				break;

			default:
				return apply_filters( 'modula_exif_get_icon', '', $icon );
				break;

		}

	}


	/**
	 * Add modula migrator data for EXIF
	 *
	 * @param $modula_settings
	 * @param $guest_settings
	 * @param $source
	 *
	 * @return mixed
	 *
	 * @since 1.0.1
	 */
	public function modula_exif_migrator_data( $modula_settings, $guest_settings, $source ) {

		if ( isset( $source ) ) {
			switch ( $source ) {
				case 'envira':
					if ( isset( $guest_settings['config']['exif_lightbox'] ) && 1 == $guest_settings['config']['exif_lightbox'] ) {
						$modula_settings['enable_exif'] = 1;
					}

					if ( isset( $guest_settings['config']['exif_lightbox_aperture'] ) && 1 == $guest_settings['config']['exif_lightbox_aperture'] ) {
						$modula_settings['exif_aperture'] = 1;
					}

					if ( isset( $guest_settings['config']['exif_lightbox_shutter_speed'] ) && 1 == $guest_settings['config']['exif_lightbox_shutter_speed'] ) {
						$modula_settings['exif_shutter_speed'] = 1;
					}

					if ( isset( $guest_settings['config']['exif_lightbox_focal_length'] ) && 1 == $guest_settings['config']['exif_lightbox_focal_length'] ) {
						$modula_settings['exif_focal_length'] = 1;
					}

					if ( isset( $guest_settings['config']['exif_lightbox_iso'] ) && 1 == $guest_settings['config']['exif_lightbox_iso'] ) {
						$modula_settings['exif_iso'] = 1;
					}

					if ( isset( $guest_settings['config']['exif_lightbox_capture_time'] ) && 1 == $guest_settings['config']['exif_lightbox_capture_time'] ) {
						$modula_settings['exif_date'] = 1;
					}

					break;
			}
		}

		return $modula_settings;
	}

	/**
	 * Loads the plugin textdomain for translation.
	 *
	 * @since 1.0.2
	 */
	public function set_locale() {
		load_plugin_textdomain( $this->plugin_slug, false, MODULA_EXIF_TRANSLATE );
	}
}
